#!/usr/bin/env perl

=head1 NAME

golem-events-to-tsv.pl - история событий из голема в виде таблички

=head1 DESCRIPTION

    Опции:
    --date-start
        дата начала события в формате YYYY-MM-DD
    --date-end
        дата окончания события в формате YYYY-MM-DD
    --resp
        ответственные за хост (аггрегатор), можно указывать несколько опций --resp
    --show-developer
        показывать события по хостам с галкой "не звонит никогда" (в големе она называется "хост в разработке")
    --priority
        приоритет - any|critical|warning, по-умолчанию - critical
    --object
        не обязательно - хост (аггрегатор), по которому фильтровать события 

    Примеры:
    golem-events-to-tsv.pl --date-start 2016-08-01 --date-end 2016-09-01 --resp rivik --resp yandex_monetize_search_direct_interface_exp --priority critical | awk '{ print $2 }' | sort | uniq -c | sort -rn | head -20

    топ-20 событий, по которым должны были звонить (или не успевали, если они флапали/красили руками и тд). Аналогично, плюс хосты: ... | awk '{ print $1,$2 }' | ...


    DEBUG=1 golem-events-to-tsv.pl --date-start 2016-08-01 --date-end 2016-09-01 --resp rivik --resp yandex_monetize_search_direct_interface_exp --priority any --show-developer --object='checks.direct.yandex.ru'

    выведет на STDOUT историю событий с warning + critical + по хостам "не звонить" + только по checks.direct.yandex.ru


    golem-events-to-tsv.pl - < golem-events.html

    выведет историю событий из готовой html (копия странички "События для администратора" в Големе)


    Время хранения истории в Големе:

    История хранится 2 месяца. Остальное можно найти в hadoop, но я не нашел там информации по ответственным, неудобно отбирать хосты. Для того, чтобы получить выгрузку по событиям direct.yandx.ru, потребуется зайти на https://tasmania.hdp.yandex.net. Сверху "Query editors" - "Hive", далее слева "Database" в выпадающем списке "logs". В поле запроса пишем: select object_name, eventtype_name,priority, time_start, time_end from golem_events_log where object_name = 'direct.yandex.ru' and eventtype_name = 'sla_direct_timings' and priority = 'critical' and time_start like '2016%' order by time_start; Ждем пока выполнится Job, в result будет вывод в виде XML

=cut

use strict;
use warnings;
use Getopt::Long;
use Time::Local;
use Data::Dumper;

use HTML::TableExtract;
use URI::Escape;

my %O = (
    "resp" => [],
    "show_developer" => 0, 
    "priority" => "critical",
    "object" => undef,
);
usage_and_exit() if ! @ARGV;

GetOptions(
    "date-start=s" => \$O{date_start},
    "date-end=s" => \$O{date_end},
    "resp=s" => $O{resp},
    "show-developer!" => \$O{show_developer},
    "priority=s" => \$O{priority},
    "object=s" => \$O{object},
    "help" => \&usage_and_exit,
) or die "can't parse options";

my $content;
if ($O{date_start} && $O{date_end} && @{$O{resp}}) {
    my $url = 'type=history&priority=%s&resp=%s&sdate_s=%s 00:00:00&sdate_e=%s 00:00:00';
    $url = sprintf $url, $O{priority}, join(',', @{$O{resp}}), $O{date_start}, $O{date_end};
    $url .= '&show_developer=on' if $O{show_developer};
    $url .= '&object=' . $O{object} if $O{object};
    $url = 'https://golem.yandex-team.ru/events.sbml?' . uri_escape($url, '^A-Za-z0-9\-\._~=&');
    warn "url: $url\n" if $ENV{DEBUG};
    
    open(my $fh, '-|', "curl -s '$url'");
    local $/;
    $content = <$fh>;
    close($fh);
} else {
    local $/;
    $content = <>;
}

my $te = HTML::TableExtract->new(headers => ["объект", "тип события", "время начала", "время окончания", "последние данные"]);
$te->parse($content) or die "$!";
my $ts = ($te->tables())[0];

my $events = [];
my $last_obj = "undef";
for my $row ($ts->rows) {
    map { $_ = defined($_) ? $_ : "undef"; s/^\h*|\h*$|\v*|\xa0//sg; s/\h+/ /g; s/ No DC//; } @$row;

    if (length($row->[0]) > 0) {
        $last_obj = $row->[0];
    } else {
        $row->[0] = $last_obj;
    }

    my $new_row = [@$row];
    push @$new_row, golem_date_to_epoch($row->[2]), golem_date_to_epoch($row->[3]);
    push @$events, $new_row;
}

map { printf "%s\n", join("\t", @$_) } sort { $a->[5] <=> $b->[5] } @$events;

sub golem_date_to_epoch {
    my @date = ($_[0] =~ /(\d{4})-(\d\d)-(\d\d) (\d\d):(\d\d):(\d\d)/);
    return 0 if ! @date;
    return timelocal($date[5], $date[4], $date[3], $date[2], $date[1] - 1, $date[0]);
}

sub usage_and_exit {
    system("podselect -section NAME -section DESCRIPTION $0 | pod2text-utf8 >&2");
    exit 0;
}
