#!/usr/bin/env perl

use strict;
use warnings;

use FindBin qw/$Bin/;

use Clone qw(clone);
use JSON;
use Path::Tiny;
use YAML;
use Template;


run() unless caller();


sub run
{
    my $tmpl = load_templates();
    my $conf = load_config($ARGV[0]);
    my $dashboard = process_template($tmpl, $conf);
    print to_json($dashboard, { pretty => 1 } );
}


sub load_templates
{
    my $dashboard_tmpl = decode_json(path("$Bin/tmpl/grafana-dashboard.tmpl.json")->slurp);
    my $row_tmpl = decode_json(path("$Bin/tmpl/grafana-row.tmpl.json")->slurp);
    my $panel_tmpl = decode_json(path("$Bin/tmpl/grafana-panel.tmpl.json")->slurp);
    my $request_tmpl = decode_json(path("$Bin/tmpl/grafana-request-create-dashboard.json")->slurp);

    return {
        dashboard => $dashboard_tmpl,
        row => $row_tmpl, 
        panel => $panel_tmpl,
        request => $request_tmpl,
    };
}

sub load_config
{
    my ($file) = @_;
    if ($file =~ /\.tt2$/) {
        my $t = Template->new();
        my $res;
        $t->process($file, {}, \$res) || die $t->error();
        return YAML::Load($res);
    } else {
        return YAML::LoadFile($file);
    }
}

sub process_template
{
    my ($t, $conf) = @_;

    my $dashboard = clone($t->{dashboard});
    $dashboard->{title} = $conf->{title} || 'new';
    $dashboard->{tags} = $conf->{tags} || [];
    $dashboard->{templating} = $conf->{templating} if exists $conf->{templating};
    unless ( grep { $_ eq "auto" } @{$dashboard->{tags}} ){
        push @{$dashboard->{tags}}, "auto";
    }

    my $res = $dashboard;
    if (1) {
        my $request = clone($t->{request});
        $request->{dashboard} = $dashboard;
        $dashboard->{id} = $conf->{id} + 0;
        $res = $request;
    }

    my $panel_id = 1; 
    for my $row_conf (@{$conf->{rows}}){
        if (ref $row_conf eq 'ARRAY') {
            $row_conf = {title => 'Row title', panels => $row_conf};
        }

        my $row = clone($t->{row});
        $row->{title} = $row_conf->{title};
        push @{$dashboard->{rows}}, $row;

        my $panels_count = scalar @{$row_conf->{panels}};
        for my $panel_conf ( @{$row_conf->{panels}} ){
            my $panel = clone($t->{panel});
            push @{$row->{panels}}, $panel;
            my $targets_conf = ref $panel_conf->{target} ? $panel_conf->{target} : [$panel_conf->{target}];
            $panel->{targets} = [ map { {target => $_ } } @{$targets_conf} ]; 
            if ($panel_conf->{grid}) {
                $panel->{grid}->{$_} = $panel_conf->{grid}->{$_} for keys %{$panel_conf->{grid}};
            }
            if ($panel_conf->{y_formats}) {
                $panel->{y_formats} = $panel_conf->{y_formats};
            }
            if (defined $panel_conf->{fill}) {
                $panel->{fill} = $panel_conf->{fill};
            }
            if (my $l = $panel_conf->{legend}) {
                $panel->{legend} = {map {$_ => bool($l->{$_})} keys %$l};
            }
            $panel->{title} = $panel_conf->{title};
            $panel->{span} = $panel_conf->{span} || int(12/$panels_count);
            $panel->{id} = $panel_id++;
        }
    }

    return $res;
}

sub bool {
    return $_[0] =~ /^(true)$/i ? JSON::true : JSON::false;
}
