#!/usr/bin/perl

=head1 SYNOPSIS

    java-deploy.pl java-intapi 1.2477414-1 --prod   # то же самое, что java-deploy.pl intapi 1.2477414-1 --prod
    java-deploy.pl java-intapi 1.2477414-1 --prod --force-restart    # перезапустит сервис командой sv force-restart
    java-deploy.pl java-intapi 1.2477414-1 --prod --tmux    # запустить параллельно в tmux, для --prod включено по умолчанию, если групп хостов (группа -- "основной Директ", песочница, limtest)
    java-deploy.pl java-intapi 1.2477414-1 --prod --no-tmux # запустить последовательно по группам

    java-deploy.pl java-intapi 1.2477414-1 --host ppctest-java-test.ppc.yandex.ru
    java-deploy.pl java-intapi 1.2477414-1 --host localhost  # подменит localhost на `hostname -f`, чтобы работали переменные из кондукторных тегов

    # не используется, пользуйтесь java-test-update
    java-deploy.pl java-intapi 1.2477414-1 --test
    java-deploy.pl java-intapi 1.2476982-1 --test --force-yes    # для downgrade и зависимостей
    java-deploy.pl java-intapi 1.2477414-1 --test --dpkg-options=force-confnew,force-confmiss    # apt-get ... -o Dpkg::Options::="--force-confnew" -o Dpkg::Options::="--force-confmiss"
    
=cut

use strict;
use warnings;

use utf8;

use open qw/:std :encoding(UTF-8)/;

use FindBin qw/$Bin/;
use Getopt::Long;
use YAML;
use JSON;

#our $TEST_HOST = "ppctest-java-test.ppc.yandex.ru";
our $TEST_HOST = "direct_java_test_rtc";
our %HOSTS_FOR_SERVICE = (
    test => {
        'java-api5' => [$TEST_HOST, 'ppctest-sandbox1-front.ppc.yandex.ru'],
        'java-web' =>[$TEST_HOST],
        'java-intapi' =>[$TEST_HOST, 'ppctest-sandbox1-front.ppc.yandex.ru'],
        'java-logviewer' => [$TEST_HOST],
        'java-jobs' => [$TEST_HOST],
        'java-alw' => [$TEST_HOST],
        'lb-moderation' => [$TEST_HOST],
    },
);
our $PLAYBOOK_FILE = '/etc/direct-java-deploy/playbook.yml';
our $WORKING_COPY_PLAYBOOK_FILE = "$Bin/.." . $PLAYBOOK_FILE;
$PLAYBOOK_FILE = $WORKING_COPY_PLAYBOOK_FILE if -f $WORKING_COPY_PLAYBOOK_FILE;

run() unless caller();

sub run {
    my $opt = parse_options();
    
    my $app_full_name = $opt->{app_full_name};
    my $APPS_CONF = YAML::LoadFile("/etc/yandex-direct/direct-apps.conf.yaml")->{apps};
    if (!exists $APPS_CONF->{$app_full_name}){
        die "unknown app '$app_full_name', stop.\nKnown apps: ".(join(", ", sort keys($APPS_CONF)))."\n";
    }
    my $app_conf = $APPS_CONF->{$app_full_name};
    if ( $app_conf->{type} ne "arcadia-java" ){
        die "unsupported app type '$app_conf->{type}', stop.\n";
    }

    my @hosts;
    if ($opt->{host}) {
        @hosts = ($opt->{host});
    } else {
        my ($env) = grep { $opt->{$_} } qw/test prod/;
        if ( $env eq 'test' ){
            @hosts = @{ $HOSTS_FOR_SERVICE{$env}->{$opt->{app_full_name}} || [] };
        } elsif ( $env eq 'prod' ){
            @hosts = @{ $app_conf->{conductor_groups} };
            push @hosts, 'limtest'; # limtest обрабатывается ниже по особенному
        } else {
            die;
        }
        die "no hosts for env '$env' and app '$opt->{app_full_name}', exiting\n" if !@hosts;
    }
    my %cmd_for_host;
    my %cmd_str_for_host;
    $opt->{tmux} //= $opt->{prod} && (@hosts > 1);
    for my $host (@hosts) {
        my @cmd;
        if ($host eq 'limtest') {
            @cmd = ('limtest-up', $app_full_name, $opt->{version}, $opt->{dry_run} ? '--dry-run' : ());
        } else {
            @cmd = ('ansible-playbook', $PLAYBOOK_FILE);
            if ($host eq 'localhost') {
                # переменные из тегов кондуктора (например, ya_environment) работают только для fqdn, поэтому подменяем localhost на fqdn машины
                chomp($host = `hostname -f`);
                push @cmd, '-c', 'local';   # иначе из-под root попытается соединиться по ssh и не сможет из-за отсутсвия ключа
                                            # то, что можно указать любой хост, но действия выполнять на localhost -- удивительно, но работает
            }
            my %vars = (
                app => $opt->{app_full_name},
                version => $opt->{version},
                service_name => $opt->{service},
                package => $app_conf->{package},
                $opt->{force_yes} || $opt->{rollback} ? (apt_force => 'yes') : (),
                $opt->{dpkg_options} ? (apt_dpkg_options => $opt->{dpkg_options}) : (),
                host => $host,
                ansible_ssh_port => 10022,
                $opt->{force_restart} ? (force_restart => 'true') : (),
		$opt->{rollback} ? ( rollback => 'true') : (),
		$opt->{rollback} ? ( serial_mode => '100%') : (),
            );
            if ($app_conf->{sv_services}) {
                $vars{sv_services} = $app_conf->{sv_services};
            }
            if ($app_conf->{sv_dont_wait_on_restart}) {
                $vars{sv_restart_cmd} = "term";
                $vars{sv_dont_wait_on_restart} = 1;
            }
            if ($opt->{force_restart}) {
                $vars{sv_restart_cmd} = "-w 40 force-restart";
            }
            if ($opt->{rollback}) {
                $vars{sv_restart_cmd} = "kill";
            }
            push @cmd, '-e', to_json(\%vars);
            push @cmd, '--check' if $opt->{dry_run};
        }
        $cmd_for_host{$host} = \@cmd;
        $cmd_str_for_host{$host} = join ' ', (map { my_yash_quote($_) } @cmd);
    }
    
    if ($opt->{tmux}) {
        (my $tmux_session = "java-deploy_$opt->{version}") =~ s/[^0-9a-zA-Z_-]/_/g;
        my $tmux_session_created = 0;
        my $tmux_windows_cnt = 0;
        my $tmux_main_prod_window_id = 0;
        for my $host (sort keys %cmd_for_host) {
            my $short_host = (split /\./, $host)[0];
            my $cmd_str = $cmd_str_for_host{$host};
            my $cmd_for_tmux = "set -x; $cmd_str && echo OK || echo FAILED; exec bash -i";
            if ($tmux_session_created) {
                system 'tmux', 'new-window', '-t', $tmux_session, '-n', "${short_host}_$opt->{version}", $cmd_for_tmux;
                $tmux_windows_cnt++;
                unless ($host =~ m/limtest|sandbox/) {
                    # хотим, чтобы в tmux по-умолчанию было открыто окно главного продакшена (не лимтест и не sandbox)
                    $tmux_main_prod_window_id = $tmux_windows_cnt;
                }
            } else {
                system 'tmux', 'new-session', '-s', $tmux_session, '-d', $cmd_for_tmux;
                $tmux_session_created = 1;
            }
        }
        system "tmux a -t $tmux_session \\; select-window -t $tmux_main_prod_window_id \\;";
    } else {
        my %exit_code_for_host;
        for my $host (sort keys %cmd_for_host) {
            my $cmd_str = $cmd_str_for_host{$host};
            print STDERR "going to run: $cmd_str\n";
            system @{ $cmd_for_host{$host} };
            $exit_code_for_host{$host} = $? >> 8;
        }
        if (my @failed_hosts = grep {$exit_code_for_host{$_} != 0} @hosts) {
            print STDERR "deploy failed for hosts:\n";
            for my $host (@failed_hosts) {
                print STDERR "$host: cmd '$cmd_str_for_host{$host}' failed with code $exit_code_for_host{$host}\n";
            }
            exit 1;
        } else {
            print STDERR "OK\n";
        }
    }
}

sub parse_options {
    my %O;
    GetOptions(
        'n|dry-run' => \$O{dry_run},
        'test' => \$O{test},
        'prod' => \$O{prod},
        'host=s' => \$O{host},
        'force-yes' => \$O{force_yes},
	'rollback' => \$O{rollback},
        'dpkg-options=s' => \$O{dpkg_options},
        'force-restart' => \$O{force_restart},
        'tmux!' => \$O{tmux},
        'h|help' => sub { system("podselect -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8 >&2"); exit 0 },
    ) || die "can't parse options, stop\n";
    ($O{app_full_name}, $O{version}) = @ARGV;
    # "сервис" -- это то, как называются init-файлы и т.п.; "приложение" -- имя, которое понимает direct-release
    # у большинства приложений название сервиса совпадает с названием приложения, но не у всех
    # если расхождения будут накапливаться -- можно внести "сервис" в конфиг приложений
    my %app_to_service = (
        'java-alw' => 'user-action-log-writer',
        'java-api5' => 'api5',
        'java-logviewer' => 'logviewer',
        'java-intapi' => 'intapi',
        'java-web' => 'web',
    );
    $O{service} = $app_to_service{$O{app_full_name}} || $O{app_full_name};

    die "--test or --prod or explicit --host should be given\n" unless $O{test} || $O{prod} || $O{host};
    die "--test, --prod and --host are mutually exclusive\n" if scalar (grep {/^(test|host|prod)$/ && $O{$_}} keys %O) > 1;
    die "no service name given\n" unless $O{service};
    die "no version given\n" unless $O{version};
    return \%O;
}


=head2 my_yash_quote

    copy-paste из Y::Shell

=cut
sub my_yash_quote {
    my $param = shift;
    if ($param =~ /^[\w:\/\-=\.]+$/) {
        return $param;
    }
    $param =~ s/(["\$\\!])/\\$1/g;
    return "\"$param\"";;
}


1;
