# -*- coding: utf-8 -*-
# pylint: disable=missing-docstring,invalid-name,too-many-locals,line-too-long,too-many-branches,too-many-arguments

from __future__ import absolute_import, unicode_literals

import logging
import urlparse

from . import check_sync
from . import cleanup
from . import common
from . import context
from . import downtimes
from . import errors
from . import list_checks
from . import remove_checks


class JugglerApi(object):
    """
    Приводит конфигурацию агрегатов сервера к нужному виду, добавляя или обновляя только те
    агрегаты, которые появились или изменились.

    Ожидается, что методу :func:`upsert_check` будут переданы настройки всех агрегатов, которые
    должны быть сконфигурированы на сервере. Все те агрегаты, что были созданы ранее, но не были
    добавлены в текущую сессию будут удалены при вызове метода :func:`cleanup`.

    Можно использовать экземпляр класса как менеджер контекста, тогда по выходу из него будет вызван
    метод :func:`cleanup`.

    Пример использования::

        from juggler_sdk import JugglerApi, Check

        with JugglerApi("http://juggler-api.search.yandex.net", mark="mark", dry_run=True) as api:
            print(api.upsert_check(Check(host="host", service="service")))

    :param mark: пометить созданные агрегаты данной меткой, необходимо для поиска и удаления
        устаревших агрегатов. В противном случае в качестве метки будет использован хост агрегата.
    :param dry_run: применять ли изменения на север или нет
    :param force: применять параметры агрегата даже если он был создан другим клиентом
    :param oauth_token: токен для доступа к проверкам
    """

    def __init__(self, api_url, mark=None, dry_run=False, force=False, oauth_token=None):
        self.mark = mark
        self.dry_run = dry_run
        self.force = force
        parsed_url = urlparse.urlparse(api_url)
        if parsed_url.path:
            logging.warning("Specifying path in api_url is deprecated")
        if parsed_url.port is not None:
            logging.warning("Specifying port in api_url is deprecated")

        api_url = '{}://{}'.format(parsed_url.scheme, parsed_url.netloc)

        self._context = context.Context(api_url=api_url, oauth_token=oauth_token)

        self._cleaned_up_marks = set()
        # (host, service) tuples
        self._seen_checks = set()
        # simple strings set
        self._seen_marks = set()

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if exc_val is None and not self._cleaned_up_marks:
            self.cleanup()

    def cleanup(self, mark=None):
        """
        Удалить все агрегаты, которые исчезли из данной конфигурации но были созданы ранее.

        :param mark: метка для которой производится cleanup, вместо установленной на уровне класса
        :rtype: juggler_sdk.cleanup.CleanupResult
        """
        if mark is not None:
            marks_to_cleanup = set(common.list_of_str(mark))
        elif self.mark is not None:
            marks_to_cleanup = set(common.list_of_str(self.mark))
        else:
            marks_to_cleanup = self._seen_marks

        if marks_to_cleanup & self._cleaned_up_marks:
            raise errors.ValidationError("don't call cleanup more than once for given mark")
        self._cleaned_up_marks.update(marks_to_cleanup)

        return cleanup.cleanup_checks(self._context, self._seen_checks, marks_to_cleanup, dry_run=self.dry_run)

    def upsert_check(self, check, force=None):
        """
        Добавить новый агрегат к конфигурации. Агрегат сразу отправляется на сервер при наличии изменений.

        :param check: проверка для отправки на сервер
        :type check: juggler_sdk.Check
        :param force: перезаписывает проверку на сервере даже если она создана не через sdk
        :rtype: juggler_sdk.check_sync.ApplyCheckResult
        """
        if (check.host, check.service) in self._seen_checks:
            raise errors.ValidationError(
                "Trying to apply check '{0}:{1}' twice in single session".format(check.host, check.service)
            )

        mark = check.mark or self.mark or check.host
        result = check_sync.apply_check(self._context, check, mark, dry_run=self.dry_run,
                                        force=force if force is not None else self.force)
        self._seen_checks.add((check.host, check.service))
        self._seen_marks.add(mark)
        return result

    def get_one_check(self, host, service, fetch_responsibles=False):
        """
        Получить настройки некоторого агрегата

        :param host: хост агрегат
        :param service: сервис агрегат
        :param fetch_responsibles: получить ответственных из Golem
        :rtype: juggler_sdk.Check
        """
        return check_sync.get_server_check(self._context, host, service, fetch_responsibles)

    def get_checks(self, filters):
        """
        Получить настройки агрегатов

        :param filters: фильтры, по которым надо получить проверки
        :type filters: list[juggler_sdk.CheckFilter]
        :rtype: juggler_sdk.list_checks.ChecksHandleReply
        """
        return list_checks.list_checks(self._context, filters)

    def remove_checks(self, filters):
        """

        Удалить набор агрегатов

        :param filters: фильтры, по которым надо удалить проверки
        :type filters: list[juggler_sdk.CheckFilter]
        :rtype: juggler_sdk.remove_checks.RemoveResult
        """
        return remove_checks.remove_checks(self._context, filters)

    def set_downtimes(self, filters, start_time=None, end_time=None, description=None, source=None, downtime_id=None):
        """
        Установить downtime на набор объектов

        :param filters: список объектов, на которые нужно поставить downtime
        :type filters: list[juggler_sdk.DowntimeSelector]
        :param start_time: время начала downtime, UNIX timestamp, по умолчанию сейчас
        :param end_time: время начала downtime, UNIX timestamp, по умолчанию навсегда
        :param description: текстовое описание downtime
        :param source: проект, к которому относится даунтайм. Используется для упрощения поиска в дальнейшем
        :param downtime_id: id даунтайма, который хочется отредактировать. Если пуст, то будет создан новый даунтайм
        :rtype: juggler_sdk.downtimes.SetDowntimesResponse
        """
        return downtimes.set_downtimes(
            self._context,
            filters=filters,
            end_time=end_time, start_time=start_time,
            description=description, source=source,
            downtime_id=downtime_id
        )

    def remove_downtimes(self, downtime_ids):
        """
        Удалить список downtime по их id

        :param downtime_ids: список id даунтаймов
        :type downtime_ids: list[str]
        :rtype: juggler_sdk.downtimes.RemoveDowntimesResponse
        """
        return downtimes.remove_downtimes(self._context, downtime_ids=downtime_ids)

    def get_downtimes(self, filters, sort_by="ID", page=1, page_size=50):
        """
        Получить список даунтаймов, подходящих под заданный набор фильтров

        :param filters: список фильтров поиска
        :type filters: list[juggler_sdk.DowntimeSearchFilter]
        :param sort_by: поле, по которому надо сортировать даунтаймы. Допустимые значения - ID, START_TIME, END_TIME
        :type sort_by: str
        :param page: страница, которую надо загрузить. Начинается с единицы.
        :param page_size: размер одной страницы
        :rtype: juggler_sdk.downtimes.GetDowntimesResponse
        """
        return downtimes.get_downtimes(self._context, filters=filters, sort_by=sort_by, page=page, page_size=page_size)
