# -*- coding: utf-8 -*-
# pylint: disable=missing-docstring,invalid-name,too-many-locals,line-too-long,too-many-branches

from __future__ import absolute_import, unicode_literals

import collections
import datetime
import getpass
import json
import logging
import numbers
import operator
import socket
import time

from . import common
from . import const
from . import converters
from . import downtimes
from . import errors
from . import types


def make_mark_tag(mark):
    return "a_mark_{0}".format(mark)


def _maybe_number(value):
    # ansible can't pass numbers properly, so let's cast something to it
    if isinstance(value, numbers.Number):
        return value
    elif not isinstance(value, basestring):
        return None
    if value.isdigit():
        return int(value)
    try:
        return float(value)
    except ValueError:
        return None


def _sanitize_module_kwargs(kwargs):
    if kwargs is not None:
        if not isinstance(kwargs, dict):
            raise errors.JugglerError('must be dict')
        if not kwargs:  # empty dict
            kwargs = None
    return kwargs


def _extract_shortcut_options(params, option_key, kwargs_key, known_values):
    matched_keys = [s for s in known_values + [option_key] if s in params]
    if len(matched_keys) > 1:
        raise errors.ValidationError('{0} specified more than once'.format(option_key),
                                     keys=matched_keys)

    if len(matched_keys) == 0:
        kwargs_value = _sanitize_module_kwargs(common.pop_value(params, kwargs_key, default=None))
        return None, kwargs_value

    found_key = matched_keys[0]
    if found_key == option_key:
        option_value = str(common.pop_value(params, found_key))
        kwargs_value = _sanitize_module_kwargs(common.pop_value(params, kwargs_key, default=None))
    else:
        option_value = str(found_key)
        kwargs_value = _sanitize_module_kwargs(common.pop_value(params, found_key))

    return option_value, kwargs_value


def _create_child(child_def, default_host, default_service):
    if isinstance(child_def, basestring):
        return types.Child.from_string(child_def, default_host, default_service)
    elif isinstance(child_def, dict):
        return types.Child.from_dict(child_def, default_host, default_service)
    elif isinstance(child_def, types.Child):
        return child_def
    else:
        raise errors.ValidationError(
            "Child must be either a string or a dictionary, {0} found".format(type(child_def).__name__),
            child=child_def)


def _sanitize_ansible_flap_config(raw_flaps, default_flaps):
    if raw_flaps in (True, "true"):
        return default_flaps
    elif raw_flaps in (False, None, "", "false"):
        return None
    elif isinstance(raw_flaps, types.FlapOptions):
        return raw_flaps
    elif _maybe_number(raw_flaps) is not None:
        raw_flaps = _maybe_number(raw_flaps)
        return types.FlapOptions(
            stable=raw_flaps,
            critical=raw_flaps * default_flaps.critical / default_flaps.stable,
            boost=default_flaps.boost
        )
    elif isinstance(raw_flaps, dict):
        if 'turbulence' in raw_flaps:
            # todo: remove in Jan, 2018
            raise errors.ValidationError("'turbulence' is renamed to 'critical', please replace it in your playbook")
        if 'delay' in raw_flaps:
            # todo: remove in Feb, 2018
            raise errors.ValidationError("'delay' in flap config is deprecated, please remove it from your playbook")

        required_keys = {'stable', 'critical'}
        optional_keys = {'delay', 'boost'}
        missing_required_keys = required_keys - raw_flaps.viewkeys()
        if missing_required_keys:
            raise errors.ValidationError('missing required keys ({}) for flaps configuration'.format(
                missing_required_keys
            ))
        unknown_keys = raw_flaps.viewkeys() - (required_keys | optional_keys)
        if unknown_keys:
            raise errors.ValidationError('unknown keys ({}) in flaps configuration'.format(unknown_keys))
        return types.FlapOptions(
            stable=int(raw_flaps['stable']),
            critical=int(raw_flaps['critical']),
            boost=int(raw_flaps['boost']) if 'boost' in raw_flaps else default_flaps.boost
        )
    else:
        raise errors.ValidationError("Flaps must be either a dict, bool, number or None, found {0!r}".format(raw_flaps))


def make_check_from_ansible_dict(params, context):
    pop = common.pop_value

    params = params.copy()
    host = str(pop(params, 'host'))
    service = str(pop(params, 'service'))
    mark = str(pop(params, 'jcheck_mark', default=host))

    aggregator, aggregator_kwargs = _extract_shortcut_options(params,
                                                              option_key='aggregator',
                                                              kwargs_key='aggregator_kwargs',
                                                              known_values=context.known_aggregators)
    active, active_kwargs = _extract_shortcut_options(params,
                                                      option_key='active',
                                                      kwargs_key='active_kwargs',
                                                      known_values=context.known_actives)

    check_options = pop(params, 'check_options', default=None)

    # todo: remove in Feb, 2018
    if "passive_owner" in params:
        raise errors.ValidationError("'passive_owner' is deprecated, please remove it from your playbook")
    if "active_owner" in params:
        raise errors.ValidationError("'active_owner' is deprecated, please remove it from your playbook")

    ttl = pop(params, 'ttl', default=None)
    if ttl is not None:
        ttl = int(ttl)

    refresh_time = pop(params, 'refresh_time', default=None)
    if refresh_time is not None:
        refresh_time = int(refresh_time)

    responsible = None
    if 'responsible' in params:
        responsible = common.list_of_str(pop(params, 'responsible'))

    alert_methods = common.list_of_str(pop(params, 'alert_method', default=[]))

    raw_children = pop(params, 'children', default=[])
    children = [
        _create_child(x, host, service)
        for x in
        (raw_children if isinstance(raw_children, (list, tuple, set)) else [raw_children])
    ]

    counter = collections.Counter(children)
    if len(counter) != len(children):
        raise errors.ValidationError("Following children appear more than once: {0}".format(
            ", ".join(map(repr, [child for child, count in counter.iteritems() if count > 1]))
        ))

    flaps_config = _sanitize_ansible_flap_config(pop(params, 'flap', default=None), context.default_flaps)

    raw_notifications = pop(params, 'notifications', default=[])
    if not isinstance(raw_notifications, (list, tuple, set)):
        raise errors.ValidationError("Notifications must be a list, {0!r} found".format(raw_notifications))
    notifications = [(types.NotificationOptions(
        template_name=x.pop('template'),
        description=x.pop('description', None),
        template_kwargs=x
    ) if not isinstance(x, types.NotificationOptions) else x) for x in raw_notifications]

    # we keep tags order here
    tags = common.list_of_str(pop(params, 'tags', default=[]))

    meta = pop(params, 'meta', default=None)
    if meta is not None and not isinstance(meta, dict):
        raise errors.ValidationError("Meta must be a dictionary")

    namespace = common.str_none(pop(params, 'namespace', default=None))

    alert_interval = pop(params, 'alert_interval', default=None)
    if alert_interval is not None:
        if not isinstance(alert_interval, (list, tuple)) or not len(alert_interval) in (0, 2):
            raise errors.ValidationError('Alert interval must be none or a list of zero or two items')

    if params:
        raise errors.ValidationError("Excessive configuration, fields '{0}' are unknown".format(
            ', '.join(map(str, params.viewkeys()))
        ), details=params)

    return types.Check(
        host, service, ttl, refresh_time,
        aggregator, aggregator_kwargs, active, active_kwargs, check_options,
        responsible, alert_methods, alert_interval, children,
        flaps_config, notifications, tags, meta, namespace, mark=mark
    )


def _fetch_checks_by_tag(context, tag_name):
    checks_dict = common.fetch_json(context, handle='/api/checks/checks', tag_name=tag_name,
                                    include_notifications=True, include_children=True, include_meta=True)
    for host, service_dict in checks_dict.iteritems():
        for service, srv_check in service_dict.iteritems():
            yield host, service, converters.check_from_api_v1_reply(host, service, srv_check)


def _get_server_check_cached(context, host, service, mark):
    if mark not in context.fetched_marks:
        for check_host, check_service, check in _fetch_checks_by_tag(context, make_mark_tag(mark)):
            context.server_checks_cache[(check_host, check_service)] = check

        context.fetched_marks.add(mark)

    return context.server_checks_cache.get((host, service))


def _fetch_golem_responsibles(context, host, retries=3):
    for idx in xrange(retries):
        try:
            data = common.fetch_json(context, api_url=context.golem_read_url,
                                     handle='/api/get_object_resps.sbml', object=host)
        except errors.JugglerError as exc:
            if exc.kwargs.get('code') == 400:
                # golem can return 400 for non-existed object
                return []
            elif idx + 1 == retries:
                raise
        else:
            return map(str, filter(None, (resp['name'] for resp in data['resps'])))


def _get_golem_responsible(context, host):
    if host not in context.golem_responsible_cache:
        context.golem_responsible_cache[host] = _fetch_golem_responsibles(context, host)

    return context.golem_responsible_cache[host]


def get_server_check(context, host, service, fetch_responsibles=False):
    reply = common.fetch_json(context, handle='/api/checks/checks', host_name=host, service_name=service,
                              include_notifications=True, include_children=True, include_meta=True)
    if host not in reply or service not in reply[host]:
        return None

    check = converters.check_from_api_v1_reply(host, service, reply[host][service])
    if fetch_responsibles:
        check.responsible = _get_golem_responsible(context, host)

    return check


def _raise_if_untouchable(server_check, mark):
    if not server_check:
        return

    description = server_check.description
    try:
        description = json.loads(description)
        description = dict(description)
    except Exception:
        raise errors.JugglerError(
            "Non-JSON description of check. Keep clear or use __force__", description=description
        )
    if description.get('with') != const.USER_AGENT:
        raise errors.JugglerError(
            "'description.with' is not {0!r}. Keep clear or use __force__".format(const.USER_AGENT),
            description=description)
    if description.get('mark') != mark:
        raise errors.JugglerError(
            "'description.mark' is not {0!r}, that's probably not your check. Keep clear or use __force__".format(mark),
            description=description)


def _make_description(jcheck_mark=None):
    desc = {
        'by': getpass.getuser(),
        'when_tzlocal': str(datetime.datetime.fromtimestamp(int(time.time()))),
        'where': socket.gethostname(),
        'with': const.USER_AGENT,
    }
    if jcheck_mark is not None:
        desc['mark'] = jcheck_mark
    return json.dumps(desc, sort_keys=True)


def _set_server_check(context, check, mark):
    description = _make_description(mark)
    body = converters.check_to_add_or_update_object(check, description)

    common.fetch_json(context, handle="/api/checks/add_or_update", body=json.dumps(body),
                      headers={"Content-Type": "application/json"})


@common.dict_based_equality
class FieldDifference(object):
    def __init__(self, name, current, desired):
        self.name = name
        self.current = current
        self.desired = desired


class CheckDifference(object):
    def __init__(self):
        self._fields = []

    def __nonzero__(self):
        return not self.is_empty

    @property
    def is_empty(self):
        return not self._fields

    def add(self, name, current, desired):
        self._fields.append(FieldDifference(name, current, desired))

    def contains_field(self, name):
        return any(x.name == name for x in self._fields)

    def to_dict(self):
        return {'changes': {x.name: {'current': x.current, 'desired': x.desired} for x in self._fields}}


class FullCheckDifference(object):
    def __init__(self, check):
        self._check = check
        self.is_empty = False

    def to_dict(self):
        return converters.check_to_add_or_update_object(self._check, '')

    def contains_field(self, name):
        return True

    def __nonzero__(self):
        return not self.is_empty


def find_difference(desired_check, server_check, context):
    if not server_check:
        return FullCheckDifference(desired_check)

    changes = CheckDifference()

    def compare_field(attr, preprocess=lambda x: x):
        desired = getattr(desired_check, attr)
        current = getattr(server_check, attr)
        if preprocess(current) != preprocess(desired):
            changes.add(attr, current, desired)

    compare_field("aggregator")
    compare_field("aggregator_kwargs")
    compare_field("active")
    compare_field("active_kwargs")
    compare_field("check_options")
    compare_field("ttl", preprocess=lambda x: None if x == context.default_ttl else x)
    compare_field("refresh_time", preprocess=lambda x: None if x == context.default_refresh_time else x)
    compare_field("alert_methods")
    compare_field("alert_interval")
    compare_field("tags", sorted)
    compare_field("meta")
    compare_field("namespace")

    if desired_check.flaps_config != server_check.flaps_config:
        changes.add('flaps_config',
                    current=server_check.flaps_config.to_dict() if server_check.flaps_config is not None else None,
                    desired=desired_check.flaps_config.to_dict() if desired_check.flaps_config is not None else None)

    sort_key = operator.attrgetter('template_name', 'template_kwargs', 'description')
    if sorted(desired_check.notifications, key=sort_key) != sorted(server_check.notifications, key=sort_key):
        changes.add('notifications',
                    current=[x.to_dict() for x in server_check.notifications],
                    desired=[x.to_dict() for x in desired_check.notifications])

    sort_key = operator.attrgetter('group_type', 'host', 'service', 'instance')
    if sorted(desired_check.children, key=sort_key) != sorted(server_check.children, key=sort_key):
        changes.add('children',
                    current=[x.to_dict() for x in server_check.children],
                    desired=[x.to_dict() for x in desired_check.children])

    return changes


def _sync_golem_responsible(context, check, dry_run=False):
    desired_list = check.responsible or []

    has_alerts = check.alert_methods or any(x.template_name == "golem" for x in check.notifications)
    if not has_alerts:
        return []  # not syncing if there's no alerts

    actual_list = _get_golem_responsible(context, check.host)

    if not desired_list:
        if not actual_list and has_alerts:
            raise errors.JugglerError("responsible list was not set and there are no responsible in Golem")
    else:
        if not actual_list:
            if not dry_run:
                common.fetch_json(context, handle='/api/proxy/golem/create_host',
                                  host_name=check.host, resps_list=desired_list)
                # we assume that everything worked fine in golem if we didn't die with exception here
                context.golem_responsible_cache[check.host] = desired_list
            return desired_list
        elif actual_list != desired_list:
            logging.warning(
                "Responsible list didn't match, fix it at https://golem.yandex-team.ru/hostinfo.sbml?object=%s: golem %s, playbook %s",
                check.host, actual_list, desired_list)
    return []


class ApplyCheckResult(object):
    """
    Результат применения проверки на сервере

    :param changed: изменилась проверка или нет
    :param diff: найденные изменения в проверке
    :type diff: juggler_sdk.check_sync.CheckDifference
    :param golem_resps: установленные ответственные за хост в големе
    """

    def __init__(self, changed, diff, golem_resps):
        self.changed = changed
        self.diff = diff
        self.updated_golem_resps = golem_resps


def apply_check(context, check, mark, dry_run=False, force=False):
    """
    :type check: types.Check
    """
    mark_tag = make_mark_tag(mark)
    if mark_tag not in check.tags:
        check.tags.append(mark_tag)

    server_check = _get_server_check_cached(context, check.host, check.service, mark)

    if server_check and not force:
        _raise_if_untouchable(server_check, mark)

    todo_golem_resps = _sync_golem_responsible(context, check, dry_run)
    difference = find_difference(check, server_check, context)

    if not dry_run and difference:
        if (not server_check or difference.contains_field('children')) and context.new_check_downtime:
            # it's not very efficient to do this for every check separately
            # but unless we want to use a post-task, this will do
            downtimes.set_downtimes(
                context=context,
                filters=[downtimes.DowntimeSelector(
                    host=check.host,
                    service=check.service,
                    namespace=check.namespace
                )],
                end_time=time.time() + context.new_check_downtime,
                description="Automatic downtime for the new check",
            )

        _set_server_check(context, check, mark)

    return ApplyCheckResult(
        changed=not difference.is_empty or bool(todo_golem_resps),
        diff=difference,
        golem_resps=todo_golem_resps
    )
