# -*- coding: utf-8 -*-

from __future__ import absolute_import, unicode_literals

import json
import itertools

from . import const
from . import common


class CleanupResult(object):
    """
    Результат чистки проверок на сервере

    :param removed: проверки которые были удалены
    :type removed: list[tuple[host, service]]
    """

    def __init__(self, removed):
        # список туплей (host, service)
        self.removed = removed


def _can_be_removed(srv_check, known_marks):
    description = srv_check['description']
    try:
        description = json.loads(description)
        return description.get('with') == const.USER_AGENT and description.get('mark') in known_marks
    except Exception:  # pylint: disable=broad-except
        return False


def _chunks(iterable, chunk_size=1000):
    iterable = iter(iterable)
    while True:
        yield list(itertools.chain([next(iterable)], itertools.islice(iterable, chunk_size - 1)))


def _get_known_server_checks(context, known_marks):
    known_checks = set()

    for tags_chunk in _chunks(sorted(known_marks), 10):
        blinov_filter = ' '.join('a_mark_{0}'.format(x) for x in tags_chunk)
        reply = common.fetch_json(context, '/api/checks/checks', tag_name=blinov_filter)

        for host, svc_dict in reply.iteritems():
            for service, srv_check in svc_dict.iteritems():
                if _can_be_removed(srv_check, known_marks):
                    known_checks.add((host, service))

    return known_checks


def cleanup_checks(context, seen_checks, marks_set, dry_run=False):
    server_check_keys = _get_known_server_checks(context, marks_set)
    todo = sorted(server_check_keys - seen_checks)

    if not dry_run:
        for host, service in todo:
            common.fetch_json(context, handle='/api/checks/remove_check', host_name=host, service_name=service)
    return CleanupResult(removed=todo)
