# -*- coding: utf-8 -*-
# pylint: disable=missing-docstring,invalid-name,too-many-locals,line-too-long,too-many-branches

from __future__ import absolute_import

import json

from . import errors
from . import common


@common.dict_based_equality
class DowntimeSelector(object):
    """
    Селектор объектов к которым относится даунтайм.

    :param host: имя объекта/хоста
    :param service: имя сервиса
    :param instance: инстанс событий
    :param namespace: неймспейс проверок
    :param tags: список тегов объектов
    """

    def __init__(self, host=None, service=None, instance=None, namespace=None, tags=None, **kwargs):
        self.fields = common.cleanup_none({
            "host": host,
            "service": service,
            "instance": instance,
            "namespace": namespace,
            "tags": tags
        })
        self.fields.update(kwargs)

    def __getattr__(self, name):
        if name in self.fields:
            return self.fields[name]
        raise AttributeError()


@common.dict_based_equality
class DowntimeSearchFilter(object):
    """
    Фильтр, по которому можно искать установленные даунтаймы

    :param host: имя объекта/хоста
    :param service: имя сервиса
    :param instance: инстанс событий
    :param namespace: неймспейс проверок
    :param tags: список тегов объектов
    :param source: проект, установивший даунтайм
    :param downtime_id: непосредственный id объекта даунтайма
    """

    def __init__(self, host=None, service=None, instance=None, namespace=None,
                 tags=None, downtime_id=None, source=None, **kwargs):
        self.fields = common.cleanup_none({
            "host": host,
            "service": service,
            "instance": instance,
            "namespace": namespace,
            "tags": tags,
            "downtime_id": downtime_id,
            "source": source
        })
        self.fields.update(kwargs)

    def __getattr__(self, name):
        if name in self.fields:
            return self.fields[name]
        raise AttributeError()


@common.dict_based_equality
class Downtime(object):
    """
    Объект даунтайма, установленного на сервере

    :param downtime_id: id объекта, по которому его потом можно найти
    :param filters: список фильтров объектов
    :param start_time: время, с которого должен действовать этот даунтайм
    :param end_time: время, до которого должен действовать этот даунтайм. 0 означает навсегда.
    :param description: текстовое описание даунтайма
    :param source: проект, установивший даунтайм
    """

    def __init__(self, downtime_id, filters, start_time, end_time, description, source):
        self.downtime_id = downtime_id
        self.filters = filters
        self.start_time = start_time
        self.end_time = end_time
        self.description = description
        self.source = source


@common.dict_based_equality
class SetDowntimesResponse(object):
    """
    :param downtime_id: id установленного даунтайма
    """

    def __init__(self, downtime_id):
        self.downtime_id = downtime_id


@common.dict_based_equality
class RemoveDowntimesResponse(object):
    """
    :param downtime_id: удаленные даунтаймы
    :type downtimes: list[juggler_sdk.Downtime]
    """
    def __init__(self, downtimes):
        self.downtimes = downtimes


@common.dict_based_equality
class GetDowntimesResponse(object):
    """
    :param items: найденные даунтаймы
    :type items: list[juggler_sdk.Downtime]
    :param page: текущая страница
    :param page_size: размер страницы
    :param total: общее количество даунтаймов, подходяищих под фильтр
    """

    def __init__(self, items, page, page_size, total):
        self.items = items
        self.page = page
        self.page_size = page_size
        self.total = total


def _downtime_from_dict(dt):
    return Downtime(
        downtime_id=dt["downtime_id"],
        filters=[DowntimeSelector(**x) for x in dt["filters"]],
        start_time=dt["start_time"],
        end_time=dt["end_time"],
        description=dt["description"],
        source=dt["source"]
    )


def set_downtimes(context, filters, end_time, start_time=None, description=None, source=None, downtime_id=None):
    if not filters:
        raise errors.ValidationError('No filters specified')

    request = common.cleanup_none({
        "filters": [common.cleanup_none(x.fields) for x in filters],
        "start_time": start_time,
        "end_time": end_time,
        "source": source,
        "description": description,
        "downtime_id": downtime_id
    })
    reply = common.fetch_json(context, '/v2/downtimes/set_downtimes', body=json.dumps(request))
    return SetDowntimesResponse(downtime_id=reply["downtime_id"])


def remove_downtimes(context, downtime_ids):
    if not downtime_ids:
        raise errors.ValidationError("You have to specify at least one downtime id")
    reply = common.fetch_json(context, '/v2/downtimes/remove_downtimes', body=json.dumps({
        "downtime_ids": downtime_ids
    }))
    return RemoveDowntimesResponse(downtimes=map(_downtime_from_dict, reply["downtimes"]))


def get_downtimes(context, filters, sort_by="ID", page=1, page_size=50):
    reply = common.fetch_json(context, '/v2/downtimes/get_downtimes', body=json.dumps({
        "filters": [common.cleanup_none(x.fields) for x in filters],
        "sort_by": sort_by,
        "page": page,
        "page_size": page_size
    }))
    return GetDowntimesResponse(
        items=map(_downtime_from_dict, reply["items"]),
        page=reply["page"],
        page_size=reply["page_size"],
        total=reply["total"]
    )
