# -*- coding: utf-8 -*-
from __future__ import absolute_import

from . import common
from . import errors

HOST_GROUP_TYPE = 'HOST'


class Child(object):
    """
    Child проверки, может представлять группу или сырое событие

    :param host: имя хоста сырого события или имя группы
    :param service: сервис объектов для проверки
    :param instance: числовой инстанс событий, либо специальное слово all
    :param group_type: тип группы объекта, HOST по-умолчанию (сырое событие)
    """
    __slots__ = ('host', 'service', 'instance', 'group_type')

    def __init__(self, host, service, instance='', group_type=None):
        self.host = host
        self.service = service
        self.instance = instance or ''
        self.group_type = group_type or HOST_GROUP_TYPE

    @property
    def is_group(self):
        return self.group_type != HOST_GROUP_TYPE

    @classmethod
    def from_string(cls, string, default_name=None, default_service=None):
        if string in ('', ':'):
            raise errors.ValidationError("Invalid child, possible configuration loop", child=string)

        if '%' in string:
            child_type, child_name = string.split('%', 1)
            if not child_type:
                raise errors.ValidationError("Child cannot start with % - child type is required",
                                             child=string, object_name=child_name)
        else:
            child_type, child_name = None, string

        parts = child_name.split(":")
        if len(parts) > 3:
            raise errors.ValidationError("Invalid child, too many colons in name", child=string, object_name=child_name)

        return cls(
            group_type=child_type,
            host=parts[0] or default_name,
            service=parts[1] if (len(parts) > 1 and parts[1]) else default_service,
            instance=parts[2] if len(parts) > 2 else None
        )

    @classmethod
    def from_dict(cls, data, default_name=None, default_service=None):
        allowed_fields = {'type', 'is_group', 'name', 'host', 'service', 'instance'}
        extra_fields = data.viewkeys() - allowed_fields
        if extra_fields:
            raise errors.ValidationError('Extra fields specified in child configuration',
                                         child=data, extra_fields=extra_fields)

        if 'name' in data and 'host' in data:
            raise errors.ValidationError('Both name and host cannot be present in dict', child=data)
        host = data.get('host', data.get('name')) or default_name
        service = data.get('service') or default_service
        instance = str(data['instance']) if data.get('instance') is not None else None

        if 'is_group' in data:
            if data['is_group']:
                if data.get('type') is None:
                    raise errors.ValidationError(
                        "Explicit group type is required, you cannot have 'is_group' without 'type'", child=data
                    )
                child_type = data['type']
            else:
                child_type = None
        else:
            child_type = data.get('type')

        return cls(group_type=child_type, host=host, service=service, instance=instance)

    def to_dict(self):
        return {'group_type': self.group_type, 'host': self.host, 'service': self.service, 'instance': self.instance}

    def __eq__(self, other):
        return self.group_type == other.group_type and \
               self.host == other.host and \
               self.service == other.service and \
               self.instance == other.instance

    def __ne__(self, other):
        return not self == other

    def __hash__(self):
        return hash((self.group_type, self.host, self.service, self.instance))

    def __repr__(self):
        return 'Child(type=%r, host=%r, service=%r, instance=%r)' % (
            self.group_type, self.host, self.service, self.instance)


@common.dict_based_equality
class FlapOptions(object):
    """
        Конфигурация флаподава агрегата
    """

    def __init__(self, stable, critical, boost=0):
        self.stable = stable
        self.critical = critical
        self.boost = boost

    def to_dict(self):
        return self.__dict__.copy()


@common.dict_based_equality
class NotificationOptions(object):
    """
    Конфигурация нотификации

    :param template_name: имя шаблона нотификации
    :param template_kwargs: словарь с параметрами этого шаблона
    :param description: строковое описание
    """
    def __init__(self, template_name, template_kwargs, description=None):
        self.template_name = template_name
        self.template_kwargs = template_kwargs
        self.description = description

    def to_dict(self):
        return self.__dict__.copy()


@common.dict_based_equality
class Check(object):
    """
    Конфигурация агрегатной проверки

    :param host: хост проверки
    :param service: сервис проверки
    :param ttl: время после которого зажигается NO DATA, по-умолчанию используется дефолт сервера
    :param refresh_time: время обновления проверки, по-умолчанию используется дефолт сервера
    :param aggregator: имя агрегатора, дефолтно пустой
    :param aggregator_kwargs: словарь с параметрами агрегатора
    :param active: имя модуля активной проверки, дефолтно пустое
    :param active_kwargs: словарь с параметрами модуля активной проверки
    :param check_options: словарь с опциями для пассивной проверки, дефолтно пустой
    :param responsible: список ответственных за хост в Golem
    :param alert_methods: методы нотификации, доступен только GOLEM, deprecated
    :type alert_methods: list[str]
    :param alert_interval: интервал оповещения о проверке, deprecated
    :type alert_interval: list[str]|None
    :param children: объекты которые надо агрегировать, дефолтно пусой массив
    :type children: list[juggler_sdk.Child]
    :param flaps_config: конфигурация флаподава, дефолтно выключен
    :type flaps_config: juggler_sdk.FlapOptions
    :param notifications: список нотификаций, дефолтно пустой
    :type notifications: list[juggler_sdk.NotificationOptions]
    :param tags: список тэгов для удобного поиска по ним
    :type tags: list[str]
    :param meta: словарь с метаданными
    :param namespace: имя пространста имен проверки для ограничения доступа
    :param description: строковое описание
    :param mark: маркер проверки для дальнейшего удаления проверок по маркеру
    """
    def __init__(self, host, service, ttl=None, refresh_time=None,
                 aggregator=None, aggregator_kwargs=None,
                 active=None, active_kwargs=None, check_options=None,
                 responsible=None, alert_methods=(), alert_interval=None, children=(),
                 flaps_config=None, notifications=(), tags=(),
                 meta=None, namespace=None, description=None, mark=None):
        self.host = host
        self.service = service
        self.aggregator = aggregator
        self.aggregator_kwargs = aggregator_kwargs
        self.active = active
        self.active_kwargs = active_kwargs
        self.check_options = check_options
        self.ttl = ttl
        self.refresh_time = refresh_time
        self.responsible = responsible
        self.alert_methods = list(alert_methods)
        self.alert_interval = alert_interval
        self.children = list(children)
        self.flaps_config = flaps_config
        self.notifications = list(notifications)
        self.tags = list(tags)
        self.meta = meta or {}
        self.namespace = namespace
        self.description = description
        self.mark = mark

    def to_dict(self, minimize=False):
        d = {
            'host': self.host,
            'service': self.service,
            'refresh_time': self.refresh_time,
            "ttl": self.ttl,
            "description": self.description,
            "aggregator": self.aggregator,
            "aggregator_kwargs": self.aggregator_kwargs,
            "active": self.active,
            "active_kwargs": self.active_kwargs,
            "check_options": self.check_options,
            "flaps": self.flaps_config.to_dict() if self.flaps_config is not None else None,
            "tags": self.tags,
            "methods": self.alert_methods,
            "alert_interval": self.alert_interval,
            "children": [x.to_dict() for x in self.children],
            "notifications": [x.to_dict() for x in self.notifications],
            "meta": self.meta,
            "namespace": self.namespace,
            "mark": self.mark
        }
        if minimize:
            d = self._minimize_dict(d)
        return d

    @classmethod
    def from_dict(cls, data):
        return cls(
            host=data['host'],
            service=data['service'],
            refresh_time=data.get('refresh_time'),
            ttl=data.get('ttl'),
            description=data.get('description'),
            aggregator=data.get('aggregator'),
            aggregator_kwargs=data.get('aggregator_kwargs'),
            active=data.get('active'),
            active_kwargs=data.get('active_kwargs'),
            check_options=data.get('check_options'),
            flaps_config=FlapOptions(**data['flaps']) if data.get('flaps') else None,
            tags=data.get('tags') or (),
            responsible=data.get('responsible') or (),
            alert_methods=data.get('methods') or (),
            alert_interval=data.get('alert_interval'),
            children=[Child(**x) for x in (data.get('children') or ())],
            notifications=[NotificationOptions(**x) for x in (data.get('notifications') or ())],
            meta=data.get('meta'),
            namespace=data.get('namespace'),
            mark=data.get('mark')
        )

    @staticmethod
    def _minimize_dict(dict_data):
        minimized = {k: v for k, v in dict_data.iteritems() if v}
        minimized.pop('description', None)
        return minimized

    def __repr__(self):
        return repr(self.__dict__)


@common.dict_based_equality
class CheckFilter(object):
    """
    Фильтр проверок. Хотя бы один параметр должен быть установлен

    :param host: хост проверки
    :param service: сервис проверки
    :param namespace: имя пространста имен проверки
    :param tags: тэги проверки
    :type tags: list[str]
    """

    def __init__(self, host=None, service=None, namespace=None, tags=None):
        self.host = host
        self.namespace = namespace
        self.service = service
        self.tags = tags

    @property
    def is_empty(self):
        return not (self.host or self.service or self.namespace or self.tags)
