#!/usr/bin/perl

use strict;
use warnings;

use YAML;
use List::MoreUtils qw/zip uniq/;
use List::Util qw/sum/;

my @quantiles = qw/10 20 30 40 50 60 70 80 90 95 98 99/;

run() unless caller();

sub run
{
    my @sources = @ARGV;

    my %STAT;
    for my $source (@sources){
        open(my $fh, "<", $source) or die;
        $source =~ s!^.*/!!;
        my @data = <$fh>;
        $STAT{$source} = calc_stat(\@data);
    }
    print Dump(\%STAT);

    my @row_titles = sort { $a cmp $b } uniq map {keys %{$STAT{$_}} } @sources;

    my %col_titles = map { $_ => [keys %{$STAT{$_}->{$row_titles[0]}}] } @sources;

    my @headings;
    for my $s (@sources){
        push @headings, map { "$s/$_" } @{$col_titles{$s}};
    }
    print join(" ", map {sprintf("%15s", $_)} " ", @headings)."\n";
    for my $r (@row_titles){
        my @values = (sprintf("%15s", $r));
        for my $s (@sources){
            push @values, map { sprintf("%15.1f", $STAT{$s}->{$r}->{$_}) } @{$col_titles{$s}};
        }
        print join(" ", @values)."\n";
    }
}

sub calc_stat
{
    my ($raw_data) = @_;
    my @titles = split /\s+/, shift @$raw_data;
    my $key = $titles[0];
    my @data;
    for my $d (@$raw_data){
        my @values = split /\s+/, $d;
        push @data, { zip @titles, @values };
    }
    @data = sort { $a->{$key} <=> $b->{$key} } @data;
    my @indices = (1 .. scalar(@data));
    print Dump([ zip @indices, @data ]);

    my %stat;
    my $N = scalar @data;
    my %border = map { $_ => int(($N * $_)/100) } @quantiles;

    my $low_border = 0;
    my $prev_quantile = '00';
    for my $q (@quantiles){
        my $upper_border = $border{$q};
        my $h = {
            "${key}_max" => $data[$upper_border]->{$key},
        };
        my $key_total = sum map {$_->{$key}} @data[$low_border, $upper_border];
        for my $f (grep { $_ ne $key } @titles){
            my $f_total = sum map {$_->{$f}} @data[$low_border, $upper_border];
            $h->{"${f}_avg"} = int( 1000 * $f_total / $key_total )/10;
        }
        $stat{"q_${prev_quantile}_$q"} = $h;
        $low_border = $upper_border + 1;
        $prev_quantile = $q;
    }

    return \%stat;
}

