#!/usr/bin/perl -w

=head1 NAME
    
    log_arc.pl

=head1 DESCRIPTION

    Архивирование старых логов

    Обязательная опция --logroot -- директория с логами
    Необязательные опции
       --days-back N - не удалять логи за последние N дней (default 7)
       --num-processes N - архивировать в N процессов (default 1)
       --archive-all - архивировать все логи (days-back игнорируется)
    Принимаем множественные опции
        --logroot - для обработки нескольких директорий
        --ignore filename.log (не обрабатываем)
        --emptify xls_import.log=N (файлы удаляем через N дней)

=cut

use strict;
use File::Basename;
use File::Path;
use Getopt::Long;
use Pid::File::Flock qw/:auto/;
use POSIX qw/strftime/;
use Parallel::ForkManager;


my @LOG_ROOT = ();
my @IGNORE = ();
my %EMPTIFY = ();

# на сколько дней назад отступаем для архивирования/переноса в дирректории по месяцам
my $DAYS_BACK = 7;

# количество процессов для архивирования логов. Один процесс - один лог
# If you specify 0, then NO fork will be done; this is good for debugging purposes.
my $NUM_PROCESSES = 0;
my $ARCHIVE_ALL;

GetOptions(
    "help"      => sub {system("podselect -section NAME -section DESCRIPTION $0 | pod2text-utf8 >&2"); exit 0;},
    'logroot=s' => \@LOG_ROOT,
    'ignore=s'  => \@IGNORE,
    'emptify=s' => \%EMPTIFY,
    'days-back=i' => \$DAYS_BACK,
    'num-processes=i' => \$NUM_PROCESSES,
    'archive-all' => \$ARCHIVE_ALL,
);

die 'no logroot was specified, see --help' unless scalar @LOG_ROOT;
$DAYS_BACK = -32 if $ARCHIVE_ALL;

my $RE_IGNORE = '^('.join("|", map {"\Q$_\E"} @IGNORE).')$';

my @files = ();
# получаем список логов
for my $cur_log_root (@LOG_ROOT) {
    opendir(DIR, $cur_log_root) || die "Can't open dir $cur_log_root: $!";
    push @files,
        grep {-f $_}
        map {"$cur_log_root/$_"}
        sort
        grep {/^([\w\.\-]+)$/}
        grep {!/$RE_IGNORE/}
        readdir(DIR);
    closedir(DIR);
}

# определяем границы
my $day_border = strftime("%Y%m%d", localtime(time-$DAYS_BACK*24*60*60));
my $mon_border = strftime("%Y%m", localtime(time-$DAYS_BACK*24*60*60));

# работа с файлами
my @arc_logs = ();
my $EMPTIFY_RE = join("|", map {"\Q$_\E"} keys %EMPTIFY);
for my $file (@files) {
    print STDERR "process file $file\n" if $ENV{DEBUG};
    if ($file =~ /\/($EMPTIFY_RE)\.\d+$/) {
        my $days = $EMPTIFY{$1};
        print STDERR "$file in emptify list\n" if $ENV{DEBUG};
        if ((stat $file)[9] < time - $days * 24 * 60 * 60) {
            print STDERR " unlink\n" if $ENV{DEBUG};
            unlink($file) || die "Can't unlink $file: $!";
        } else {
            print STDERR " skip\n" if $ENV{DEBUG};
        }
    } elsif ($file =~ /\.((\d{6})\d{2})$/) {
        # дневной лог
        if ($1 < $day_border) {
            push @arc_logs, [$file, dirname($file) . "/$2"];
        }
    } elsif ($file =~ /\.(\d{6})$/) {
        # месячный лог
        if ($1 < $mon_border) {
            push @arc_logs, [$file, dirname($file) . "/$1"];
        }
    }
}

my $pm = new Parallel::ForkManager($NUM_PROCESSES);
# собственно архивируем/переносим логи
for my $rec (@arc_logs) {
    $pm->start and next;
    # не пытаться удалять лок при смерти дочернего процесса с gzip
    Pid::File::Flock->abandon;

    my ($file, $arc_dir) = @$rec;
    # создаём архивную директорию
    mkpath($arc_dir);

    # архивируем
    system("gzip -c $file >$arc_dir/".basename($file).".gz") == 0
        or die "Can't gzip $file: $!";
    unlink($file) || die "Can't unlink $file: $!";

    $pm->finish;
}
$pm->wait_all_children;
