#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
Загрузка файловых логов (с ppcscripts и ppcbackup) в YT по заданному
файл-листу.

Скрипт не должен падать на битых архивах с логами. Но может падать,
если лог был удален (совсем побился, права сменились и тд) к моменту
загрузки. В этом случае его нужно руками убирать из файл-листа.

Если обрабатывать это автоматически, нужно учитывать возможность
расхождения позиции в файллисте и количества реально загруженных
файлов. Но этого не хочется, и проблема пока редкая.
"""

import argparse
import fcntl
import gzip
import json
import logging
import os
import Queue
import cStringIO as StringIO  # io.byteio и io.stringio в python2 работают сильно медленее
import shutil
import signal
import sys
import traceback
import threading
import time
import yt.wrapper as yt
import yt.packages.requests as ytreq
import yt_yson_bindings
import zlib

from direct_juggler.juggler import queue_events

# какую пачку вычитывать из лога за один раз
# после сериализации должна умещаться в MAX_ROW_WEIGHT
# в идеале для этого надо сериализовать, проверять, резать, и тд, но это значение пока работает
LOG_CHUNK_SIZE = 32*1024*1024
# размер подготовленных данных, после которых начинать писать их в yt, >= дефолтному размеру yt chunk (512mb)
YT_WRITE_SIZE = 512*1024*1024
# максимальный размер строки в YT
MAX_ROW_WEIGHT = 128*1024*1024

MAX_CHUNK_UPLOAD_TIME = 1800

TABLE_ATTR = {
    'schema': [
        {'name': 'log_date', 'type': 'string'},
        {'name': 'log_name', 'type': 'string'},
        {'name': 'host', 'type': 'string'},
        {'name': 'chunk_num', 'type': 'int64'},
        {'name': 'cluster', 'type': 'string'},
        {'name': 'file_host', 'type': 'string'},
        {'name': 'file_path', 'type': 'string'},
        {'name': 'file_name', 'type': 'string'},
        {'name': 'chunk_data', 'type': 'string'},
    ],
    'replication_factor': 1,
    'erasure_codec': 'lrc_12_2_2',
    'compression_codec': 'brotli_5',
    'optimize_for': 'scan',
}

FILELIST_COLUMNS = ('log_date', 'log_name', 'host', 'cluster', 'file_host', 'file_path', 'file_name', 'is_gzip', 'file_size',)

# игнорируем IOError с такими сообщениями - у нас много битых архивов, просто вычитываем из них, сколько сможем
IGNORE_IOERRS = [
    'CRC check failed',
    'Not a gzipped file',
    'Incorrect length of data produced',
]

SIGINT_H, SIGTERM_H = signal.getsignal(signal.SIGINT), signal.getsignal(signal.SIGTERM)
WRITER_THREAD = None
CONF = {
    'table_prefix': None,
    'journal_yt_data': False,
    'filelist': None,
    'stat_dir': None,
    # compresslevel должен соответствовать уровню сжатия исходных логов, тогда один чанк в yt - примерно 512mb исходных логов
    'gzip_level': 6,
    'yt_token_path': None,
    'yt_proxy': 'hahn',
}


def configure_yt():
    yt.config.set_proxy(CONF['yt_proxy'])
    yt.config['token_path'] = CONF['yt_token_path']
    # yt.config.CHUNK_SIZE = 1024*1024*1024
    # print yt.config.CHUNK_SIZE
    # yt.config['write_retries']['enable'] = True

    # увеличиваем дефолты в 5 раз
    yt.config['proxy']['request_timeout'] = 5 * 20 * 1000
    yt.config['proxy']['heavy_request_timeout'] = 5 * 60 * 1000
    yt.config['proxy']['proxy_ban_timeout'] = 5 * 120 * 1000

    yt.config['proxy']['accept_encoding'] = 'gzip, identity'
    yt.config['proxy']['content_encoding'] = 'gzip'
    # is_stream_compressed их вроде как отключает, но на всякий случай
    yt.config["write_retries"]["enable"] = False


def upload_chunk(table_path, data):
    # is_stream_compressed выключает ретраи yt
    # а еще они память жрут, судя по документации
    # поэтому ретраим сами
    tries = 3
    for t in range(tries):
        try:
            # это stream, перематываем в начало каждый раз
            data.seek(0)
            logging.info('yt: table %s: starting write attempt number %d' % (table_path, t))
            yt.write_table(
                yt.TablePath(table_path, append=True),
                data,
                table_writer={"max_row_weight": MAX_ROW_WEIGHT, 'content_encoding': 'gzip'},  # "desired_chunk_size": 1024*1024*1024},
                format=yt.YsonFormat(require_yson_bindings=True),
                is_stream_compressed=True,
                force_create=False,
                raw=True,
            )
            break
        except (yt.errors.YtError, ytreq.exceptions.RequestException) as e:
            if t >= tries - 1:
                raise
            logging.info('yt: write exception %s %s, try again' % (type(e), e))


def create_yt_table(table_path):
    logging.info('yt: table %s: check table existance' % (table_path,))
    if not yt.exists(table_path):
        yt.create(type='map_node', path='/'.join(table_path.split('/')[:-1]), recursive=True, ignore_existing=True)
        yt.create(type='table', path=table_path, attributes=TABLE_ATTR, ignore_existing=True)
        logging.info('yt: table %s: created' % (table_path,))
    return table_path


def get_yt_table_path(log_date):
    y, m, d = log_date.split('-')
    table_path = '%s/%s-%s' % (CONF['table_prefix'], y, m)
    return table_path


def disable_signals():
    signal.signal(signal.SIGINT, signal.SIG_IGN)
    signal.signal(signal.SIGTERM, signal.SIG_IGN)


def override_signals():
    signal.signal(signal.SIGINT, exit_handler)
    signal.signal(signal.SIGTERM, exit_handler)


def exit_handler(signum, frame):
    logging.info('run: got signal {}, exiting'.format(signum))
    sys.exit(signum)


def read_chunk(stream, extra_data, size):
    """
    Читаем из потока кусок размером size, находим последний newline, возвращаем
    data - кусок с newline на конце <= size, или без, но размером size
    extra_data - оставшийся буфер после newline
    err - одно из игнорируемых исключений при работе с архивами
    """

    data = ''
    error_context = 'filepos byte before error ' + str(stream.tell())
    try:
        data = stream.read(size - len(extra_data))
    except (IOError, zlib.error) as e:
        ignore_by_msg = any([str(e).startswith(x) for x in IGNORE_IOERRS])
        if isinstance(e, zlib.error) or ignore_by_msg:
            logging.error('file line: %s %s - cannot do anything with that, return current buffer; %s' %
                          (type(e), e, error_context))
            return data, extra_data, e

    data = extra_data + data
    if data == '':
        return data, extra_data, None

    sep = len(data)

    # пытаемся найти последний newline и разбить прочитанный кусок по нему
    last_nl = data.rfind('\n')
    nl_seq_start = last_nl
    if last_nl > 0:
        while nl_seq_start > 0 and data[nl_seq_start - 1] == '\n':
            nl_seq_start -= 1
        if nl_seq_start > 0:
            sep = last_nl + 1

    extra_data = data[sep:]
    data = data[:sep]

    return data, extra_data, None


def read_chunk_timed(stream, extra_data, size):
    b = time.time()
    res = read_chunk(stream, extra_data, size)
    e = time.time()
    return res + (e - b,)


def prepare_yt_data(out_stream, chunk, chunk_num, log_metadata):
    row = {}
    for column in [x['name'] for x in TABLE_ATTR['schema'] if not x['name'].startswith('chunk')]:
        row[column] = log_metadata[column]

    row['chunk_num'] = chunk_num
    row['chunk_data'] = chunk

    b = time.time()
    s = yt_yson_bindings.dumps(row)
    e = time.time()
    stime = e - b

    b = time.time()
    # yt.write хочет набор разделенных ';' yson map'ов
    out_stream.write(s + ';')
    # обязательно, иначе stringio/byteio не увидит записанных данных сразу, а от его размера зависит, когда писать пачку в yt
    out_stream.flush()
    e = time.time()
    ctime = e - b

    logging.info('file %s/%s: serialization: %.6f s, %d byte/s; compression %.6f s, %d byte/s' %
            (log_metadata['file_path'], log_metadata['file_name'], stime, len(s) / stime, ctime, len(s) / ctime))
    return stime, ctime


def start_writer_thread():
    q = Queue.Queue(maxsize=1)
    t = threading.Thread(target=yt_writer, name='YtWriter', args=(q,))
    logging.info('yt: writer thread created')
    t.start()
    return {'thread': t, 'queue': q}


def yt_writer(queue):
    logging.info('yt: writer thread started')
    configure_yt()
    logging.info('yt: module configured')
    while True:
        kwargs = queue.get()
        try:
            logging.info('yt: got new task: ' + str(kwargs.keys()))
            if kwargs.get('stop'):
                break
            upload_buf_to_yt(**kwargs)
        except BaseException as e:
            # иначе трейс попадает не в лог, а в stderr, сложно соотносить время
            logging.critical('yt: exception %s %s' % (type(e), e))
            logging.critical(traceback.format_exc())
            raise
        finally:
            queue.task_done()


def check_writer_or_die():
    if not WRITER_THREAD['thread'].is_alive():
        logging.critical('run: writer thread is dead, exiting')
        # нет смысла продолжать, исключение обработается в main()
        sys.exit(1)


def upload_buf(write_context, partial=False):
    logging.info('file: check writer thread, wait for running upload and put next chunk; stream pos %d, gzip_stream pos %d' %
                 (write_context['stream'].tell(), write_context['gzip_stream'].tell()))
    check_writer_or_die()
    # ждем, пока текущая загрузка (но не весь тред) завершится
    WRITER_THREAD['queue'].join()
    check_writer_or_die()

    wc_to_upload = prepare_write_context_for_upload(write_context)
    logging.info('file: write context splitted for upload, new context: %s' % (print_write_context(write_context)))
    WRITER_THREAD['queue'].put({'write_context': wc_to_upload, 'partial': partial})


def upload_buf_to_yt(write_context, partial=False):
    logging.info('yt: write context: ' + print_write_context(write_context))
    journal_st2 = write_context['journal_file'] + '.stage2'
    if os.path.isfile(journal_st2):
        raise Exception('stage2 journal file found - fix posfiles manually: ' + journal_st2)

    # загрузить 0 байт не страшно, если все позиции правильные (сам файл при этом может быть ненулевого размера, просто прочитать не смогли ничего)
    assert is_write_context_ready(write_context)
    # иногда падает после записи в yt, но до окончательной записи позиций
    write_atomic(write_context['journal_file'], print_write_context(write_context, with_stat=True))

    rtime, stime, ctime = [write_context['timings'][x] for x in ('read', 'serialize', 'compress')]
    prepare_time = rtime + stime + ctime
    stream_size = write_context['stream'].tell()
    logging.info('yt: table %s: prepared %d bytes in %d seconds (%d read, %d serialize, %d compress), preparation speed %d bytes/s; starting upload to yt' %
            (write_context['table_path'], stream_size, prepare_time, rtime, stime, ctime, stream_size / prepare_time))

    write_context['gzip_stream'].flush()
    # обязательно надо делать close, иначе будет ошибка чтения
    write_context['gzip_stream'].close()
    create_yt_table(write_context['table_path'])

    if CONF['journal_yt_data']:
        write_atomic(write_context['journal_file'] + '.yt_data', write_context['stream'], stream=True)

    b = time.time()
    upload_chunk(write_context['table_path'], write_context['stream'])
    os.rename(write_context['journal_file'], journal_st2)
    e = time.time()
    logging.info('yt: table %s: uploaded, now going to write all pos and stat files' % (write_context['table_path']))

    if CONF['journal_yt_data']:
        os.remove(write_context['journal_file'] + '.yt_data')

    log_pos_file = write_context['log_pos_file']
    log_pos = '%d\t%d\t%s' % (write_context['log_pos']['byte'], write_context['log_pos']['chunk'], write_context['log_pos']['path'])
    if partial:
        # позиция текущего загружаемого файла
        write_atomic(log_pos_file, log_pos)
    elif os.path.isfile(log_pos_file):
        # файл полностью загружен, удаляем posfile
        os.remove(log_pos_file)

    # сохраняем статистику по полностью загруженным файлам
    with open(write_context['stat_file'], 'a') as f:
        for stat in write_context['stat']:
            f.write(stat)
        f.flush()

    filelist_pos_file = write_context['filelist_pos_file']
    # сохраняем позицию в файл-листе
    old_filelist_pos = int(read_pos(filelist_pos_file, '0'))
    logging.info('yt: table %s: logfile pos %s; filelist pos %d, old filelist pos %d, fully uploaded files %d' %
            (write_context['table_path'], log_pos, write_context['filelist_line'], old_filelist_pos, len(write_context['stat'])))
    assert write_context['filelist_line'] == old_filelist_pos + len(write_context['stat'])
    write_atomic(filelist_pos_file, str(write_context['filelist_line']))

    os.remove(journal_st2)
    logging.info('yt: table %s: uploaded to yt in %d seconds, speed %d bytes/s' % (write_context['table_path'], e - b, stream_size / (e - b)))


def process_chunks(log_stream, write_context, log_metadata):
    old_ytstream_size = write_context['stream'].tell()
    # сколько байт для данного файла ушло в yt.write
    yt_write_size = 0
    logging.debug('file %s: stream %s, gzip_stream %s; stream pos %d, gzip_stream pos %d' %
                  (log_stream.name, write_context['stream'], write_context['gzip_stream'], write_context['stream'].tell(), write_context['gzip_stream'].tell()))

    chunk, extra_data, ioerr, rtime = read_chunk_timed(log_stream, '', LOG_CHUNK_SIZE)

    while chunk:
        check_writer_or_die()

        write_context['timings']['read'] += rtime
        chunk_size = len(chunk)
        extra_size = len(extra_data)
        logging.info('file %s: read chunk with size %d, extra data size %d; read time %.6f s, %d bytes/s' %
                     (log_stream.name, chunk_size, extra_size, rtime, chunk_size / rtime))

        stime, ctime = prepare_yt_data(write_context['gzip_stream'], chunk, write_context['log_pos']['chunk'], log_metadata)
        write_context['timings']['serialize'] += stime
        write_context['timings']['compress'] += ctime

        write_context['log_pos']['byte'] += chunk_size
        write_context['log_pos']['chunk'] += 1

        logging.info('file %s: chunk_size %d, extra_size %d, log_pos byte %d, log_pos chunk %d; calculated pos byte %d, read stream pos %d; write buffer size %d' %
                (log_stream.name, chunk_size, extra_size, write_context['log_pos']['byte'], write_context['log_pos']['chunk'],
                write_context['log_pos']['byte'] + extra_size, log_stream.tell(), write_context['stream'].tell())
        )

        # при ошибке чтения архива позиция файла может не совпадать с количеством успешно считанных байт, но чтение можно продолжить
        assert ioerr or (write_context['log_pos']['byte'] + extra_size == log_stream.tell())

        if write_context['stream'].tell() > YT_WRITE_SIZE:
            yt_write_size += write_context['stream'].tell() - old_ytstream_size
            upload_buf(write_context, partial=True)

            old_ytstream_size = write_context['stream'].tell()
            logging.debug('file %s: stream %s, gzip_stream %s; stream pos %d, gzip_stream pos %d' %
                          (log_stream.name, write_context['stream'], write_context['gzip_stream'], write_context['stream'].tell(), write_context['gzip_stream'].tell()))
        # touch stat_file - для мониторинга живости заливки
        os.utime(write_context['stat_file'], None)

        chunk, extra_data, ioerr, rtime = read_chunk_timed(log_stream, extra_data, LOG_CHUNK_SIZE)

    yt_write_size += write_context['stream'].tell() - old_ytstream_size

    stat = []
    for key in ('log_date', 'log_name', 'host', 'file_path', 'file_name'):
        stat.append(key + '=' + log_metadata[key])
    stat.append('bytes=' + str(write_context['log_pos']['byte']))
    stat.append('chunks=' + str(write_context['log_pos']['chunk']))
    stat.append('size_on_disk=' + str(log_metadata['file_size']))
    stat.append('yt_write_size=' + str(yt_write_size))

    write_context['stat'].append('\t'.join(stat) + '\n')


def upload_log(write_context, log_metadata):
    logging.info('file: log metadata: %s' % (json.dumps(log_metadata, sort_keys=True),))
    logging.info('file: write context: %s' % (print_write_context(write_context),))
    check_writer_or_die()

    path = log_metadata['file_path'] + '/' + log_metadata['file_name']
    start_log_byte = 0
    start_log_chunk = 0

    if write_context['log_pos']['path'] is None:
        logging.info('file %s: first run, read posfile %s' % (path, write_context['log_pos_file']))
        s, c, saved_path = read_pos(write_context['log_pos_file'], '0\t0\t' + path).split()
        start_log_byte = int(s)
        start_log_chunk = int(c)
        assert path == saved_path

    table_path = get_yt_table_path(log_metadata['log_date'])
    if write_context['table_path'] is not None and table_path != write_context['table_path']:
        # буфер держим только для одной таблицы, поменялась таблица - загружаем все старые куски
        # не эффективно по yt-чанкам, но хранить куски логов и надеяться, что когда-нибудь сможем накопить и загрузить кучей - сложно и не факт, что получится
        logging.info('file %s: new table path "%s", upload old buffer for "%s" first' % (path, table_path, write_context['table_path']))
        upload_buf(write_context)

    write_context['table_path'] = table_path
    write_context['log_pos']['path'] = path
    # количество считанных из лога байт (только чанки целиком, без extra)
    write_context['log_pos']['byte'] = start_log_byte
    # количество считанных кусков из лога (каждый кусок не больше максимальной yt-строки)
    write_context['log_pos']['chunk'] = start_log_chunk

    logging.info('file %s: prepare to read from %d byte, %d chunk' % (path, start_log_byte, start_log_chunk))
    log_stream = gzip.open(path, 'rb') if log_metadata['is_gzip'] else open(path, 'rb')
    log_stream.seek(start_log_byte)

    process_chunks(log_stream, write_context, log_metadata)

    log_stream.close()
    logging.info('file %s: fully read' % (path,))


def init_write_context():
    write_context = {
        # текущая таблица в yt
        'table_path': None,
        # поток для yt.write
        'stream': None,
        # над ним сжатый gzip поток для записи в буфер перед отправкой в yt.write
        'gzip_stream': None,
        # статистика по времени подготовки данных, только инкрементируется, поэтому не None
        'timings': {'read': 0, 'serialize': 0, 'compress': 0},
        # статистика по загруженным файлам
        'stat': [],
        'stat_file': None,
        # позиции лога (заполняются в upload_log каждый раз)
        'log_pos': {'path': None, 'byte': None, 'chunk': None},
        'log_pos_file': None,
        # позиция лога в файллисте
        'filelist_line': None,
        'filelist_pos_file': None,
        'journal_file': None,
    }
    init_buf_streams(write_context)
    return write_context


def prepare_write_context_for_upload(write_context):
    wc_copy = write_context.copy()
    clear_write_context(write_context)
    return wc_copy


def clear_write_context(write_context):
    # меняем все существующие ссылки, но не чистим сами объекты - они могут использоваться в другом потоке
    write_context['stat'] = []
    write_context['timings'] = {'read': 0, 'serialize': 0, 'compress': 0}
    write_context['log_pos'] = write_context['log_pos'].copy()
    init_buf_streams(write_context)


def is_write_context_ready(write_context):
    return all([x is not None for x in write_context.values()]) and \
           all([x is not None for x in write_context['log_pos'].values()])


def init_buf_streams(write_context):
    # старые закрывать не надо, они могут загружаться в другом треде
    write_context['stream'] = StringIO.StringIO()
    write_context['gzip_stream'] = gzip.GzipFile(fileobj=write_context['stream'], compresslevel=CONF['gzip_level'], mode='wb')


def print_write_context(write_context, with_stat=False):
    d = {k: v for k, v in write_context.iteritems() if not k.endswith('stream') and not k.endswith('stat')}
    d['stat'] = '<{} lines>'.format(len(write_context['stat']))
    if with_stat:
        d['stat'] = write_context['stat']
    d['stream'] = str(write_context['stream'])
    d['gzip_stream'] = str(write_context['gzip_stream'])
    return json.dumps(d, sort_keys=True)


def get_metadata(line):
    logging.info('filelist line: fetching metadata from string: %s' % (line,))

    row = line.split('\t')
    metadata = dict(zip(FILELIST_COLUMNS, row))
    metadata['is_gzip'] = (metadata['is_gzip'] == '1')
    metadata['file_size'] = int(metadata['file_size'])

    return metadata


def process_filelist(filelist_stream, stat_dir):
    filelist = filelist_stream.name

    logging.info('filelist: start ' + filelist)
    check_writer_or_die()

    filelist_pos_file = '%s/%s.pos.filelist' % (stat_dir, os.path.basename(filelist))
    stat_file = '%s/%s.stat' % (stat_dir, os.path.basename(filelist))
    finished_file = '%s/%s.finished' % (stat_dir, os.path.basename(filelist))

    # в принципе, pos-файл не позволит загрузить повторно, но так удобнее мониторинг сделать
    if os.path.isfile(finished_file):
        logging.info('filelist: ' + filelist + ' already uploaded, exiting')
        return

    # время модификации stat-файла используется для мониторинга
    # создаем его сразу, если его нет, но не перезаписываем!
    open(stat_file, 'a').close()

    write_context = init_write_context()
    write_context['filelist_pos_file'] = filelist_pos_file
    write_context['stat_file'] = stat_file
    write_context['log_pos_file'] = '%s/%s.pos.logfile' % (stat_dir, os.path.basename(filelist))
    write_context['journal_file'] = '%s/%s.journal' % (stat_dir, os.path.basename(filelist))

    start_line = int(read_pos(filelist_pos_file, '0'))
    logging.info('filelist: prepare to process %s from %d line' % (filelist, start_line))

    line_num = 0
    for line in filelist_stream:
        # файллист небольшой, проще пропустить нужное количество строк, tell/seek работают неправильно в текстовом режиме с for line/readline из-за readahead
        if line_num < start_line:
            line_num += 1
            continue

        logging.info('filelist %s: prepare to process line %d' % (filelist, line_num))
        write_context['filelist_line'] = line_num
        upload_log(write_context, get_metadata(line))

        line_num += 1
        logging.info('filelist %s: line processed' % (filelist,))

    # догружаем остатки (даже если на самом деле там 0 байт - статистику дописать надо)
    if is_write_context_ready(write_context):
        logging.info('filelist %s: upload last buffer')
        write_context['filelist_line'] = line_num
        upload_buf(write_context)
        # чтобы не создать finished файл до того, как данные будут реально записаны
        WRITER_THREAD['queue'].join()

    write_atomic(finished_file, '')
    logging.info('filelist %s: uploaded' % (filelist,))


def write_atomic(filepath, data, stream=False):
    tmppath = filepath + '~'
    while os.path.isfile(tmppath):
        tmppath += '~'
    try:
        with open(tmppath, 'wb') as f:
            if stream:
                data.seek(0)
                shutil.copyfileobj(data, f)
            else:
                f.write(data)
            f.flush()
            os.fsync(f.fileno())
        os.rename(tmppath, filepath)
    finally:
        try:
            os.remove(tmppath)
        except (IOError, OSError):
            pass


def read_pos(filepath, default=None):
    try:
        with open(filepath) as f:
            return f.read()
    except (IOError, OSError):
        if default is not None:
            return default
        raise


def global_monitor(stat_dir, max_proc):
    filelists = [x[:-5] for x in os.listdir(stat_dir) if x.endswith('.stat')]

    # stat_dir /.../logbackup-stat.prod
    env = os.path.basename(os.path.normpath(stat_dir)).split('.')[-1]
    event = {'service': 'logbackup-to-yt.' + env, 'status': 'OK'}
    active = 0
    stale = 0
    for logfile in sorted(set(filelists)):
        stat_file = '%s/%s.stat' % (stat_dir, logfile)
        finished_file = '%s/%s.finished' % (stat_dir, logfile)

        if not os.path.isfile(finished_file):
            delta = int(time.time() - os.stat(stat_file).st_mtime)
            if delta > MAX_CHUNK_UPLOAD_TIME:
                stale += 1
                logging.info('monitor: %s not finished and not updated for %d sec' % (logfile, delta))
            else:
                active += 1
                logging.info('monitor: %s updated recently: %d sec' % (logfile, delta))
        elif os.path.isfile(finished_file):
            logging.info('monitor: %s finished %d seconds ago' % (logfile, int(time.time() - os.stat(finished_file).st_mtime)))

    if not (active >= max_proc or stale == 0):
        event['status'] = 'CRIT'
        logging.info('monitor: something failed, should be %d active or 0 stale. Now %d active, %d stale' % (max_proc, active, stale))

    logging.info(str(event))
    queue_events([event])


def main(argv):
    CONF['table_prefix'] = argv.table_prefix
    CONF['yt_token_path'] = argv.yt_token_path
    CONF['filelist'] = argv.filelist
    CONF['stat_dir'] = argv.stat_dir
    CONF['journal_yt_data'] = argv.journal_yt_data
    CONF['gzip_level'] = argv.gzip_level
    for k, v in CONF.iteritems():
        if v is None:
            logging.error('run: argument --{} is required in upload mode, exiting'.format(k))
            sys.exit(1)

    logging.info('run: started with config: ' + json.dumps(CONF, sort_keys=True))

    with open(CONF['filelist'], 'r') as filelist_stream:
        logging.info('run: trying to lock ' + CONF['filelist'])
        try:
            fcntl.flock(filelist_stream, fcntl.LOCK_EX | fcntl.LOCK_NB)
        except IOError:
            logging.info('run: ' + CONF['filelist'] + ' - already running, exiting')
            sys.exit()
        logging.info('run: ' + CONF['filelist'] + ' locked')

        # переопределяем стандартные сигналы завершения, чтобы вызвать в них исключение и нормально завершить тред
        override_signals()
        global WRITER_THREAD
        WRITER_THREAD = start_writer_thread()

        try:
            process_filelist(filelist_stream, CONF['stat_dir'])
        except SystemExit as e:
            logging.info('run: termination requested %s %s' % (type(e), e))
        except BaseException as e:
            logging.critical('run: exception %s %s' % (type(e), e))
            logging.critical(traceback.format_exc())
        finally:
            # возможно, сюда мы уже попали по sigint/sigterm, попытаемся спокойно завершить тред
            disable_signals()
            if WRITER_THREAD['thread'].is_alive():
                logging.info('run: queue stop task')
                WRITER_THREAD['queue'].put({'stop': True})
                logging.info('run: join() writer thread')
                # ждем именно завершения треда, а не просто всех задач в очереди
                WRITER_THREAD['thread'].join()
            logging.info('run: unlock filelist and exit')
            fcntl.flock(filelist_stream, fcntl.LOCK_UN)
            logging.info('run: bye!')


if __name__ == "__main__":
    parser = argparse.ArgumentParser(formatter_class=argparse.RawDescriptionHelpFormatter, description=__doc__)
    parser.add_argument('--yt-token-path', help='Путь к yt-токену')
    parser.add_argument('--script-log-file', default=None, help='Писать логи скрипта в файл')
    parser.add_argument('--stat-dir', required=True, help='Где создавать файлы позиций и статистики по файллисту')
    parser.add_argument('--table-prefix', help='Префикс для логовых табличек')
    parser.add_argument('--filelist', help='Файл со списком логов и метаданных для загрузки')
    parser.add_argument('--global-monitor', type=int, default=0, help='Режим мониторинга состояния stat-dir. Параметр - количество одновременно загружаемых файллистов. Пишет одно событие logbackup-to-yt в juggler')
    parser.add_argument('--journal-yt-data', action='store_true', help='Сохранять yt_chunk в файл перед yt.write (stat_dir/filelist.journal.yt_data). Полезно для отладки')
    parser.add_argument('--gzip-level', type=int, default=6, help='Сжимать поток в gzip перед записью в yt, чем сильнее сжатие, тем меньше чанков расходуется')
    args = parser.parse_args()

    logfmt = '[%(asctime)s]\t%(levelname)-8s\t' + str(os.getpid()) + '\t%(threadName)-15s\t%(message)s'
    if args.script_log_file:
        logging.basicConfig(filename=args.script_log_file, level=logging.DEBUG, format=logfmt)
    else:
        logging.basicConfig(stream=sys.stderr, level=logging.DEBUG, format=logfmt)

    if args.global_monitor > 0:
        global_monitor(args.stat_dir, args.global_monitor)
        sys.exit()

    main(args)
