#!/usr/bin/pypy
# -*- coding: utf-8 -*-

import sys
sys.path.append('/usr/lib/pymodules/python2.7')
sys.path.append('/usr/lib/python2.7/dist-packages')

import MTRSLogpusher
import threading
import datetime
import urllib2
import logging
import socket
import random
import requests
import time
import sys
import re
import os
from subprocess import Popen, PIPE
import traceback
import json
import random
import fcntl
from dateutil import tz
from pytz import timezone

DCFILE = '/etc/ppcinv/localhost.root_dc'

def get_metadata_filename(logfile, logType):
    logging.debug("Use logType: %s" % logType)
    return MTRSLogpusher.get_metadata_filename(logfile, logType, config.main.cache_dir)


def writeLogPos(logFile, logType, inode, offset, first_run=0):
    dataFile = get_metadata_filename(logFile, logType)
    with open(dataFile, 'w') as posdata:
        if MTRSLogpusher.set_lock_file(posdata):
            posdata.write("%s:%s:%s" % (str(inode), str(offset), str(first_run)))
            fcntl.flock(posdata, fcntl.LOCK_UN)

def getMyDC(fqdn):
    myDc = '0'
    try:
        if os.path.exists(DCFILE):
            fileDc = open(DCFILE).read().strip()
            myDc = fileDc if len(fileDc) > 0 else myDc
        else:
            botUrl = config.main.bot_url_template % (fqdn,)
            botRequest = urllib2.Request(botUrl)
            botResponse = urllib2.urlopen(botRequest).read().strip()
            myDc = botResponse.split('-')[0].strip()
    except Exception, e:
        logging.error("Can't get data from bot")
        raise e
    if myDc == '0':
        url = 'http://c.yandex-team.ru/api-cached/hosts/{fqdn}/?format={format}'.format(format='json', fqdn=fqdn)
        try:
            response = urllib2.urlopen(url, timeout = 5).read()
        except Exception, e:
            loggging.debug('Failed fetch data from {0}'.format(url))
            raise e
        if response:
            try:
                info = json.loads(response)[0]
            except Exception, e:
                logging.debug('Failed to parse result from {0}'.format(url))
                raise e
            if info is None:
                logging.warning('Host is not configured in conductor.')
            else:
                myDc = info['root_datacenter']
    return myDc.strip()


def findFileByInode(fileName, inode):
    # Ищем файл с запомненным inode в той же директории
    baseDir = os.path.dirname(fileName)
    out, err = Popen('find %s -inum %s' % (baseDir, inode), shell=True, stdout=PIPE).communicate()
    return out.strip()


def escape_column(line):
    line = line.replace('\t', '\\t')
    line = line.replace('\\', '\\\\')
    return line


def saveToCh(servers, table, columns, data, config):
    # Формируем блок для вставки
    # Строки разделены \n
    # Столбцы разделены \t
    # В каждом столбце экранируем \t
    start_time = time.time()
    trying = 0
    servers = servers[:]
    while time.time() - start_time < config.ch_send_timeout:
        errors = []
        server = servers[random.randrange(0, len(servers))]
        user = 'user=' + config.ch_user
        auth = user + '&password=' + config.ch_passwd if config.ch_passwd else user
        chUrlPrefix = 'http://%s/?%s&query=INSERT' % (server, auth) + '%20INTO'
        logging.debug(chUrlPrefix)
        for keys in data:
            url = '%20'.join((chUrlPrefix, table, '(' + '%2C'.join(keys) + ')', 'FORMAT', 'TabSeparated'))
            toSend = "\n".join(["\t".join(map(escape_column, line)) for line in data[keys]])
            try:
                req = requests.post(url, data=toSend, timeout=(config.connect_timeout, config.request_timeout))
                req.raise_for_status()
            except requests.HTTPError as e:
                try:
                    err = e.read()
                except:
                    err = str(e)
                msg = "%s on server %s" % (err, server)
                logging.critical(msg)
                errors.append(msg)
                continue
            except Exception as e:
                msg = "%s on server %s" % (str(e), server)
                logging.critical(msg)
                errors.append(msg)
                continue

        end_time = time.time()
        save_duration = end_time - start_time

        if trying > 3: break
        if len(errors) > 0:
            trying += 1
            logging.error("Send data to {0} failed! Number trying {1}. Will sleep for a while. Errors: {2}".format(server, trying, ', '.join(errors)))
            servers.remove(server)
            time.sleep(random.random()*10)
            continue
        logging.debug("Saved to CH in %f seconds. Server: %s." % (save_duration, server))
        return True
    logging.critical("Timeout exceeded. All servers can't store data for %f seconds." % ((time.time() - start_time),))
    return False


def seek_last_line(loglines, from_pos=os.SEEK_CUR):
    loglines.seek(max(0 - loglines.tell(), -1), from_pos)
    while loglines.tell() > 0 and loglines.read(1) != b"\n":
        loglines.seek(-2, os.SEEK_CUR)

    position = loglines.tell()
    logging.debug("Last line of found on position %d" % (position,))
    return position


def safe_unlink(filename):
    try:
        os.unlink(filename)
    except Exception as e:
        logging.debug("Can't remove %s: %s", (filename, str(e)))


def get_latest_log_and_flush_position(current_logfile, logfile, logType):
    logging.debug("Can't continue with rotated log, flush all metadata (%s, %s) and start from latest log" % (get_metadata_filename(current_logfile, logType), get_metadata_filename(logfile, logType)))
    safe_unlink(get_metadata_filename(current_logfile, logType))
    writeLogPos(logfile, logType, 0, 0, 1)
    return logfile

def find_item(obj, key):
    if key in obj: 
        return str(obj[key])
    for k, v in obj.items():
        if isinstance(v,dict):
            return find_item(v, key)
    return ''

def parse_time(time_string, config):
    base_time = datetime.datetime.strptime(time_string, config.date_format)
    regxp = config.regexp_convert_time
    try:
        tzname = regxp.search(time_string)
        if tzname:
            base_time = base_time.replace(tzinfo=timezone(tzname.group('zone')))
            base_time = base_time.astimezone(tz.tzlocal())
    except Exception as err:
        logging.error("Error parse timezone {0} tzname: {1}".format(time_string, str(err)))
    return base_time.strftime("%s")

def get_date(timestamp):
    date = datetime.datetime.fromtimestamp(int(timestamp))
    date = date.date()
    return date.strftime('%Y-%m-%d')

def parse_json_line(line, data, stats, config):
    stats[2] += 1
    try:
        jdata = json.loads(line)
    except Exception as err:
        logging.error("Error load json: {0}".format(line))
        stats[0] +=1
        return

    values = {}
    for c in config.columns:
        if c in ['hostname', 'fqdn']:
            values['fqdn'] = config.fqdn
            continue
        values[c] = find_item(jdata, c)
    try:
        if config.date_column:
            date = find_item(jdata, config.date_column)
            values['date'] = parse_time(date, config)
            values['day'] = get_date(values['date'])
    except Exception as err:
        logging.error("Error convert {0} {2} in {1}: {3}".format(config.date_column, date, config.date_format, str(err)))
    for i in config.constant_columns:
        values[i] = config.constant_columns[i]
    keys = tuple(values.keys())
    if not data.has_key(keys): data[keys] = []
    data[keys].append(values.values())

def parse_log_line(line, data, stats, config):
    stats[2] += 1
    m = config.regexp_compiled.search(line)
    if m is None:
        logging.error("Malformed logline: " + line)
        stats[0] += 1
        return

    if 'request' in m.groupdict() and m.group('request') in config.skip_urls:
        stats[1] += 1
        return
    # Приводим время в милисекунды:
    # reqtime - в секундах

    if 'reqtime' in m.groupdict():
        reqtime_raw = m.group('reqtime')
        multiplier = 1000
    # reqtime_ms - в милисекундах
    elif 'reqtime_ms' in m.groupdict():
        reqtime_raw = m.group('reqtime_ms')
        multiplier = 1
    # reqtime_mcs - в микросекундах
    elif 'reqtime_mcs' in m.groupdict():
        reqtime_raw = m.group('reqtime_mcs')
        multiplier = 0.001
    else:
        reqtime_raw = None

    if reqtime_raw:
        # nginx пишет '-' в upstream_response_time, если обрабатывает запрос сам, без проксирования
        if reqtime_raw == '-':
            static = 1
            reqtime = 0
        else:
            static = 0
            # суммируем все говно, что пишется в reqtime_raw,
            # не забывая, что reduce ничего не делает, если в списке только один элемент
            reqtime = reduce(lambda x, y: float(x) + float(y), numbers.findall(reqtime_raw), 0) * multiplier

    if config.store_full_request_time:
        if 'full_time_mcs' in m.groupdict():
            full_time_mcs =  m.group('full_time_mcs')
            multiplier = 0.001
            full_time_mcs = float(full_time_mcs) * multiplier
        else:
            full_time_mcs = 0
        full_request_time = str(full_time_mcs)

    # Парсим дату
    # группа date - d/m/y:h:m:s
    # используется формат, указанный в конфиге (date_format)
    # группа time - h:m:s
    # группы HH,MM,SS - h,m,s соответственно
    if 'date' in m.groupdict():
        d = parse_time(m.group('date'), config)
    elif 'custom_date' in m.groupdict():
        d = parse_time(m.group('custom_date'), config)
    elif 'time' in m.groupdict():
        (hh, mm, ss) = m.group('time').split(':')
        today = datetime.date.today()
        d = datetime.datetime(today.year, today.month, today.day, int(hh), int(mm), int(ss))
    elif 'HH' in m.groupdict() and 'MM' in m.groupdict() and 'SS' in m.groupdict():
        today = datetime.date.today()
        d = datetime.datetime(today.year, today.month, today.day, m.group('HH'), m.group('MM'), m.group('SS'))
    else:
        raise Exception("Can't parse date!")

    values = {}
    for c in config.columns:
        if c in ['hostname', 'fqdn']: 
            values[c] = config.fqdn
        elif c == 'dc': 
            values[c] = myDc
        elif c == 'day': 
            values[c] = get_date(d)
        elif c == 'date': 
            values[c] = d
        elif c == 'src_ip': 
            values[c] = m.group('ip')
        elif c == 'vhost': 
            values[c] = config.vhost if config.vhost else m.group('vhost')
        elif c == 'url': 
            values[c] = m.group('request') if config.store_urls else ''
        elif c == 'code': 
            values[c] = str(int(m.group('code')))
        elif c == 'static': 
            values[c] = str(static)
        elif c == 'time': 
            values[c] =str(reqtime)
        elif c in m.groupdict():
            values[c] = str(m.group(c)) 
        else:
            logging.error("dont find {0} in regexp {1}".format(c, config.regexp))
            stats[0] += 1

    if config.store_full_request_time:
        values['full_request_time'] = full_request_time
    for i in config.constant_columns:
        values[i] = config.constant_columns[i]

    keys = tuple(values.keys())
    if not data.has_key(keys): data[keys] = []
    data[keys].append(values.values())

def read_and_parse_file(loglines, is_rotated_log, config):
    (data, stats, start_time) = ({}, [0, 0, 0], time.time())
    line = ''
    total = 0 
    # read raw bytes to buffer first, if too slow, but not "for line in loglines:" - bad file position after break!
    while True:
        line = loglines.readline()
        if len(line) == 0 or line[-1] != b"\n":
            logging.debug("Line is empty or no newline symbol, EOF? Incomplete line and EOF? Non-unix format? Stop reading")
            break
        if config.regexp:
            parse_log_line(line, data, stats, config)
        else:
            parse_json_line(line, data, stats, config)
        total = stats[2]

        if total > config.lines_bundle_min_size and (time.time() - start_time) > config.lines_bundle_timeout:
            logging.debug("Stop reading, limits exceeded")
            break
    parsing_duration = time.time() - start_time
    logging.debug("Reading stopped on %d, parsed %d raw lines in %f sec" % (loglines.tell(), total, parsing_duration))

    if len(line) > 0 and line[-1] != b"\n" and not is_rotated_log:
        seek_last_line(loglines, os.SEEK_CUR)
    return (data, stats)


def process_log(config, logType):
    logType = logType
    logfile = config.file
    current_logfile = logfile
    while True:
        start_time = time.time()
        (savedInode, position, first_run) = MTRSLogpusher.getLogPos(current_logfile, logType, config.cache_dir)
        is_rotated_log = current_logfile != logfile

        # Проверяем, тот ли это файл, из которого мы читали в прошлый раз
        try:
            loglines = open(current_logfile, 'r')
        except Exception as e:
            logging.critical("Can't open file %s: %s" % (current_logfile, str(e)))
            if is_rotated_log:
                current_logfile = get_latest_log_and_flush_position(current_logfile, logfile, logType)
            else: 
                time.sleep(60)
            continue

        fileStat = os.fstat(loglines.fileno())
        inode = fileStat.st_ino
        fileSize = fileStat.st_size

        if (inode != savedInode) and not first_run: 
            if (current_logfile == logfile):
                logging.debug("Inode changed! (current %s, but saved %s)" % (inode, savedInode))
            else:
                logging.debug("Inode changed for rotated log (current %s, but saved %s)" % (inode, savedInode))
                current_logfile = get_latest_log_and_flush_position(current_logfile, logfile, logType)
                loglines.close()
                continue
 
            # Пытаемся дочитать лог из отротированного файла
            rotatedLogfile = findFileByInode(logfile, savedInode)
            if rotatedLogfile != "":
                logging.debug("File with saved inode found: %s, start new iteration with rotated log" % rotatedLogfile)
                current_logfile = rotatedLogfile
                writeLogPos(current_logfile, logType, 0, position, 1)
                loglines.close()
                continue
            else:
                logging.debug("File with saved inode not found.")
                position = 0

        if position > fileSize:
            if (current_logfile == logfile):
                logging.debug("Position %d greater than file size %d. File truncated? Starting from position=0", (position, fileSize))
                position = 0
            else:
                logging.debug("Position %d greater than file size %d for rotated log, archived?", (position, fileSize))
                current_logfile = get_latest_log_and_flush_position(current_logfile, logfile, logType)
                loglines.close()
                continue

        if first_run and config.skip_dangerous_file_first_time and (fileSize - position) > config.dangerous_file_size:
            logging.info("Seems like log is too big for full parsing. Search for last line ...")
            position = seek_last_line(loglines, os.SEEK_END)

        loglines.seek(position)
        logging.debug("Start reading file %s inode %d size %d from position %d" % (current_logfile, inode, fileSize, position))

        (data, (bad_lines, skipped, total)) = read_and_parse_file(loglines, is_rotated_log, config)
        offset = loglines.tell()
        loglines.close()

        # Save data to clickhouse
        saved = False

        if len(data) > 0:
            logging.info("Saving %s lines to clickhouse (%s skip_urls, %s malformed lines)" % (len(data), skipped, bad_lines))
            saved = saveToCh(config.clickhouse_host, config.table, config.columns, data, config)
        else:
            logging.info("Nothing to send: %s skip_urls, %s malformed lines" % (skipped, bad_lines))
            saved = True

        if saved:
            if is_rotated_log and ((offset >= fileSize) or (len(data) == 0)):
                current_logfile = get_latest_log_and_flush_position(current_logfile, logfile, logType)
            else:
                logging.debug("Saving postion.")
                writeLogPos(current_logfile, logType, inode, offset, 0)

        end_time = time.time()
        iter_duration = end_time - start_time
        logging.info("Iteration done in %f seconds." % iter_duration)
        if iter_duration < config.lines_bundle_timeout:
            time.sleep(int(config.lines_bundle_timeout - iter_duration))


def monitor_thread(threads, interval=60, wait=10):
    """
    Отдельный тред для проверки живости основных тредов для разборки логов. Чекает раз в interval секунд.
    Если тред не в состоянии alive, попробуем его перезапустить.
    Если через wait секунд тред все равно упал, выходим из приложения с кодом 1.
    """
    while True:
        try:
            for i, thread in enumerate(threads):
                if not thread.isAlive():
                    log_type = thread.name
                    logging.warning("Thread %s is not alive! Try to respawn in..." % log_type)
                    new_thread = threading.Thread(name=log_type, target=process_log, args=(config[log_type], log_type))
                    new_thread.start()
                    time.sleep(wait)
                    if new_thread.isAlive():
                        threads.pop(i)
                        threads.append(new_thread)
                        logging.info("Thread %s has been respawned" % log_type)
                    else:
                        logging.error("New thread can't start! Exiting program...")
                        for old_thread in threads:
                            old_thread.join(1)
                        sys.exit(1)

            time.sleep(interval)
        except:
            tb = sys.exc_info()[2]
            while 1:
                if not tb.tb_next:
                    break
                tb = tb.tb_next
            stack = []
            f = tb.tb_frame
            while f:
                stack.append(f)
                f = f.f_back
            stack.reverse()
            traceback.print_exc()
            logging.debug("Locals by frame, innermost last")
            for frame in stack:
                logging.debug("Frame %s in %s at line %s" % (frame.f_code.co_name,
                                                             frame.f_code.co_filename,
                                                             frame.f_lineno
                                                             )
                              )
                for key, value in frame.f_locals.items():
                    logging.debug("\t%20s = " % key)
                    try:
                        print value
                    except:
                        logging.fatal("<ERROR WHILE PRINTING VALUE>")


def main():
    global fqdn
    global myDc
    global config
    global numbers

    numbers = re.compile('[0-9.]+')
    formatter = logging.Formatter('%(asctime)s %(threadName)-20s %(message)s')
    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG)
    ch = logging.StreamHandler()
    ch.setLevel(logging.DEBUG)
    ch.setFormatter(formatter)
    logger.addHandler(ch)

    config = MTRSLogpusher.get_config()
    if not config: sys.exit(1)

    fqdn = socket.getfqdn()
    try:
        myDc = getMyDC(fqdn)
    except Exception as e:
        logging.exception("Can't get my DC: %s" % (str(e),))
        sys.exit(1)
    if not os.path.exists(config.main.cache_dir):
        os.makedirs(config.main.cache_dir)

    threads = []
    for logType in config.keys():
        if logType == 'main':
            continue
        logging.info("Creating thread for %s" % logType)
        senderThread = threading.Thread(name=logType, target=process_log, args=(config[logType], logType))
        threads.append(senderThread)

    for thread in threads:
        logging.debug("Starting thread")
        thread.start()

    monitorThread = threading.Thread(name="LogpusherMonitor", target=monitor_thread, args=(threads,))
    monitorThread.daemon = True
    logging.debug("Starting monitor thread")
    monitorThread.start()

    # for theread in threads:
    #   thread.join()

if __name__ == '__main__':
        main()
