#!/usr/bin/perl -w

# $Id$


=head1 NAME

    collect and write metatracker data to DB

=head1 DESCRIPTION


    Опции:
        -h, --help
            показать этот текст и завершиться
        --release <issue_key>
            (необязательно)
            собрать тикеты из указанного релизного тикета
            Если не указано -- будет обработан крайний релиз

=head1 DB

    Заполняются табицы в базе releaser:

    metatracker_issue
    metatracker_issue_commits
    metatracker_problem

=head1 TODO


=cut

#use lib '/home/lena-san/usr/bin/direct-utils/project_specific_pm/trunk/lib/';
#use lib '/home/lena-san/usr/bin/direct-utils/jira_client_pm/trunk';

use List::Util qw{ first };

use Startrek::Client::Easy;
use ProjectSpecific qw/releaser_dbh_config get_startrek_client_token/;
use Data::Dumper;
use Getopt::Long;
use List::MoreUtils qw(uniq);
use DBI;
use Pid::File::Flock;

use strict;

use utf8;
use open ':locale';

our $STARTREK_ROBOT_TOKEN_FILE = '/etc/direct-tokens/startrek';
#................................................
# константы из releaser/metatracker/models.py
our $ORIGIN_DESCRIPTION   =  1;
our $ORIGIN_HOTFIX        = 10;
our $ORIGIN_COMMENT       = 20;
our $ORIGIN_AFTER_RELEASE = 50;

our $RELEVANCE_UNKNOWN  = 0;
our $RELEVANCE_NO  = 10;
our $RELEVANCE_YES = 20;

our $SIGNIFICANCE_UNKNOWN  = 0;
our $SIGNIFICANCE_NO  = 10;
our $SIGNIFICANCE_YES = 20;


my %release_to_ignore = map { $_ => 1 } qw/
    DIRECT-12187
    DIRECT-24373
/;

my %OPT = parse_options();
$ProjectSpecific::PROJECT = $OPT{project} if $OPT{project};
my $queue = ProjectSpecific::get_project_data('default_tracker_queue');

$ProjectSpecific::STARTREK_INSTANCE = 'prod';

# для одного проекта должен работать не более чем один экземпляр скрипта одновременно
Pid::File::Flock->new(ext=>".$ProjectSpecific::PROJECT.pid");

my $startrek = Startrek::Client::Easy->new(startrek => $ProjectSpecific::STARTREK_INSTANCE, token => get_startrek_client_token(file => $STARTREK_ROBOT_TOKEN_FILE));

my $dbh_config = releaser_dbh_config();

my $dbh = DBI->connect( @$dbh_config{qw/data_source user pass/}, {RaiseError => 1, AutoCommit => 1});
$OPT{dbh} = $dbh;

my $release;
my $recent_releases;
if ( $OPT{release_key} ){
    $release = $startrek->get(key => $OPT{release_key} );
    $recent_releases = [ $release ];
} else {
    $recent_releases = $startrek->get(query => ProjectSpecific::make_last_releases_query(), limit => 10);
    @$recent_releases = grep { ! $release_to_ignore{$_->{key}} } @$recent_releases;
    $release = $recent_releases->[0];
}

for my $release ( @$recent_releases ){
    my $release_key = $release->{key};
    my ($release_name, $release_version) = $release->{summary} =~ /^(?:релиз.*?:)\s*(.*)\s*-+\s*выложить\s+([\d\.\-]+)\b/;

    my $issues = extract_issues(startrek => $startrek, release_key => $release_key);

    my %release_hash = (
        release_key => $release_key,
        release_version => $release_version,
        release_name => $release_name,
    );
    write_db_issues(issues => $issues, dbh => $dbh, %release_hash);
}

fetch_issues_commits(dbh => $dbh);

exit;


#................................................

# возвращает хеш %O с опциями из командной строки
sub parse_options
{
    my %O = (
    );

    GetOptions (
        "help"                => \&usage,
        "release=s"           => \$O{release_key},
        "p|project=s"         => \$O{project},
    );

    return %O;
}


# Печатает usage-сообщение
sub usage {
    system("podselect -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8 >&2");
    exit(1);
}


=head2

    результат: ссылка на массив хешей.
    Каждый хеш
    {
        key     => 'DIRECT-1234',
        context_type => 'comment', # или 'description', или 'hotfix', или 'created after release'
        context_details => '2010-06-01 11:23',
    }

=cut

sub extract_issues
{
    my %OPT = @_;

    my $startrek = $OPT{startrek};

    my $release = $startrek->get( key => $OPT{release_key} );
    #print "got release $OPT{release_key}\n";

    my $issues = extract_issues_from_text(text => $release->{description}, context_type => 'description');

    my $comments = $startrek->request(GET => "/issues/$OPT{release_key}/comments");

    for my $c (@$comments){
        # зависимости релизов из автокомментариев не записываем в "тикеты, относящиеся к релизу"
        next if $c->{text} =~ /^С этим релизом связана зависимость:\n/;
        my $comment_type = $c->{text} =~ /^(В релиз включен новый код|Добавлены хотфиксы)/ ? "hotfix" : "comment";
        my $context_details;
        # костыль для сохранения обратной совместимости: преобразуем время в тот формат, который выдаёт джира-клиент
        (my $created = $c->{createdAt}) =~ s/\+0000$/Z/;
        $context_details = $c->{createdBy}->{id} . ' ' . $created;
        my $issues_from_comment = extract_issues_from_text(text => $c->{text}, context_type => $comment_type, context_details => $context_details);
        push @$issues, @$issues_from_comment;
    }

    $issues = [ grep { $_->{key} ne $OPT{release_key} } @$issues ];
    return $issues;

}


=head2 key2number

   id тикета --> номер

=cut
sub key2number
{
    my $key = shift;

    my ($number) = $key =~ /-(\d{4,5})$/g;

    return $number;
}


=head2 extract_issues_from_text

    текст --> ссылка на массив id тикетов оттуда

    Тикеты ищутся из основгого проекта или из важных смежных проектов (TESTIRT, BALANCE и т.п.)
    Можно когда-нибудь обобщить как ProjectSpecific::additional_jira_projects()

=cut

sub extract_issues_from_text
{
    my (%O) = @_;

    my $QUEUE = ProjectSpecific::get_project_data('default_tracker_queue');

    my $text = $O{text} || '';
    my $res = [ map { {key => $_, context_type => $O{context_type} || '', context_details => $O{context_details} || ''} } uniq $text =~ /\b((?:$QUEUE|TESTIRT|BALANCE|DIRECTADMIN|TESTBALANCE)-\d{2,7})/g ];

    return $res;
}



=head2 write_db_issues

    запись информации о тикетах в БД

=cut

sub write_db_issues
{
    my %O = @_;

    my $dbh = $O{dbh};

    my %saved = ();
    for my $issue ( @{$O{issues}} ){
        my $issue_key = $issue->{key};
        $O{startrek} ||= Startrek::Client::Easy->new(startrek => $ProjectSpecific::STARTREK_INSTANCE, token => get_startrek_client_token(file => $STARTREK_ROBOT_TOKEN_FILE));
        #print "$issue_key...\n";
        next if $saved{$issue_key}++;
        #print "...need to be saved!\n";
        my $issue_from_tracker = eval { $O{startrek}->get(key => $issue_key) };
        my $tracker_fail = $@;
        if ($tracker_fail) {
            warn $tracker_fail;
            next;
        }
        $issue_from_tracker->{assignee} ||= 'Unassigned';
        $issue_from_tracker->{qa_engineer} = $issue_from_tracker->{qaEngineer};

        # длинные заголовки укорачиваем
        if (length $issue_from_tracker->{summary} > 180) {
            $issue_from_tracker->{summary} = substr($issue_from_tracker->{summary}, 180) . "...";
        }

        my $sth = $dbh->prepare("select issue_id from metatracker_issue where issue_id = ?");
        $sth->execute( $issue->{key} );
        my $issue_from_db = $sth->fetchrow_hashref;
        #print "issue_from_db: $issue_from_db, tracker_fail: $tracker_fail\n";
        if ( $issue_from_db && ! $tracker_fail ) {
            # уже записанный в базу тикет, обновляем
            #print "updating $issue->{key}\n";
            my $sth = $dbh->prepare("update metatracker_issue set summary = ?, assignee = ?, qa_engineer = ? where issue_id = ?");

            $sth->execute( sanitize_text($issue_from_tracker->{summary}), $issue_from_tracker->{assignee}, $issue_from_tracker->{qa_engineer} // '', $issue_from_tracker->{key} );
        } elsif ( !$issue_from_db ) {
            # сейчас тикеты, информацию по которым не удалось получить, не обрабатываются вообще, т. к. нельзя получить таймлайн для них, так что следующая строчка никогда не выполняется
            $issue_from_tracker = {key => $issue_key, summary => 'fake', assignee => ''} if $tracker_fail;
            # новый тикет
            #print "inserting $issue->{key}\n";
            my $sth = $dbh->prepare("insert into metatracker_issue set issue_id = ?, summary = ?, assignee = ?, qa_engineer = ?");

            $sth->execute( $issue_from_tracker->{key}, sanitize_text($issue_from_tracker->{summary})//'', $issue_from_tracker->{assignee}//'', $issue_from_tracker->{qa_engineer}//'' );
        }
    }

    my $sth = $dbh->prepare("select release_id from svnrelease_svnrelease where jira_id = ? order by create_time desc limit 1");
    $sth->execute($O{release_key});
    my $release = $sth->fetchrow_hashref;# or die "Can't find release_id for jira_id $O{release_key}";
    unless ($release){
        warn "Can't find release_id for jira_id $O{release_key}";
        next;
    }
    my $release_id = $release->{release_id};

    for my $issue (@{$O{issues}}){
        my $origin = context_type2origin($issue->{context_type});
        my $sth = $dbh->prepare("select issue_id from metatracker_problem where release_id = ? and issue_id = ? and origin = ? and context_details = ?");
        $sth->execute( $release_id, $issue->{key}, $origin, $issue->{context_details} || '' );
        if ( ! $sth->fetchrow_hashref ){
            #print "inserting problem $issue->{key}, context = $issue->{context_type}, release_id: $release_id\n";
            my $sth = $dbh->prepare("insert into metatracker_problem set release_id = ?,
                issue_id = ?,
                origin = ?,
                context_details = ?,
                relevance = ?,
                significance = ?"
            );
            $sth->execute(
                $release_id,
                $issue->{key},
                $origin,
                $issue->{context_details}||'',
                $RELEVANCE_UNKNOWN,
                $SIGNIFICANCE_UNKNOWN
            );
        }
    }

    return;
}

=head2 sanitize_text($text)

    Оставляем в тексте только цифры, буквы, пробелы и знаки препинания
    Нужно чтобы обойти проблему сохранения эмоджи в mysql DIRECT-116699

=cut

sub sanitize_text
{
    my ($text) = @_;
    return $text =~ s/[^0-9\s\w\p{PosixPunct}]+//ugr;
}

sub context_type2origin
{
    my ($context_type) = @_;

    return $ORIGIN_DESCRIPTION if $context_type eq 'description';
    return $ORIGIN_HOTFIX      if $context_type eq 'hotfix';
    return $ORIGIN_COMMENT     if $context_type eq 'comment';
    die "unknown \$context_type = '$context_type'";
}


sub fetch_issues_commits
{
    my %O = @_;

    my $dbh = $O{dbh};

    my $sth = $dbh->prepare("select sl.rev, sl.message from svnlog_svnlog sl left join metatracker_issue_commits ic on ic.svnlog_id = sl.rev where ic.svnlog_id is null order by sl.rev desc limit 400");
    $sth->execute();
    while ( my $c = $sth->fetchrow_hashref ) {
        #print "processing rev $c->{rev}, $c->{message}\n";
        #print "processing rev $c->{rev}\n";
        my $issues_from_comment = extract_issues_from_text(text => $c->{message});
        #print "issues: ".join(', ', map {$_->{key}} @$issues_from_comment)."\n";

        for my $issue (@$issues_from_comment){
            my $sth = $dbh->prepare("select issue_id from metatracker_issue where issue_id = ?");
            $sth->execute( $issue->{key} );
            if ( !$sth->fetchrow_hashref ) {
                #print "inserting fake $issue->{key}\n";
                my $sth = $dbh->prepare("insert into metatracker_issue set issue_id = ?, summary = 'fake', assignee = '', qa_engineer = ''");
                $sth->execute( $issue->{key});
            }

            my $sth2 = $dbh->prepare("insert into metatracker_issue_commits set svnlog_id = ?, issue_id = ?");
            $sth2->execute( $c->{rev}, $issue->{key});
        }
    }

    return;
}


