#!/usr/bin/perl -w


=head1 NAME
    

=head1 SYNOPSIS

    migrator --svn <svn url> --server <mirgator server url> --status-file <file>

=head1 DESCRIPTION

    Опции:
        -h, --help
            показать справку

        --svn <svn url>
            url в репозитории -- проект, из которого надо собирать миграции

        --server <url>
            url, по которому следует обращаться к веб-сервису хранения миграций

        --status-file <path>
            файл, в котором хранится состояние выполнения (номер обработанной ревизии)


    ./register_migrations.pl --status-file status-file -v svn+ssh://svn.yandex.ru/direct/trunk -p 27000 -h HEAD -w "http://direct-dev.yandex-team.ru:8800/migrations/update"


    $Id$

=head1 METHODS

=cut

use strict;
use warnings;

use Safe;
use File::Slurp;
use Getopt::Long;
use List::MoreUtils qw/before/;
use Data::Dumper;
use LWP::UserAgent;
use JSON;
use YAML;
use Encode;
use Digest::MD5 qw(md5_hex);

use ProjectSpecific qw/svn_url/;

# нестандартный use из-за того, что Module::Info (и debosh) имеют какую-то проблему с обработкой DBIx::MyParsePP (сборка зависает)
eval "use MigratorB::Parse";

use utf8;
use open ':std' => ':utf8';

#................................................

run() unless caller();

sub run
{
    my $opt = parse_options();

    my $migrations = list_migrations(
        svn => $opt->{svn},
        processed_revision => $opt->{processed_revision}, 
        head_revision => $opt->{head_revision}
    );
    #print STDERR "migrations:\n".Dumper($migrations);
    exit unless @$migrations;

    parse_tasks($migrations);
    #print STDERR "parsed:\n".Dumper($migrations);

    my $form = form($migrations);
    #print STDERR "form:\n".Dumper($form);
    exit unless keys %$form;

    submit($opt->{server}, $form);

    write_status($opt);

    exit;
}


#................................................


=head2 list_migrations

    # Верхняя ревизия в репозитории
    # Берем дифф от ревизии с последней обработанной + 1 до верхней (список модифицированных файлов)
    # отфильтровываем только файлы миграций: из транка и из всех бранчей
    # Для каждого файла миграции 
    #    svn info --> последняя модифицированная версия, если она меньше последней обработанной -- пропускаем файл

    на выходе: 
    [
    {
        filename => "20110934_mega_feature.migr", 
        source => "/trunk", 
        url => "svn+ssh://svn.yandex.ru.../20110934_mega_feature.migr", 
        rev => 123456,
    },
    {
        filename => "20111541_zouzoules_fill.migr", 
        source => "/branches/zouzoules", 
        url => "svn+ssh://svn.yandex.ru.../20111541_zouzoules_fill.migr", 
        rev => 123480,
    },
    ]

=cut
sub list_migrations
{
    my %O = @_;

    my $svn_root = svn_url('root');
    my $svn_to_inspect = $O{svn} || $svn_root;

    my $diff_files = `svn diff --summarize -r $O{processed_revision}:$O{head_revision} $svn_to_inspect`;
    return [] unless $diff_files;

    my @urls = grep { m!\Q$svn_root\E/(branches/[^/]+|trunk)/deploy/! } split "\n", $diff_files;
    s/^.*(\Q$svn_root\E[^ ]*)\s*$/$1/ for @urls;
    #print STDERR join "", map {"$_\n"} @urls;

    my @migrations;

    for my $url ( @urls ){
        my $file_svn_info = `svn info $url`;
        next unless $? >> 8 == 0;
        $file_svn_info =~ /Last Changed Rev: +(\d+)/;
        my $rev = $1;
        $file_svn_info =~ /Last Changed Date: +([\d-]+ +[\d:]+)/;
        my $time = $1;
        $url =~ m!\Q$svn_root\E(/branches/[^/]+|/trunk)/deploy/([^/]+)$!;
        my ($source, $filename) = ($1, $2);
        next unless $filename;
        next if $rev < $O{processed_revision} && $source ne "/trunk";
        push @migrations, { filename => $filename, source => $source, url => $url, rev => $rev, time => $time };
    }

    return \@migrations;
}


=head2 

    svn cat --> текст
    разобрать и проверить

=cut
sub parse_tasks
{
    my ($migrations) = @_;

    for my $m (@$migrations){
        $m->{text} = `svn cat $m->{url}`;

        my $arr = MigratorB::Parse::parse_migrations( {$m->{filename} => $m->{text}} );
        die "unexpected array length" unless @$arr == 1; 

        my @errors = MigratorB::Parse::extract_errors($arr);
        if ( @errors ){
            print STDERR "skipping $m->{url} due to parse errors:\n". join("\n", @errors)."\n";
            next;
        }

        $m->{parsed} = $arr->[0]; 
    }

    return;
}

=head2 form

    Составление формы для отправки на сервер

    migration_$i=<filename, source и т.п. в json'е>

=cut
sub form
{
    my ($migrations, %O) =@_;

    my $form = { };
    my $i = 0;
    for my $m (@$migrations){
        next unless exists $m->{parsed};
        my $to_submit = {};
        $to_submit->{$_} = $m->{$_} for qw/filename source url rev time text/;
        $to_submit->{file_hash} = md5_hex( encode('utf8', $m->{text}) );
        my $j = 0;
        my $tasks = $m->{parsed}->{format} eq 'plaintext' ? [{}] : $m->{parsed}->{tasks};
        $to_submit->{task_count} = scalar @$tasks;
        $form->{"migration_$i"} = encode 'utf8', to_json $to_submit;
        $i++;
    }

    return $form;
}


=head2 submit

    отправляет все найденные таски на сервер

=cut
sub submit
{
    my ($url, $full_form) = @_;

    while(keys %$full_form){
        my $form = { map { $_ => delete $full_form->{$_} } grep { $_ } ((keys %$full_form)[0..19]) };
        my $response = LWP::UserAgent->new(timeout => 10)->post($url, Content => $form);

        print "response_status: ".$response->code." content: ".$response->content."\n";
        die unless $response->code == 200;
    }

    return;
}


# возвращает хеш %O с опциями из командной строки
sub parse_options
{
    my %O = (
        head_revision => 'HEAD',
    );

    GetOptions (
        "help"              => \&usage,
        "project=s"         => \$ProjectSpecific::PROJECT,
        "v|svn=s"           => \$O{svn},
        "w|server=s"        => \$O{server},
        "f|status-file=s"   => \$O{status_file},
        "p|processed-rev=s" => \$O{processed_revision},
        "h|head-rev=s"      => \$O{head_revision},
    );

    die "status file missed" unless $O{status_file};
    write_file($O{status_file}, {atomic => 1}, Dump({})) unless !-f $O{status_file};
    my $status = -f $O{status_file} ? YAML::LoadFile($O{status_file}) : {}; 
    $O{processed_revision} ||= $status->{processed_revision};

    die "svn url missed" unless $O{svn};
    die "migrator web service url missed" unless $O{server};
    die "processed_revision missed" unless $O{processed_revision}; 

    if( $O{head_revision} !~ /^\d+$/ ){
        my $svn_info = `svn info $O{svn}\@$O{head_revision}`;
        die unless $? >> 8 == 0;
        $svn_info =~ /Revision: +(\d+)/;
        $O{head_revision} = $1;
    }

    return \%O;
}


sub write_status
{
    my %O = %{$_[0]};

    die "incorrect head_revision $O{head_revision}" unless $O{head_revision} =~ /^\d+$/;
    write_file($O{status_file}, {atomic => 1}, Dump({processed_revision => $O{head_revision}}));
}

# Печатает usage-сообщение
sub usage {
    system("podselect -section NAME -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8 >&2");
    exit(1);
}

