#!/usr/bin/perl -w


=head1 NAME
    
    MigratorB::Server -- функции для сервера миграций  

=head1 DESCRIPTION

    $Id$

=head1 METHODS

=cut

package MigratorB::Server;

use strict;
use warnings;

use Safe;
use File::Slurp;
use List::MoreUtils qw/before uniq/;
use Yandex::DBTools;
use Yandex::DBSchema;
use JSON;
use Data::Dumper;


use utf8;
use open ':std' => ':utf8';

#................................................

our $DEFAULT_DB;

=head2 update

    обновить состояние переданных заданий

=cut
sub update
{
    my ($multiform) = @_;

    my $tasks = form2tasks($multiform);
    my $todo = filter_tasks($tasks);
    update_db($todo);

}


=head2 _form2tasks

    разобрать форму в удобный массив заданий

=cut
sub form2tasks
{
    my ($multiform) = @_;

    my @tasks;
    for my $k (keys %$multiform){
        next unless $k =~ /^task_\d+$/;
        my $t = decode_json( $multiform->{$k} );
        push @tasks, $t;
    }

    return \@tasks;
}


=head2 _update_db


=cut 
sub update_db
{
    my ( $todo ) = @_;

    # TODO
    # собрать все базы, в которых будут изменения
    # для каждой -- проверить, что в ней есть таблица
    # если таблицы нет -- создать + записать в databases

    for my $cond (@{$todo->{delete}}){
        for my $db (@{migrator_databases()}){
            do_delete_from_table($db, "migrator_tasks", where => $cond);
        }
    }

    my @dbs_to_insert = uniq grep {$_ ne ''} map {$_->{db}} @{$todo->{insert}};
    for my $db (@dbs_to_insert){
        check_or_create_tasks_table($db);
    }

    for my $t (@{$todo->{insert}}){
        do_insert_into_table(_eff_db($t->{db}), "migrator_tasks", $t);
    }

    return;
}


=head2 

    На входе: имя базы из миграции
    На выходе: имя базы, в которую надо записать миграцию 

=cut
sub _eff_db
{
    my ($db_name) = @_;

    eval {get_db_config($db_name)};
    return $@ || $db_name eq 'ppcprofile' ? $DEFAULT_DB : $db_name; #TODO исключение для ppcprofile -- убрать
}


=head2 

=cut
sub check_or_create_tasks_table
{
    my ($db) = @_;

    $db = _eff_db($db);

    my $ok = get_one_field_sql($DEFAULT_DB, "select db from migrator_databases where db = ?", $db);

    return if $ok;

    create_table_by_schema($db, 'migrator_tasks', like => "$DEFAULT_DB.migrator_tasks", if_not_exists => 1);
    do_insert_into_table($DEFAULT_DB, 'migrator_databases', {db => $db}, ignore => 1);

    return;
}


=head2 _filter_tasks

    на входе -- ссылка на массив тасков (из формы)

    на выходе: 
    {
        delete => [ {}, {} ], 
        insert => [$t1, $t2, ...]
    }

=cut
sub filter_tasks
{
    my ($tasks) = @_;

    my $todo = { 
        delete => [],
        insert => [],
    };

    my @old;
    my @filenames = map {$_->{filename}} @$tasks;
    for my $db (@{migrator_databases()}){
        my $old_tasks_db = get_all_sql($db, [
            "select db, hash, filename, task, source from migrator_tasks", 
            where => {filename => \@filenames}]
        ) || [];
        push @old, @$old_tasks_db;
    }

    for my $t (@$tasks){
        if ( ! grep { hashes_match($_, $t, qw/filename/) } @old ) {
            # нет записей с таким же filename -- просто записываем
            push @{$todo->{insert}}, $t;
        } elsif ( grep { hashes_match($_, $t, qw/filename source hash/) } @old ) {
            # есть запись с таким же filename, source, md5 (hash)
            # -- пропускаем
        } elsif ( grep { hashes_match($_, $t, qw/filename hash/) && $_->{source} eq "trunk" } @old ) {
            # есть транковая миграция с таким же filename, hash
            # -- пропускаем, это мерж из транка приехал
        } elsif( grep { hashes_match($_, $t, qw/filename source/) && $_->{hash} ne $t->{hash} } @old ){
            # есть старая версия этой же миграции -- 
            # удаляем старую, записываем новую
            push @{$todo->{delete}}, {filename => $t->{filename}, source => $t->{source}};
            push @{$todo->{insert}}, $t;
        } else {
            push @{$todo->{insert}}, $t;
        }
    }

    return $todo;
}


=head2 migrator_databases

=cut
sub migrator_databases()
{
    return get_one_column_sql($DEFAULT_DB, "select db from migrator_databases")||[]
}

=head2 hashes_match

    Параметры: две ссылки на хеши + список полей

    Результат: 1, если эти поля в хешах совпадают, иначе 0

=cut
sub hashes_match
{
    my ($h1, $h2, @keys) = @_;

    for my $k (@keys){
        if ( 
            exists $h1->{$k} != exists $h2->{$k} || 
            defined $h1->{$k} != defined $h2->{$k} ||
            ($h1->{$k} || '') ne ($h2->{$k} || '')
        ){
            return 0;
        }
    }
    return 1;
}


=head2 list

    выдать список миграций с указанными именами 
    TODO другие критерии отбора: примененность, возраст

=cut
sub list 
{
    my ($multiform) = @_;

    my @list;
    for my $db (@{migrator_databases()}){
        push @list, @{get_all_sql($db, "select source, filename, count(*) from migrator_tasks group by source, filename")||[]};
    }

    return \@list;
}


=head2 get

    выдать записи о запрошенных миграциях

=cut
sub get
{
    my ($multiform) = @_;

    my @tasks; 

    for my $db (@{migrator_databases()}){
        push @tasks, get_all_sql($db, [
        "select * from migrator_tasks", 
        where => {filename => [$multiform->get_all('filename')]}
        ]
        ) || [];
    }

    return \@tasks;
}


=head2 drop_all_tasks

=cut
sub drop_all_tasks
{
    my @dbs;
    for my $db (@{migrator_databases()}){
        push @dbs, $db;
        do_delete_from_table($DEFAULT_DB, 'migrator_databases', where => {db => $db});
        do_sql($db, "drop table migrator_tasks");
    }

    return \@dbs;
}

1;
