use strict;
use warnings;

=head1 NAME
    
    MigratorB::Parse -- разбор и проверка Direct-style миграций 

=head1 MIGRATION_SYNTAX

    Формат миграций, модель данных: 

    * миграция == файл (обычно в каталоге deploy);
    * миграция пишется в виде Perl-структуры данных; 
    * миграция состоит из одного или нескольких заданий (тасков, task);
    * задание может быть одного из типов: sql, script, manual;
    * в задании должно быть указано, когда его выполнять: до или после пакетов, 
      в любое время или по особой инструкции (before|after|any|instructions) 
    * для задания типа sql должно быть указано, в какой базе оно должно быть выполнено;
    * задание типа sql|script содержит одно или несколько действий (action), 
      т.е. несколько sql-запросов или несколько скриптов;
    * к заданию можно указать комментарий; 

    $migr = [
        {
            type => 'sql', 
            when => 'any',
            db => 'ppc', 
            sql => 'alter table campaigns drop index i_uid',
            comment => 'верный способ сломать Директ',
        }, 
        {
            type => 'script',
            when => 'after',  # в действительности запускать скрипты "до пакетов" бессмысленно
            script => [ 'protected/getNews.pl', 'protected/ppcGetFraudStat.pl' ],
        },
    ]

    ВАЖНО! Лучше не ориентироваться на это описание, а генерировать шаблон миграции скриптом migr-create.

=head1 TODO

 * убрать обработку no_plaintext, теперь это единственный правильный режим

=head1 METHODS

=cut

# $Id$

package MigratorB::Parse;

use Safe;
use JSON;
use YAML;
use File::Slurp;
use List::MoreUtils qw/before/;

# нестандартный use из-за того, что Module::Info (и debosh) имеют какую-то проблему с обработкой DBIx::MyParsePP (сборка зависает)
eval "use DBIx::MyParsePP";

use base qw/Exporter/;
our @EXPORT = qw/
to_text
extract_errors
read_migrations
parse_migrations
validate_migrations
compile_texts
/;


use utf8;
use open ':std' => ':utf8';

#................................................
our $MAX_ALLOWED_SQL_LENGTH = 1700;

our %ALLOWED_FIELDS_FOR_TASK = (
    script => { 
        obligatory => [qw/
            script
            when
            time_estimate
        /
        ],
        optional => [qw/
            instructions
            comment
        /]
    },
    sql => {
        obligatory => [qw/
            sql
            db
            when
            time_estimate
        /
        ],
        optional => [qw/
            instructions
            comment
            webstop
        /]
    },
    manual => {
        obligatory => [qw/
            text
            when
            time_estimate
        /],
        optional => [qw/
            instructions
        /]
    },
);


#................................................

=head2 to_text

    На входе -- имя файла и его текст (позиционные параметры), 
    на выходе -- скомпилированный удобный для человека текст миграции, если миграция была в формате .migr;
                 или исходный текст без изменений, если миграция не была в соотв. формате

=cut 
sub to_text
{
    my ($filename, $deploy_text, %O) = @_;

    my $migrations = parse_migrations({$filename => $deploy_text}, no_plaintext => $O{no_plaintext});

    compile_texts($migrations);

    if ($O{die_on_errors}) {
        my @errors = extract_errors($migrations);
	die "errors:\n".join("\n", @errors)."\n" if @errors;
    }

    # TODO вернуть сообщения об ошибках после введения обязательной валидации
    #return join "\n", map {( @{$_->{errors} || []}, $_->{text})} @$migrations;
    return join "\n", map {( $_->{text})} @$migrations;
}


sub extract_errors
{
    my ($migrations) = @_;
    return map {@{$_->{errors} || []}} @$migrations;
}


=head2 read_migrations

    Читает файлы, перечисленные в @ARGV 
    
    Параметры
        $opt

    Результат -- ссылка на хеш
        {
            имя_файла_1 => 'содержимое_файла_1',
            имя_файла_2 => 'содержимое_файла_2',
            ...
        }

=cut
sub read_migrations
{
    my ($opt) = @_;

    my $files = {};

    for my $file (@ARGV){
        my $text = $file eq '-' ? read_file(\*STDIN, binmode => ":utf8") : read_file($file, binmode => ':utf8');
        $file = $opt->{stdin_name} if $file eq '-' && $opt->{stdin_name}; 

        $files->{$file} = $text;
    }

    return $files;
}


=head2 parse_migrations

    Параметры
        $files -- хеш, возвращенный read_migrations

    Результат 
        $migrations -- ссылка на массив немного обработанных миграций
        у миграций проставлен формат а аппрувер, 
        а задания и действия приведены к ссылкам на массивы

=cut
sub parse_migrations
{
    my ($files, %O) = @_;

    my $cpt = new Safe;
    $cpt->permit_only(qw(const padany lineseq leaveeval pushmark list stringify anonlist anonhash refgen rv2gv));

    my @migrations;

    for my $file (keys %$files){
        my $text = $files->{$file};

        my $migration = {};
        push @migrations, $migration;

        $migration->{filename} = $file; 
        # аппрувер из комментария, не из структурированной миграции
        my $textual_approver = '';
        if( $text =~ /^[#\s]*(?:approved\s+by)\s+(\S+)\s*$/sm){
            $textual_approver = $1;
        }

        # Если файл .pl -- выкидываем код, оставляем только комментарии к запуску
        # это нужно и для plaintext-миграций, и для структурированных
        # почти copy-paste из pod2crontab
        if ( $file =~ /\.pl/ ) {
            # Извлекаем: pod-секцию DEPLOY...
            my $pod_deploy = '';
            if ( $text =~ /^=head.\s+DEPLOY/sm ){
                ($pod_deploy) = $text =~ /^=head.\s+DEPLOY\s*\n(.*?)^=[a-zA-Z]/sm ;
            }
            # ... и первый блок комментариев
            my @lines = before { $_ !~ m/^(#|\s*$)/ } split "\n", $text;
            push @lines, $pod_deploy;
            push @lines, "#... perl-code skipped ...";
            $text = join "\n", @lines;
        } elsif ( $file =~ /\.js$/ ){
            # в js-файле берем первый блок комментариев и считаем, что там записана обыкновенная миграция
            unless ($text =~ m!^\s*/\*(.*?)^\s*\*/!sm){
                die "$migration->{filename}: can't find deploy instructions\n"; 
            }
            $text = $1."\n#... js code skipped ...";
        }

        # проверяем формат файла
        my $format = '';
        if ( 
            $file =~ /\.migr$/ ||  # файл .migr -- perl-структура
            $text =~ /^\s*#?\s*\.migr\s*$/sm  # в тексте встречается строчка/комментарий .migr -- perl-структура
        ){
            $format = "pl-migr";
        } elsif ( $file =~ /\.migr\.json$/ ) {
            $format = "json-migr";
        } elsif ( $file =~ /\.migr\.yaml$/ ) {
            $format = "yaml-migr";
        } elsif ( $file =~ /\.js$/ ){
            $format = "js-migr";
        } else {
            die "unsupported format";
            # все остальное -- не разрешаем
        }

        $migration->{format} = $format;

        my $data;
        if ( $format eq "pl-migr" ) {
            ($data) = $cpt->reval($text);
            if ( $@ ) {
                $migration->{errors} = [ "$migration->{filename}: $@" ]; 
            }
        } elsif ( $format eq "json-migr" ) {
            eval {$data = from_json($text);};
            if ( $@ ) {
                $migration->{errors} = [ "$migration->{filename}: $@" ]; 
            }
        } elsif ( $format eq "yaml-migr" ) {
            eval {$data = YAML::Load($text);};
            if ( $@ ) {
                $migration->{errors} = [ "$migration->{filename}: $@" ]; 
            }
        } else {
            die "can't be";
        }
        
        if ( !scalar @{$migration->{errors} ||[]} ){
            if ( $format =~ /^(yaml-migr|json-migr)$/ && ref $data ne 'HASH' ){
                push @{$migration->{errors}}, "expecting top-level hashref";
            } elsif ( $format =~ /^(yaml-migr|json-migr)$/ && !exists $data->{tasks} ){
                push @{$migration->{errors}}, "expecting tasks array";
            } elsif ( $format =~ /^(yaml-migr|json-migr)$/ && ref $data->{tasks} ne "ARRAY" ){
                push @{$migration->{errors}}, "expecting tasks array";
            }
        }

        if ( scalar @{$migration->{errors} ||[]} ){
            next;
        }

        if ( $format =~ /^(pl-migr)$/ && (ref($data) eq 'ARRAY' || !exists $data->{tasks} ) ) {
            # адаптивный perl-формат: можно хеш, можно массив хешей, аппрув не является частью данных
            $migration->{format} = 'pl-migr';
            $migration->{tasks} = arrayref($data);
            $migration->{approved_by} = $textual_approver;
        } elsif ( $format =~ /^(yaml-migr|json-migr|pl-migr)$/ ){
            # ригидный json- или yaml- формат: строго массив тасков, аппрув является частью данных
            # или perl-версия того же ригидного формата
            for my $k ( keys %$data ){
                $migration->{$k} = $data->{$k}; 
            }
        } else {
            die;
        }

        if (ref $migration->{approved_by} eq ''){
            $migration->{approved_by} = [ split /\s*,\s*/, $migration->{approved_by} // '' ];
        }

        my $tasks = $migration->{tasks};

        for my $task ( @$tasks ){
            if ( $task->{type} =~ /^sql$/ ){ 
                $task->{sql} = arrayref($task->{sql});
            } elsif ( $task->{type} eq 'script' ){
                $task->{script} = arrayref($task->{script} || '');
                $_ ||= $file for @{$task->{script}}                    
            } elsif ( $task->{type} eq 'manual' ){
            } else {
                message("unknown migration type $task->{type}");
            }
        }
    }

    validate_migrations(\@migrations, no_plaintext => $O{no_plaintext});

    return \@migrations;
}


sub validate_migrations
{
    my ($migrations, %O) = @_;

    my %ALLOWED;
    # преобразуем списки разрешенных полей в удобные хеши
    for my $type ( keys %ALLOWED_FIELDS_FOR_TASK ){
        for my $status (qw/obligatory optional/){
            $ALLOWED{$type}->{$status} = {map {$_ => 1} @{$ALLOWED_FIELDS_FOR_TASK{$type}->{$status}} };
        }
    }

    for my $m (@$migrations){
        # допустимые форматы миграций
        push(@{$m->{errors}}, "incorrect migration format '$m->{format}'"), next unless $m->{format} =~ /^(pl-migr|json-migr|yaml-migr)$/ || $m->{format} eq 'plaintext' && !$O{no_plaintext};

        my @errors;
        for my $t (@{$m->{tasks}}){
            # допустимые типы заданий
            push(@errors, "incorrect task type '$t->{type}'"), next if !exists $ALLOWED{$t->{type}};
            # в заданиях должны быть только разрешенные поля
            for my $f ( keys %$t ){
                next if $f eq 'type';
                if ( !$ALLOWED{$t->{type}}->{obligatory}->{$f} && !$ALLOWED{$t->{type}}->{optional}->{$f} ){ 
                    push(@errors, "field '$f' isn't allowed for task of type '$t->{type}'");
                }
            }
            # обязательные поля должны присутствовать обязательно
            for my $f (keys %{$ALLOWED{$t->{type}}->{obligatory}}){
                push @errors, "field '$f' is obligatory for task of type '$t->{type}'" if !exists $t->{$f};
            }

            # допустимый формат when 
            push @errors, "incorrect 'when' field: $t->{when}" if $t->{when} && $t->{when} !~ /^(before|any|after|instructions)$/;
            push @errors, "instruction should be specified for when = instructions" if ($t->{when} || '') eq 'instructions' && !$t->{instructions};
            push @errors, "instruction may be specified only for when = instructions" if ($t->{when} || '') ne 'instructions' && $t->{instructions};
            push @errors, "script can only run after it is deployed (when => 'after|instructions')" if $t->{type} eq 'script' && ($t->{when} || '') !~ /^(after|instructions)$/;
            if( $t->{type} eq 'sql'){
                push(@errors, validate_sql(sql => $_)) for @{arrayref($t->{sql})};
            }
        }
        s/^/$m->{filename}: / for @errors;
        push @{$m->{errors}}, @errors;
    }

    return;
}


=head2 validate_sql

    Проверяем sql-запрос: 

    * не слишком длинный
    * нет синтаксических ошибок
    * не склеены в одну строчку несколько запросов через ';'

=cut
{
# инициализация парсера -- тяжелая, поэтому храним его между вызовами 
# на 120 Директовых миграциях: 1.2 сек с персистентным парсером и 8.6 сек с инициализацией на каждый запрос
my $parser;
sub validate_sql
{
    my %O = @_;
    my $sql = $O{sql};
    $sql =~ s/;\s*$//;

    my @errors;

    # длина
    push @errors, "too long sql query ".(length $sql).", expected <= $MAX_ALLOWED_SQL_LENGTH" if length $sql > $MAX_ALLOWED_SQL_LENGTH;

    # синтаксис
    $parser ||= DBIx::MyParsePP->new();
    my $query = $parser->parse($sql);
    if (not defined $query->root()) {
        my $message = join "", ( 
            "Parsing query. Error at pos ", $query->pos(), ", line ".$query->line(), 
            "\n",
            $parser->[0]->YYData->{ERRMSG} || (
                "expected: ", join(" ", @{$query->getExpected()}),
                "\n",
                "actual: ", join(" ", map { $_->value() } $query->actual()),
            ),
            "\n",
        ); 
        push @errors, $message;
    } elsif ($query->pos() != length($sql) + 1) {
        my $message = join "", (
            "query: $sql", "\n",
            "parse position = ", $query->pos(), ", ",
            "should be query length + 1, i.e. ", length($sql) + 1,
        );
        push @errors, $message;
    }

    return @errors;
}
}


=head2 compile_texts

    Добавляет к миграциям удобный для человека текст 

    В каждом задании появляется значение text

=cut
sub compile_texts
{
    my ($migrations) = @_;

    for my $m (@$migrations){
        next if $m->{format} eq 'plaintext';
        my $tc = 0;
        my @text_parts;
        
        my %field_title = (
            approved_by => 'approved by',
        );
        for my $f (qw/approved_by/){
            next unless exists $m->{$f};
            push @text_parts, (($field_title{$f} || $f) . " " . join ", ", @{arrayref($m->{$f})});
        }
        for my $task ( @{$m->{tasks}} ){
            $tc++;
            push @text_parts, "$tc.";
            my $comment_str = $task->{comment} ? "комментарий: $task->{comment}" : '';
            if ( $task->{type} eq 'sql' ){ 
                push @text_parts, 
                    (map { field_to_text($_ => $task->{$_}) || () } qw/db when instructions time_estimate/, " ", "comment"),
                    '',
                    (map { s/;*$/;/r } @{$task->{sql}});
            } elsif ( $task->{type} eq 'script' ){
                push @text_parts, 
                    $task->{instructions} || "запустить скрипт(ы)",
                    (map { field_to_text($_ => $task->{$_}) || () } qw/when time_estimate/, " ", "comment"),
                    '',
                    @{$task->{script}};
            } elsif ( $task->{type} eq 'manual' ) {
                push @text_parts, 
                    (map { field_to_text($_ => $task->{$_}) || () } qw/when instructions time_estimate/, ),
                    '',
                    $task->{text}
            } else {
                message("$m->{filename}: unknown migration type $task->{type}");
            }
        }
        push @text_parts, "", "(generated by MigratorB from file $m->{filename})";
        $m->{text} = join "\n", @text_parts;
    }

    return;
}

sub field_to_text
{
    my ($name, $val) = @_;

    return ' ' if $name eq ' ';
    return '' unless $val;

    my %str = (
        db => 'в базе %s',
        when => 'когда: %s',
        time_estimate => 'ожидаемое время выполнения: %s',
        comment => 'комментарий: %s',
        instructions => '%s',
    );

    my %vals = (
        before => 'до выкладки пакетов',
        after => 'после пакетов',
        any => 'в любое время до или после пакетов',
        instructions => 'см. инструкцию',
    );

    return sprintf($str{$name}, $vals{$val} || $val);
}


sub message
{
    my ($message, $opt) = @_;

    print "$message\n";
}


sub arrayref
{
    my ($val) = @_;

    return ref $val eq 'ARRAY' ? $val : [$val];
}

1;
