package MultilayerConfig;

=head1 SYNOPSIS

    "Многослойный" конфиг

    "настройки" == набор параметров == хеш (key/value)
    "параметр" == отдельное значение
    параметры берутся из "источников" (файлы, переменные окружения)
    порядок обхода источников фиксируется настройкой; для каждого параметра используется ПОСЛЕДНЕЕ найденное значение
    (последующие конфиги "переопределяют" значения из предыдущих)

    Пример: 
    источники -- файлы conf_1, conf_2, conf_3, conf_4, conf_5
    параметры в них присутствуют в таких сочетаниях:

           | param_1| param_2| param_3| param_4
    .......|........|........|........|........
    conf_1 | val_11 | val_21 | val_31 | val_41
    conf_2 | -      | val_22 | -      | -     
    conf_3 | val_13 | -      | -      | val_43
    conf_4 | val_14 | -      | -      | -       
    conf_5 | -      | -      | -      | val_45

    Итоговые значиня: 
    param_1 = val_14
    param_2 = val_22
    param_3 = val_31
    param_4 = val_45

=head1 DESCRIPTION

    В $METADATA описываются все параметры (имя, описание, если потребуется -- тип, десериализатор и т.п.) 

    В $SOURCES можно перечислить разные источники настроек, 
    и потом запрашивать отдельные параметры. 

    Если параметр находится в нескольких источниках, действует последнее найденное значение. 

    Виды источников: 
      * переменные окружения с заданным префиксом: 'ENV:SOME_PREFIX'
      * отдельный yaml-файл: 'file:/full/path/to/file.yaml'
      * несколько файлов по маске: 'glob:/full/path/*.yaml'

    $MultilayerConfig::METADATA_FILE = "$ROOT/perl/settings/metadata.yaml";

    $MultilayerConfig::SOURCES = [
        'ENV:DIRECT_MULTILAYER_CONFIG_",
        "file:$ROOT/perl/settings/local/db.yaml",
        "file:$ROOT/perl/settings/local/sandbox.yaml",
        "file:$ROOT/perl/settings/local/cocaine.locator.yaml",
        "glob:$ROOT/perl/settings/local/beta.*.yaml",
        "file:$ROOT/perl/settings/local/beta.yaml",
        "file:$ROOT/perl/settings/prod.yaml",
    ];

    use MultilayerConfig;

    $CONFIGURATION = MultilayerConfig::get_param('CONFIGURATION');

    $image_url = get_param("IMAGE_HOST")."/$md5.$extension";


=head1 TODO

    + нужны ли составные значения? (массивы, хеши) - да, это автоматически есть
    
    + как обходиться со всякими $PP_URL = "$PP_BASE_URL/api/v2" ? -- прямо в значениях: PP_BASE_URL: '${PP_URL}/api/v2'
    предусмотреть: эскейпинг, get_raw
    - шаблонизация в массивах и хешах (ключи, значения); рекурсивно; подставлять только простые скалярные значения
    
    - поддержка многих объектов с разными наборами источников -- надо, но позже

    - список всех параметров (из метаданных) для страницы "текущие значения параметров"
    - в частности, "получить все значения за раз, за одну инициализация LiveFile"
      полезно и для Settings, для переопределения переменных из модулей yandex-lib
    - объект "кеш параметров", от которого можно звать те же методы, что от стандартного объекта 

    - юнит-тест на корректную обработку значения undef (можно хранить, должно возвращаться, в т.ч. не портиться при шаблонизации)

    - not_before, not_after -- хочется, но можно потом. Можно предусмотреть синтаксис в микрошаблонах. Что-то вроде ${0.01|2015-05-02 00:00:00|0.15}

    + метаданные -- поддерживать множественные файлы, непересекающиеся наборы параметров
    - тест на множественные файлы метаданных

    + default -- убрать

    - на основе perl critic -- проверка, что только существующие значения используются

    "десять настроек для редактирования из интерфейса":
     - база как источник
     - изолированные наборы настроек

    "перезапускать скрипт при смене параметров": возраст данных
    ??? просто не складывается, может, не надо


    Потом, инфраструктура для продакшена: 

    - предусмотреть файл для переопределений
    - скрипт для его редактирования
      - перед сохранением дифф и проверка синтаксиса
      - логгирование: "кто когда выставил"
    - доставка этого файла по всем машинам
    - мониторинг, что переопределений не слишком много и они живут не слишком долго

=cut

use strict;
use warnings;

use feature qw/state/;

use List::MoreUtils qw/uniq/;

use Yandex::LiveFile::YAML;

use base qw/Exporter/;
our @EXPORT = qw(
    get_param
);

our $METADATA ||= [
];

our $SOURCES ||= [
];

our $MAX_SUBSTITION_DEPTH ||= 10; 

=head2 get_params_stack

    MultilayerConfig::get_params_stack(["passport_url"]) 

    {
        passport_url => [
                { source => "", value => "passport-internal" },
                { source => "", value => },
            ]
    }

    Особый случай: 
    если в качестве $params передан undef -- вернет значения всех параметров

=cut
sub get_params_stack
{
    my ($params, %O) = @_;
    
    my $metadata = _metadata();
    if ( !defined $params ){
        $params = [ keys %$metadata ];
    }
    for my $p (@$params) {
        die "can't find param '$p' in metadata" unless exists $metadata->{$p};
    }

    my %result;
    for my $source (reverse @$SOURCES){
        my @to_get = $O{stop_on_first_found} ? grep { !exists $result{$_} } @$params : @$params;
        last if scalar @to_get == 0;
        my $subresult = _get_params_from_source($source, \@to_get);
        for my $p ( keys %$subresult ){
            push @{$result{$p}}, $subresult->{$p};
        }
    }

    for my $p (@$params) {
        $result{$p} ||= [];
    }

    _process_templates(\%result) unless $O{raw};

    return \%result
}


# микрошаблонизация
# TODO: шаблонизация в нескалярных значениях (хеши, массивы) -- например, $SSRF_PROXY_HEADERS
sub _process_templates
{
    my ($result) = @_;

    for (1 .. $MAX_SUBSTITION_DEPTH){
        my @params_values = map { @$_ > 0 ? $_->[0]->{value} : () } values %$result;
        my $templ_vars = [];
        _extract_templ_vars(\@params_values, $templ_vars);
        last unless @$templ_vars > 0;

        @$templ_vars = uniq @$templ_vars;
        my @new = grep {!exists $result->{$_} } @$templ_vars;

        my $substition = get_params_stack(\@new, stop_on_first_found => 1, raw => 1);

        for my $p ( keys $result ){
            _substitute_templ_vars( \$result->{$p}->[0]->{value}, $result, $substition);
        }
    }
    
    my @params_values = map { @$_ > 0 ? $_->[0]->{value} : () } values %$result;
    my $templ_vars = [];
    _extract_templ_vars(\@params_values, $templ_vars);
    die "too deep recuursion (>$MAX_SUBSTITION_DEPTH), remaining vars: ".join(", ", @$templ_vars) if @$templ_vars > 0;

    for my $v (values %$result){
        next unless @$v > 0;
        next unless defined $v->[0]->{value};
        $v->[0]->{value} =~ s/\\\$/\$/g;
    }

    return;
}


=head2 _extract_templ_vars
# TODO unit-test

    получает ссылку на массив значений
    возвращает ссылку на массив переменных, которые используются в этих строках

=cut
sub _extract_templ_vars
{
    my ($value, $result) = @_;

    if ( !ref $value ) {

        # скаляр -- делаем всю настоящую работу
        my $v = $value;
        return unless defined $v;
        die "incorrect variable reference '$1' in value '$v'" if $v =~ /(?<!\\)(\$$|\$[^\{]|\$\{\}|\$\{[^\}]$|\$\{[^}]+\{)/;
        my @vars = $v =~ m/(?<!\\)\$\{([^\}]+)\}/g;
        my @incorrect_vars = grep { !/^[a-z0-9_\.]+$/i } @vars;
        die "incorrect variables: ".join(", ", @incorrect_vars) if @incorrect_vars > 0;
        push @$result, @vars; 
        
    } elsif ( ref $value eq "ARRAY" ){

        # массив -- обрабатываем каждый элемент по очереди
        for my $v (@$value){
            _extract_templ_vars($v, $result);
        }

    } elsif ( ref $value eq "HASH" ){

        # хеш -- обрабатываем все значения
        for my $v (values %$value){
            _extract_templ_vars($v, $result);
        }

    } else {
        die "unsupported value $value (".ref $value.")";
    }

    return;
}


=head2 _substitute_templ_vars

    TODO юнит-тест

    получает хеш с единичным значением + несколько наборов подстановок (хешей), 
    модифицирует первый хеш: заменяет в value переменные на подстановки 
    если подстановка не найдена -- умирает

=cut
sub _substitute_templ_vars
{
    my ($value_ref, @subst) = @_;

    # рекурсивно обрабатываем структуру (скаляры, массивы, хеши)
    # в хешах подстановка действует только на значения, не на ключи
    if ( !ref $$value_ref ){

        return unless defined $$value_ref;
        $$value_ref =~ s#(?<!\\)\$\{([^\}]+)\}# _find_substitution($1, @subst) || ''; #xge;

    } elsif ( ref $$value_ref eq "ARRAY" ) { 

        _substitute_templ_vars(\$_, @subst) for @$$value_ref;

    } elsif ( ref $$value_ref eq "HASH" ) {

        _substitute_templ_vars(\$_, @subst) for values %$$value_ref;

    } else {
        die "unsupported value $$value_ref (".ref $$value_ref.")";
    }

    return;
}


sub _find_substitution
{
    my ($var, @subst) = @_;

    for my $s ( @subst ){
        if ( exists $s->{$var} && @{$s->{$var} || []} > 0 ){
            # подставлять разрешаем только скаляры
            die "non-scalar substitution aren't allowed ($s->{$var}->[0]->{value} / ".(ref $s->{$var}->[0]->{value}).")" if ref $s->{$var}->[0]->{value};
            return $s->{$var}->[0]->{value};
        }
    }

    die "can't find substitution for var \$$var";
}


=head2 

    Возвращает хеш -- метаданные о параметрах
    Внутри перечитывает файлы, если они обновились

=cut
sub _metadata
{
    state %metadata_live_files;

    my %metadata;
    for my $file ( @$METADATA ){
        $metadata_live_files{$file} ||= Yandex::LiveFile::YAML->new(filename => $file);
        my $metadata_part = eval { $metadata_live_files{$file}->data };
        die "MultilayerConfig: can't get metadata: $@" if $@ ;
        for my $param ( keys %$metadata_part ){
            die "redeclaration of $param, metadata sources: ".join( ",", @$METADATA ) if exists $metadata{$param};
            $metadata{$param} = $metadata_part->{$param};
        }
    }

    return \%metadata;
}


=head2 _get_params_from_source

    _get_params_from_source("file:/var/www/beta.lena-san.9005/perl/settings/local.*.yaml", ["passport_url"]);
    { passport_url=>
        [
            { source => "/var/www/beta.lena-san.9005/perl/settings/local.beta.yaml", value => "passport-internal" },
            { source => "/var/www/beta.lena-san.9005/perl/settings/local.ts.yaml", value => "passport-rc" },
        ]
    }

    Если значение не найдено -- ссылка на пустой массив

=cut
sub _get_params_from_source
{
    my ($source, $params) = @_;
    state %live_files;

    my %result;

    if ( $source =~ /^ENV:(.*)$/){
        for my $p ( @$params ){
            if (exists $ENV{"$1$p"} ){
                $result{$p} = {
                    source => $source, 
                    value => $ENV{"$1$p"},
                };
            }
        }
    } elsif ( $source =~ /^file:(.*\.yaml)$/ ) {
        my $file = $1;
        if (-f $file){
            $live_files{$file} ||= Yandex::LiveFile::YAML->new(filename => $file);
            my $data = eval{ $live_files{$file}->data };
            die "MultilayerConfig: can't get values from file '$file': $@" if $@;
            for my $p ( @$params ){
                if( exists $data->{$p} ){
                    $result{$p} = {
                            source => $source, 
                            value => $data->{$p},
                        };
                }
            }
        } else {
            # несуществующий файл -- может, die?
            delete $live_files{$file};
        }
    } elsif ( $source =~ /^glob:(.*\.yaml)$/ ){
        my $mask = $1;
        for my $file ( sort glob($mask) ){
            my $subresult = _get_params_from_source("file:$file", $params);
            for my $p (keys %$subresult){
                die "redeclaration of param '$p' in glob source '$source'" if exists $result{$p};
                $result{$p} = {
                    source => $source,
                    value => $subresult->{$p}->{value},
                };
            }
        }
    } else {
        die "unsupported source '$source' for params '".(join ",", @$params)."'";
    }

    return \%result;
}


=head2 get_param

    $aqvq_timeout = get_param('aqvq_timeout');

    $param -- название параметра

    если параметр не найден -- умирает

=cut
sub get_param
{
    my ($param) = @_;

    my $st = get_params_stack([$param], stop_on_first_found => 1);

    return $st->{$param}->[0]->{value} if exists $st->{$param} && scalar @{$st->{$param}} > 0;
    die "can't find value of param '$param'";
}



1;
