#!/usr/bin/perl

use strict;
use warnings;

use FindBin qw/$Bin/;

use Test::More;
use Test::Deep;
use Test::Exception;

use MultilayerConfig;

my %test_cases = (
    # хороший синтаксис
    good => {
        # нет переменных
        'some_text' => [],
        'some text' => [],
        '\$some_text' => [],
        'some_text\$' => [],
        'some_\$text' => [],
        '\$some_\$text\$' => [],
        '\$\$some_\$\$text\$\$' => [],
        '\${var1}' => [],
        '\${var1}\${var2}' => [],
        'some \${var1} text' => [],

        # допустимые символы в переменных
        '${var1}' => [qw/var1/],
        '${var_1}' => [qw/var_1/],
        '${var.1}' => [qw/var.1/],
        '${1.var}' => [qw/1.var/],
        
        # разное расположение переменных в тексте
        'text ${var1}' => [qw/var1/],
        '${var1} text' => [qw/var1/],
        'some ${var1} text' => [qw/var1/],

        # сочетание переменных и заэкранированных $
        'some \$${var1} text' => [qw/var1/],
        'some ${var1}\$ text' => [qw/var1/],

        # несколько переменных в разном расположении
        '${var1}${var2}' => [qw/var1 var2/],
        '${var1}text${var2}' => [qw/var1 var2/],
        'some ${var1} text${var2}' => [qw/var1 var2/],
        '${var1}${var2}${var3}' => [qw/var1 var2 var3/],
        '${var}${var}${var}' => [qw/var var var/],
    },
    # некорректный синтаксис
    bad => {
        # неэкранированные $
        'some_text$' => [],
        '$some_text' => [],
        'some$_text' => [],
        '$$$some_text' => [],

        # сочетание экранированных и неэкранированнных $
        'some_text$\$' => [],
        'some$\$_text' => [],
        'some_text\$$' => [],
        'some\$$_text' => [],

        # недопустимые символы в именах переменных
        '${var#1}' => [],
        '${var-1}' => [],

        # сложно искаженные переменные
        '${{some_text}}' => [],
        '${}{some_text}}' => [],
        '${${some}text}some_text' => [],
        '${var1${var2}}}' => [],
        '$ {var}' => [],
        '$ {var}${var1}' => [],
        '${var1}$ {var}' => [],
    },
);

for my $value ( keys $test_cases{good} ){
    my $vars = [];
    lives_ok{ MultilayerConfig::_extract_templ_vars([$value], $vars) } "parsing of good value '$value'";
    cmp_bag( $vars, $test_cases{good}->{$value}, "vars for good value '$value'" ); 
}

for my $value ( keys $test_cases{bad} ){
    my $vars = [];
    dies_ok{ MultilayerConfig::_extract_templ_vars([$value], $vars) } "parsing of bad value '$value'";
}

done_testing();

