#!/usr/bin/perl

use strict;
use warnings;

use FindBin qw/$Bin/;

use Test::More;
use Test::Exception;
use Test::Deep;

use Path::Tiny;

use MultilayerConfig;


my $metadata_1 = 
q(
---
ROOT: 
  description: 'metadata/описание ROOT'

LOG_ROOT:
  description: 'metadata/описание LOG_ROOT'

SUB_LOG_ROOT:
  description: 'metadata/описание SUB_LOG_ROOT'

DIRS:
  description: 'metadata/описание DIRS'

DIRS_2:
  description: 'metadata/описание DIRS_2'

ALLOWED_CHARS:
  description: 'metadata/описание ALLOWED_CHARS'

TEXT_1: 
  description: 'metadata/описание TEXT_1'
);

my %test_cases = (
    good => [
        # пара простых параметров с однократной ссылкой
        {
            metadata => $metadata_1,
        source => 
q(
---
ROOT: /root

LOG_ROOT: ${ROOT}/log
),
            params => {
                ROOT => '/root',
                LOG_ROOT => '/root/log',
            },
        },

        # "рекурсивные" ссылки
        {
            metadata => $metadata_1,
        source => 
q(
---
ROOT: /root

LOG_ROOT: ${ROOT}/log

SUB_LOG_ROOT: ${LOG_ROOT}/subdir
),
            params => {
                ROOT => '/root',
                LOG_ROOT => '/root/log',
                SUB_LOG_ROOT => '/root/log/subdir',
            },
        },

        # "рекурсивные" ссылки + экранирование
        {
            metadata => $metadata_1,
        source => 
q(
---
ROOT: /root

LOG_ROOT: ${ROOT}/log

SUB_LOG_ROOT: ${LOG_ROOT}/subdir

ALLOWED_CHARS: '!@#\$%^&*()-_+'

TEXT_1: '\${unknown_var} \${ROOT}: ${ROOT} \${LOG_ROOT}: ${LOG_ROOT} \${SUB_LOG_ROOT}: ${SUB_LOG_ROOT} \${ALLOWED_CHARS}: ${ALLOWED_CHARS}'
),
            params => {
                ROOT => '/root',
                LOG_ROOT => '/root/log',
                SUB_LOG_ROOT => '/root/log/subdir',
                ALLOWED_CHARS => '!@#$%^&*()-_+',
                TEXT_1 => '${unknown_var} ${ROOT}: /root ${LOG_ROOT}: /root/log ${SUB_LOG_ROOT}: /root/log/subdir ${ALLOWED_CHARS}: !@#$%^&*()-_+',
            },
        },

        # ссылки в сложных значениях
        {
            metadata => $metadata_1,
        source => 
q(
---
ROOT: /root

DIRS: 
 - ${ROOT}
 - /tmp
),
            params => {
                ROOT => '/root',
                DIRS => ['/root', '/tmp'],
            },
        },
        {
            metadata => $metadata_1,
        source => 
q(
---
ROOT: /root

LOG_ROOT: ${ROOT}/log

SUB_LOG_ROOT: ${LOG_ROOT}/subdir

DIRS: 
 - ${ROOT}
 - ${LOG_ROOT}
 - 
   SUB_LOG_ROOT: ${SUB_LOG_ROOT}
   LOG_ROOT: ${LOG_ROOT}
 - /tmp
),
            params => {
                ROOT => '/root',
                LOG_ROOT => '/root/log',
                SUB_LOG_ROOT => '/root/log/subdir',
                DIRS => [ '/root', '/root/log', {SUB_LOG_ROOT => '/root/log/subdir', LOG_ROOT => '/root/log'}, '/tmp' ],
            },
        },


    ],
    bad => [
        # ссылка на неизвестную переменную
        {
            metadata => $metadata_1,
        source => 
q(
---
ROOT: /root

TEXT_1: '${unknown_var} \${ROOT}: ${ROOT}'
),
            params => [
                qw/
                TEXT_1
                /
            ],
        },
        # циклическая ссылка
        {
            metadata => $metadata_1,
        source => 
q(
---
ROOT: '/root/${ROOT}',
),
            params => [
                qw/
                ROOT
                /
            ],
        },
        # циклическая ссылка
        {
            metadata => $metadata_1,
        source => 
q(
---
ROOT: '/root/${LOG_ROOT}',

LOG_ROOT: '/root/${ROOT}',
),
            params => [
                qw/
                ROOT
                LOG_ROOT
                /
            ],
        },
        # нескалярные подстановки
        {
            metadata => $metadata_1,
        source => 
q(
---
ROOT: /root

LOG_ROOT: ${ROOT}/log

SUB_LOG_ROOT: ${LOG_ROOT}/subdir

DIRS: 
 - ${ROOT}
 - ${LOG_ROOT}
DIRS_2:
 - ${DIRS}
),
            params => [
                qw/
                DIRS_2
                /
            ],
        },
    ],
);



for my $type (qw/good bad/){
    for my $case (@{$test_cases{$type}}){
        my $temp_metadata = Path::Tiny->tempfile( "ut-multilayer-config-metadata-XXXXXXXXXXX" );
        $temp_metadata->spew_utf8( $case->{metadata} );
        my $temp_source = Path::Tiny->tempfile( "ut-multilayer-config-source-XXXXXXXXXXX", SUFFIX => '.yaml' );
        $temp_source->spew_utf8( $case->{source} );
        $MultilayerConfig::METADATA = [ "".$temp_metadata->absolute ];
        $MultilayerConfig::SOURCES = [ "file:".$temp_source->absolute ];
        if ($type eq 'good'){
            for my $p ( keys %{$case->{params}} ){
                is_deeply(get_param($p), $case->{params}->{$p}, "value for $p");
            }
        } elsif ( $type eq "bad" ) {
            for my $p ( @{$case->{params}} ){
                dies_ok { get_param($p) } "bad param $p";
            }
        } else {
            die "unknown test type $type";
        }
    }
}


done_testing();

