#!/usr/bin/env python2
# -*- encoding: utf-8 -*-

import copy
import datetime
import errno
import io
import os
import json
import logging
import time
import socket
import signal
import shutil
import yaml
import mds
import re
import MySQLdb
import hashlib
import random

from subprocess import PIPE, Popen
#from pwd import getpwnam

class CreateBackupError(Exception):
    def __init__(self, *args, **kwargs):
        Exception.__init__(self, *args, **kwargs)

class RestoreBackupError(Exception):
    def __init__(self, *args, **kwargs):
        Exception.__init__(self, *args, **kwargs)



WALENV = {
"AWS_ACCESS_KEY_ID": "{access_key}",
"AWS_SECRET_ACCESS_KEY": "{secret_key}",
"WALG_S3_PREFIX": "s3://{name_bucket}/{instance}",
"AWS_ENDPOINT": "https://{address}:443",
"AWS_S3_FORCE_PATH_STYLE": "true",
"WALG_COMPRESSION_METHOD": "brotli",
"WALG_GPG_KEY_ID": "",
"WALG_PGP_KEY_PATH": "/etc/direct-tokens/mdb_pgp_private_key",
"WALG_S3_MAX_PART_SIZE": "{chunk_size}",
"WALG_MYSQL_DATASOURCE_NAME": "root:@unix(/var/run/mysqld.{instance}/mysqld.sock)/mysql",
"WALG_STREAM_CREATE_COMMAND": "xtrabackup --backup --slave-info --stream=xbstream --datadir={mysql_data} --socket=/var/run/mysqld.{instance}/mysqld.sock --user=root",
"WALG_STREAM_RESTORE_COMMAND": "xbstream -x -C {mysql_data}",
"WALG_MYSQL_BACKUP_PREPARE_COMMAND": "xtrabackup --prepare --use-memory={memory_prepare} --target-dir={mysql_data}",
"WALG_MYSQL_BINLOG_REPLAY_COMMAND": "mysqlbinlog --stop-datetime=\"$WALG_MYSQL_BINLOG_END_TS\" \"$WALG_MYSQL_CURRENT_BINLOG\" | mysql",
}
INNOBACKUPEX_WALG = "/usr/bin/wal-g backup-push {mysql_data}"
RESTORE_WALG = "/usr/bin/wal-g backup-fetch {backup_name}"

MYSQL_CONFIG = "/etc/mysql/{0}.cnf"
INNOBACKUPEX_CMD = "/usr/bin/innobackupex --defaults-file={0} --user=root --parallel=4 --stream=tar ./"
PBZIP2_PACK = "/usr/bin/pbzip2 -p8 -c"
PBZIP2_UNPACK = "/usr/bin/pbzip2 -dk -p6 --stdout"
TAR_UNPACK = "/bin/tar -xv"
MYSQLDATA_DIR = "/opt/mysql.{instance}"
RESTORE_DIR = "/opt/mysql.{instance}.tmp"
RESTORE_CMD = "/usr/bin/innobackupex --apply-log --use-memory={0} ."
CHUNK_SIZE = 200*1024*1024 #200M
MEMORY_SIZE_PREPARE = 20*1024*1024*1024 #20G
MONITORING_DIR = "/var/spool/mysql-monitor"
ENVTYPE = "/etc/yandex/environment.type"
ROTATE_DAYS = 14
STALLED_DAYS = 1
HASH_PATH = "/tmp"
HASH_MAX_AGE = 129600 #36h

CA_CERT_PROD_S3 = "/usr/local/share/ca-certificates/yandex/YandexExternalCA.crt"
CA_CERT_TEST_S3 = "/usr/local/share/ca-certificates/YandexInternalRootCA.crt",

alias_types = { 'production':  ['production', 'prod', 'pr'],
                'testing':     ['testing', 'test', 'ts'],
                'development': ['development', 'dev', 'dv']
              }

def load_config(config):
    config_file = config.get('file', None)
    result = {}
    envfile = open(ENVTYPE).read().strip() if os.path.exists(ENVTYPE) else None
    envarg = config.get('enviroment', None)
    envtype = envarg if envarg else envfile
    for alias in alias_types:
        if envtype in alias_types[alias]:
            envtype = alias
            break
    if config_file:
        with open(config_file, 'rb') as fd:
            result = yaml.load(fd)
            if config.get('instances') and result.get(envtype):
                result[envtype]['instances'] = config['instances']
            result[envtype]['secret_key'] = open(result[envtype]['secret_token']).read().strip()
            config.update(result.get(envtype, {}))
    return

def start_logging(config):
    level = 'DEBUG' if config.get('debug') else 'WARNING'
    logger = logging.getLogger('stream logs to console')
    logger.setLevel(level=getattr(logging, level))
    formatter = logging.Formatter('%(asctime)s %(message)s')
    log_file = config.get('log_file')
    create_directory(os.path.dirname(log_file))
    ch = logging.FileHandler(log_file, mode='a')
    ch.setLevel(level=getattr(logging, level))
    ch.setFormatter(formatter)
    logger.addHandler(ch)
    if config['interactive']:
        sh = logging.StreamHandler()
        sh.setLevel(level=getattr(logging, level))
        sh.setFormatter(formatter)
        logger.addHandler(sh)
    config['logger'] = logger
    config['logger_ch'] = ch
    return

def save_hash(config, status, msg):
    with open(config['id'], "w+") as _file:
        w = json.dumps({'status': status,
                        'message': msg
        })
        _file.write(w)
    return

def generate_hash_file(config):
    ghash = hashlib.sha256()
    ghash.update(str(time.time() + random.random()))
    hexdigest = ghash.hexdigest()
    config['id'] = os.path.join(HASH_PATH, "mdss3_" + hexdigest)
    save_hash(config, "NEW", "start mysql-backup-mds")
    return

def clean_old_hashfiles():
    hash_files = [ i for i in os.listdir(HASH_PATH) if i.startswith("mdss3_") ]
    current_time = time.time()
    for name in hash_files:
        path = os.path.join(HASH_PATH, name)
        hash_stat = os.stat(path).st_mtime
        if (int(current_time - hash_stat) > HASH_MAX_AGE):
            try:
                os.remove(path)
            except OSError as err:
                # параллельно запущенный процесс для другого шарда мог уже удалить этот файл
                if err.errno != errno.ENOENT:
                    raise
    return

def write_status(status, instance, mondir=MONITORING_DIR):
    path = '{0}/{1}-s3-status'.format(mondir, instance)
    with open(path, 'w') as fd:
        fd.write(status)
    return

def sizeof_fmt(num, suffix=''):
    unit = ['','K','M','G','T','P','E','Z']
    position = unit.index(suffix)
    if num > 1024.0:
        num /= 1024.0
        return sizeof_fmt(num, unit[position+1])
    else:
        return "{0:.2f}{1}".format(num, unit[position])

def sizeof_int(strnum):
    unit = ['','K','M','G','T','P','E','Z']
    rgxp = re.compile('(\d+)(\D+)')
    value = rgxp.findall(strnum)[0]
    degree = 1
    if len(value) > 0 and value[1] in unit:
        degree = 1024 ** unit.index(value[1])
    return int(value[0]) * degree

def list_backup(config):
    result = {}
    list_instances = bool(False)
    backups = mds.size_keys(**config)
    logger = config.get('logger')
    logger.debug(backups)
    instances = config.get('instances', None)
    if "instances" in instances:
        instances.remove("instances")
        list_instances = bool(True)
    for name_backup in backups:
        if name_backup.find("basebackups") > -1:
            rgxp = re.compile('([a-zA-Z]+[0-9]*)_?(\S+)?/basebackups.*_(\d+)T(\d+)Z/stream*')
            walg_backup = rgxp.findall(name_backup)
	    if len(walg_backup) == 0:
                continue
            instance, _, date, _ = walg_backup[0]
        else:
            instance, date, _ = name_backup.split('_', 2)
        if instances and instance not in instances and not list_instances:
            continue
        value = { 'date': date,
                  'name_backup': name_backup,
                  'size_human': sizeof_fmt(backups[name_backup]),
                  'size': backups[name_backup]
        }
        if not result.has_key(instance):
            result[instance] = []
        result[instance].append(value)
    instances = sorted(result.keys());
    instances = sorted(instances, key=len)
    if config.get('list_table'):
        dates = {}
        for instance in result:
            for value in result[instance]:
                date = value.get('date')
                if not dates.has_key(date):
                    dates[date] = []
                dates[date].append(instance)
        all_dates = dates.keys()
        all_dates.sort()
        head = []
        head.append('{:10s}'.format(''))
        head.extend(['{:10s}'.format(i) for i in all_dates])
        print ''.join(head)
        for instance in instances:
            body = []
            body.append('{:10s}'.format(instance))
            for date in all_dates:
                flag = ' X ' if instance in dates[date] else '   '
                body.append('{:^10s}'.format(flag))
            print ''.join(body)
    elif config.get('list_json'):
        if list_instances:
            print json.dumps(result.keys())
        else:
            print json.dumps(result)
    else:
        if list_instances:
            print "{0}\n".format('\n'.join(result.keys()))
        else:
            for instance in instances:
                body = [ '{0}({1})'.format(i['name_backup'], i['size_human']) for i in result[instance]]
                total = sum([ i['size'] for i in result[instance] ])
                body.sort()
                print "{0}(total:{2})\n\t{1}\n\t".format(instance, '\n\t'.join(body), sizeof_fmt(total))

def process_backup(config):
    name_bucket = config.get('name_bucket')
    logger = config.get('logger')
    module = '[{0}/{1}]'.format(__name__, process_backup.__name__)
    bucket = mds.set_bucket(**config)
    uploads = mds.list_uploads(**config)
    uploads = sorted(uploads)
    uploads = sorted(uploads, key=len)
    if len(uploads) == 0:
        print 'not found running backups for {0}:{1}'.format(config.get('address'), config.get('name_bucket'))
    else:
        print'found running(or stalled) backups for {0}:{1}'.format(config.get('address'), config.get('name_bucket'))
        head = ' {3:^8s}    {0:15s}    {1:^20s}    {2}\n'.format('instance', 'start backup date', 'name backup', 'status')
        body = []
        current_time = datetime.datetime.now()
        for upload in uploads:
            if upload.find("basebackups") > -1:
                rgxp = re.compile('([a-zA-Z]+[0-9]*)_?(\S+)?/basebackups.*_(\d+)T(\d+)Z/stream*')
                walg_backup = rgxp.findall(upload)
                if len(walg_backup) == 0:
                    continue
                instance, src, sdate, stime = walg_backup[0]
	    else:
                instance, sdate, stime, src = upload.split('_', 3)
            backup_time = datetime.datetime.strptime(' '.join([sdate, stime]), '%Y%m%d %H%M%S')
            delta_time = current_time - backup_time
            status = '\033[92mRUNNING\033[0m' if delta_time.days < STALLED_DAYS else '\033[91mSTALLED\033[0m'
            value = '  {3:^8s}    {0:15s}    {1:^20s}    {2}'.format(instance, str(backup_time), upload, status)
            body.append(value)
        line = ' '.join([ '', '='*len(max(body)), '\n'])
        print line, head, line, '\n'.join(body)
    return

def slave_mysql_thread(mysql_config, start=False, config={}):
    module = '[{0}/{1}]'.format(__name__, slave_mysql_thread.__name__)
    logger = config.get('logger')
    try:
        db = MySQLdb.connect(read_default_file=mysql_config)
        cmd = "START SLAVE SQL_THREAD" if start else "STOP SLAVE SQL_THREAD"
        cursor = db.cursor()
        cursor.execute(cmd)
        cursor.close()
        db.close()
    except Exception as err:
        logger.critical("{0} config {1}: {2}. Skip STOP SLAVE".format(module, mysql_config, str(err)))
    return

def find_tmpdir(mysql_config, config={}):
    module = '[{0}/{1}]'.format(__name__, find_tmpdir.__name__)
    logger = config.get('logger')
    try:
        db = MySQLdb.connect(read_default_file=mysql_config)
        cmd = "SHOW GLOBAL VARIABLES LIKE 'tmpdir'"
        cursor = db.cursor()
        cursor.execute(cmd)
        result = cursor.fetchall()[0][1]
        cursor.close()
        db.close()
    except Exception as err:
        logger.critical("{0} config {1}: {2}".format(module, mysql_config, str(err)))
        result = "/tmp"
    return result

def perc_avialible_size(path):
    stat = os.statvfs(path)
    result = stat.f_bavail*100/(stat.f_blocks-(stat.f_bfree-stat.f_bavail))
    return result

def create_backup(config):
    instances = config.get('instances')
    name_bucket = config.get('name_bucket')
    chunk_size = config.get('chunk_size', CHUNK_SIZE)
    logger = config.get('logger')
    module = '[{0}/{1}]'.format(__name__, create_backup.__name__)
    if len(instances) == 0:
        raise ValueError('empty list instances in config')
    if name_bucket is None:
        raise ValueError('empty backet name in config')
    for instance in instances:
        monrun = lambda x: write_status(status=x, instance=instance)
        mysql_config = MYSQL_CONFIG.format(instance)
        temp_dir = find_tmpdir(mysql_config, config)
        if config.get('stop_replication'):
            slave_mysql_thread(mysql_config, False, config)
        mysql_data = config.get('mysql_directory') if config.get('mysql_directory') else os.path.join(MYSQLDATA_DIR.format(instance=instance), 'data')
        if not os.path.exists(mysql_config):
            logger.debug('{0} config {1} not found. Skip backup it'.format(module, mysql_config))
            continue
        if not os.path.exists(mysql_data):
            logger.debug('{0} directory {1} not found. Skip backup it'.format(module, mysql_data))
            continue
        stime = datetime.datetime.fromtimestamp(time.time()).strftime('%Y%m%d_%H%M%S')
        logger.debug('{0} config {1}'.format(module, config))
        #connect MDS, get/create bucket
        bucket = mds.set_bucket(**config)
        keyname = '{0}_{1}'.format(instance, stime)
        #clear MultiPartUploads queue in MDS
        ##logger.warning('{0} remove old MultiPartUploads {1}'.format(module, bucket))
        ##deleted = mds.clean_uploads(prefix=instance, suffix="", **config)
        ##logger.warning('{0} removed {1} in {2}'.format(module, bucket, deleted))
        #start create dump
        logger.warning('{0} start create dump to {1}:{2}/{3}'.format(module, config.get('address'), bucket, keyname))
        monrun('PROGRESS')
        save_hash(config, "PROGRESS", "creating backup {0}".format(keyname))
        if config['walg']:
            walcnf = copy.copy(config)
            walcnf['instance'] = instance
            walcnf['mysql_data'] = mysql_data
            walcnf['chunk_size'] = walcnf['chunk_size'] if walcnf.has_key('chunk_size') and len(walcnf['chunk_size']) > 0 else chunk_size
            create_walg_cmd = INNOBACKUPEX_WALG.format(**walcnf)
            logger.warning('{0} run command {1} / cwd {2}'.format(module, create_walg_cmd, mysql_data))
            walenv = dict([(k, v.format(**walcnf)) for k, v in WALENV.items()])
            if config.has_key('cacert') and len(config['cacert']) > 0:
                walenv['WALG_S3_CA_CERT_FILE'] = config['cacert']
            p1 = Popen(create_walg_cmd, cwd=mysql_data, shell=True, bufsize=0,
                       stderr=config.get('logger_ch').stream.fileno(),
                       stdout=config.get('logger_ch').stream.fileno(),
                       preexec_fn=os.setsid,
                       env=walenv)
            err_free_space = 0
            try:
                while True:
                    if perc_avialible_size(temp_dir) < 5: #если места в temp_dir меньше 95%, то прекращаем создание бекапа
                        os.killpg(os.getpgid(p1.pid), signal.SIGTERM)
                        err_free_space = 1
                    rcode1 = p1.poll()
                    if err_free_space:
                        raise CreateBackupError('{0} free space in {1} less than 5%. Stop backup!'.format(module, temp_dir))
                    if rcode1 is not None and rcode1 != 0:
                        raise CreateBackupError('{0} error command {1} / cwd {2}'.format(module, create_walg_cmd, mysql_data))
                    if rcode1 is not None and rcode1 == 0:
                        break
                p1.wait()
                monrun('SUCCESS')
                save_hash(config, "SUCCESS", "create backup walg {0}".format(keyname))
                logger.warning('create backup success: {0}:{1}/{2}'.format(config.get('address'), bucket, instance))
            except Exception as err:
                logger.critical(err)
                logger.critical('error create backup: {0}:{1}/{2}'.format(config.get('address'), bucket, instance))
                monrun('FAILURE')
                save_hash(config, "FAILURE", "create backup walg {0}".format(keyname))
            finally:
                if config.get('stop_replication'):
                    slave_mysql_thread(mysql_config, True, config)

        else:
            #clear MultiPartUploads queue in MDS
            logger.warning('{0} remove old MultiPartUploads {1} for {2}'.format(module, bucket, instance))
            deleted = mds.clean_uploads(prefix=instance, suffix="", **config)
            logger.warning('{0} removed {1} in {2}'.format(module, bucket, deleted))
            #start initiate upload
            create_dump = INNOBACKUPEX_CMD.format(mysql_config)
            logger.warning('{0} run command {1} / cwd {2}'.format(module, create_dump, mysql_data))
            mp = bucket.initiate_multipart_upload(keyname)
            #exec inobackupex
            p1 = Popen(create_dump, cwd=mysql_data, shell=True, bufsize=0,
                       stderr=config.get('logger_ch').stream.fileno(),
                       stdout=PIPE,
                       preexec_fn=os.setsid)
            p2 = Popen(PBZIP2_PACK, cwd=mysql_data, shell=True, bufsize=0,
                       stdin=p1.stdout,
                       stderr=config.get('logger_ch').stream.fileno(),
                       stdout=PIPE,
                       preexec_fn=os.setsid)

            count = 0
            err_free_space = 0
            try:
                while True:
                    count += 1
                    data = p2.stdout.read(chunk_size)
                    if perc_avialible_size(temp_dir) < 5: #если места в temp_dir меньше 95%, то прекращаем создание бекапа
                        os.killpg(os.getpgid(p1.pid), signal.SIGTERM)
                        os.killpg(os.getpgid(p2.pid), signal.SIGTERM)
                        err_free_space = 1
                    rcode1 = p1.poll()
                    rcode2 = p2.poll()
                    if err_free_space:
                        raise CreateBackupError('{0} free space in {1} less than 5%. Stop backup!'.format(module, temp_dir))
                    if rcode1 is not None and rcode1 != 0:
                        raise CreateBackupError('{0} error command {1} / cwd {2}'.format(module, create_dump, mysql_data))
                    if rcode2 is not None and rcode1 != 0:
                        raise CreateBackupError('{0} error command {1} / cwd {2}'.format(module, PBZIP2_PACK, mysql_data))
                    if rcode1 is not None and rcode2 is not None and len(data) == 0:
                        break
                    mp.upload_part_from_file(io.BytesIO(data), part_num=count)
                p1.wait()
                p2.wait()
                mp.complete_upload()
                monrun('SUCCESS')
                save_hash(config, "SUCCESS", "create backup {0}".format(keyname))
                logger.warning('create backup success: {0}:{1}/{2}'.format(config.get('address'), bucket, keyname))
            except Exception as err:
                mp.cancel_upload()
                logger.critical(err)
                logger.critical('error create backup: {0}:{1}/{2}'.format(config.get('address'), bucket, keyname))
                monrun('FAILURE')
                save_hash(config, "FAILURE", "create backup {0}".format(keyname))
            finally:
                if config.get('stop_replication'):
                    slave_mysql_thread(mysql_config, True, config)
    return

def create_directory(*args):
    try:
        [os.makedirs(path) for path in args]
    except OSError as err:
        if errno.EEXIST != err.errno:
            raise
    return

def chown_directory(path, logger):
    uid = getpwnam('mysql').pw_uid
    gid = getpwnam('mysql').pw_gid
    for root, dirs, files in os.walk(path):
        for d in dirs:
           d1r = os.path.join(root, d)
           logger.debug('fix permition {0}'.format(d1r))
           os.chmod(d1r, 0755)
           os.chown(d1r, uid, gid)
        for f in files:
           f1le = os.path.join(root, f)
           logger.debug('fix permition {0}'.format(f1le))
           os.chmod(f1le, 0644)
           os.chown(f1le, uid, gid)
    if os.path.isdir(path):
        os.chmod(path, 0755)
        os.chown(path, uid, gid)
    return

def restore_backup(config):
    logger = config.get('logger')
    bucket = config.get('name_bucket')
    backups = config.get('name_backups')
    code = 0

    if config['walg']:
        code = restore_walg(config)
    else:
        code = download_backup(config)
        if code == 0:
            code = prepare_backup(config)
    if code == 0:
        logger.warning('restore backups success: {0}:{1}/{2}'.format(config.get('address'), bucket, backups))
    else:
        logger.warning('restore backups failed: {0}:{1}/{2}'.format(config.get('address'), bucket, backups))
    return

def check_empty_directory(path):
    total_size = 0
    if os.path.exists(path):
        for dirpath, dirname, filenames in os.walk(path):
            for filename in filenames:
                f1le = os.path.join(dirpath, filename)
                total_size += os.path.getsize(f1le)
    return total_size

def restore_walg(config):
    backups = config.get('name_backups')
    name_bucket = config.get('name_bucket')
    chunk_size = config.get('chunk_size', CHUNK_SIZE)
    logger = config.get('logger')
    module = '[{0}/{1}]'.format(__name__, restore_walg.__name__)
    bucket = mds.get_bucket(**config)
    exitcode = 0
    for keyname in backups:
        rgxp = re.compile('([a-zA-Z]+[0-9]*)_?(\S+)?/basebackups.*/(.*Z)/*')
        walg_backup = rgxp.findall(keyname)
        if len(walg_backup) == 0:
            continue
        instance, _, backup_name = walg_backup[0]
        home = config.get('mysql_directory') if config.get('mysql_directory') else RESTORE_DIR.format(instance=instance)
        mysql_data, mysql_bin, mysql_relay = [os.path.join(home, i) for i in ['data', 'bin-logs', 'relay-logs']]
        logger.warning('{0} start repair backp walg {1}:{2}/{3}'.format(module, config.get('address'), name_bucket, keyname))

        try:
            if check_empty_directory(mysql_data) != 0:
                if config.get('remove_data', False):
                    shutil.rmtree(mysql_data, ignore_errors=True)
                else:
                    msg = '{0} directory {1} not empty. Skip it!'.format(module, mysql_data)
                    raise ValueError(msg)
            create_directory(mysql_data, mysql_bin, mysql_relay)
            logger.warning('{0} download to {1}'.format(module, mysql_data))
            fileno_logfile = config.get('logger_ch').stream.fileno()

            walcnf = copy.copy(config)
            walcnf['instance'] = instance
            walcnf['backup_name'] = backup_name
            walcnf['mysql_data'] = mysql_data
            walcnf['chunk_size'] = walcnf['chunk_size'] if walcnf.has_key('chunk_size') and len(walcnf['chunk_size']) > 0 else chunk_size
            restore_walg_cmd = RESTORE_WALG.format(**walcnf)
            walenv = dict([(k, v.format(**walcnf)) for k, v in WALENV.items()])
            p1 = Popen(restore_walg_cmd, cwd=mysql_data, shell=True, bufsize=0,
                       stderr=fileno_logfile, stdout=fileno_logfile,
                       preexec_fn=os.setsid, env=walenv)
            save_hash(config, "PROCESS", "start restore {0}".format(keyname))

            while True:
                rcode1 = p1.poll()
                time.sleep(10)
                if rcode1 is not None and rcode1 != 0:
                    raise RestoreBackupError('{0} error command {1} / env {2}'.format(module, restore_walg_cmd, walenv))
                if rcode1 is not None and rcode1 == 0:
                    break
            p1.wait()

            time.sleep(10)
            #отлавливаем ситуацию когда обывается соединение
            xtfile = os.path.join(mysql_data, 'xtrabackup_info')
            if not os.path.exists(xtfile):
                raise ValueError("backup not full: not found {0}".format(xtfile))
            logger.warning('restore backup walg success: {0}:{1}/{2}'.format(config.get('address'), bucket, keyname))
            save_hash(config, "SUCCESS", "complete restore walg {0}".format(keyname))
        except Exception as err:
            logger.critical(err)
            logger.critical('restore backup walg failed: {0}:{1}/{2}'.format(config.get('address'), bucket, keyname))
            save_hash(config, "FAILED", "failed restore walg {0}".format(keyname))
            exitcode = 1
    return exitcode


def download_backup(config):
    if config['walg']:
        raise RestoreBackupError("dont support download for walg. Use repair")
    backups = config.get('name_backups')
    name_bucket = config.get('name_bucket')
    chunk_size = config.get('chunk_size', CHUNK_SIZE)
    logger = config.get('logger')
    module = '[{0}/{1}]'.format(__name__, download_backup.__name__)
    bucket = mds.get_bucket(**config)
    exitcode = 0
    for keyname in backups:
        instance = keyname.split('_')[0]
        home = config.get('mysql_directory') if config.get('mysql_directory') else RESTORE_DIR.format(instance=instance)
        mysql_data, mysql_bin, mysql_relay = [os.path.join(home, i) for i in ['data', 'bin-logs', 'relay-logs']]
        logger.warning('{0} start download {1}:{2}/{3}'.format(module, config.get('address'), name_bucket, keyname))
        try:
            if check_empty_directory(mysql_data) != 0:
                if config.get('remove_data', False):
                    shutil.rmtree(mysql_data, ignore_errors=True)
                else:
                    msg = '{0} directory {1} not empty. Skip it!'.format(module, mysql_data)
                    raise ValueError(msg)
            create_directory(mysql_data, mysql_bin, mysql_relay)
            logger.warning('{0} download to {1}'.format(module, mysql_data))
            fileno_logfile = config.get('logger_ch').stream.fileno()
        #try:
            #load backup and unzip
            save_hash(config, "PROCESS", "start download {0}".format(keyname))
            p1 = Popen(PBZIP2_UNPACK, cwd=mysql_data, shell=True, bufsize=0, stdin=PIPE, stderr=fileno_logfile, stdout=PIPE)
            p2 = Popen(TAR_UNPACK, cwd=mysql_data, shell=True, bufsize=0, stdin=p1.stdout, stderr=fileno_logfile, stdout=fileno_logfile)
            key = bucket.get_key(keyname)
            keysize = key.size
            filesize = 0
            key.open_read(override_num_retries=10)
            while True:
                data = key.read(chunk_size)
                filesize += len(data)
                rcode1 = p1.poll()
                rcode2 = p2.poll()
                if rcode1 is not None and rcode1 != 0:
                    raise RestoreBackupError('{0} error unzip {1} /{2}'.format(module, PBZIP2_UNPACK, mysql_data))
                if rcode2 is not None and rcode2 != 0:
                    raise RestoreBackupError('{0} error unzip {1} /{2}'.format(module, TAR_UNPACK, mysql_data))
                if key.resp.status != 200:
                    logger.error('{0} error read bucket. Status code {1}. \nData: {2}'.format(module, key.resp.status, data[:20]))
                    continue
                if len(data) != 0: p1.stdin.write(data)
                if filesize == keysize:
                    p1.stdin.close()
                    break
            # innobackupex restore
            p1.wait()
            p2.wait()
            key.close()
            time.sleep(10)
            #отлавливаем ситуацию когда обывается соединение
            xtfile = os.path.join(mysqldata, 'xtrabackup_info')
            if not os.path.exists(xtfile):
                raise ValueError("backup not full: not found {0}".format(xtfile))
            logger.warning('download backup success: {0}:{1}/{2}'.format(config.get('address'), bucket, keyname))
            save_hash(config, "SUCCESS", "complete download {0}".format(keyname))
        except Exception as err:
            logger.critical(err)
            logger.critical('download backup failed: {0}:{1}/{2}'.format(config.get('address'), bucket, keyname))
            save_hash(config, "FAILED", "error download {0}".format(keyname))
            exitcode = 1
    return exitcode


def get_free_memory():
    free = 0
    rgxp = re.compile('.*\s+(\d+)\s+.*')
    for line in open('/proc/meminfo', 'r').readlines():
        if line.find('MemFree') > -1 or line.find('Buffers') > -1 or line.find('Cached') > -1:
            size = rgxp.findall(line)
            if len(size) > 0:
                free += int(size[0]) * 1024
    return free


def prepare_backup(config):
    if config['walg']:
        raise RestoreBackupError("dont support download for wal-g. Use repair")
    backups = config.get('name_backups')
    logger = config.get('logger')
    module = '[{0}/{1}]'.format(__name__, prepare_backup.__name__)
    exitcode = 0
    for keyname in backups:
        try:
            while True:
                free_memory = get_free_memory()
                if free_memory > sizeof_int(config.get('memory_prepare')): break
                msg = "no free memory for prepare: need {0} current {1}".format(sizeof_fmt(MEMORY_SIZE_PREPARE), sizeof_fmt(free_memory))
                save_hash(config, "DELAY", msg)
                logger.warning(msg)
                time.sleep(30)
            save_hash(config, "PROCESS", "start prepare {0}".format(keyname))
            instance = keyname.split('_')[0]
            home = config.get('mysql_directory') if config.get('mysql_directory') else RESTORE_DIR.format(instance=instance)
            mysql_data = os.path.join(home, 'data')
            fileno_logfile = config.get('logger_ch').stream.fileno()
            restore_cmd = RESTORE_CMD.format(config.get('memory_prepare'))
            p3 = Popen(restore_cmd, cwd=mysql_data, shell=True, bufsize=0, stderr=fileno_logfile, stdout=fileno_logfile)
            p3.wait()
            returncode = p3.poll()
            if returncode is not None and returncode != 0:
                raise RestoreBackupError('{0} error recovery {1} /{2}'.format(module, RESTORE_CMD, mysql_data))
                #fix permitions
            chown_directory(home, logger)
            os.remove(os.path.join(mysql_data, 'xtrabackup_logfile'))
            logger.warning('{0} prepare backup success: {1}'.format(module, mysql_data))
            save_hash(config, "SUCCESS", "complete prepare {0}".format(keyname))
        except Exception as err:
            logger.critical(err)
            logger.critical('{0} prepare backup failed: {1}'.format(module, keyname))
            save_hash(config, "FAILED", "error prepare {0}".format(keyname))
    return exitcode


def remove_backup(config):
    backups_name = config.get('name_backups')
    name_bucket = config.get('name_bucket')
    logger = config.get('logger')
    module = '[{0}/{1}]'.format(__name__, remove_backup.__name__)
    bucket = mds.get_bucket(**config)
    backups_mds = mds.list_keys(**config)
    backups_delete = list(set(backups_mds) & set(backups_name))

    if len(backups_delete) == 0:
        logger.critical('{0} not found backups {3} in {1}:{2}'.format(module, config.get('address'), name_bucket, backups_name))
        return

    for keyname in backups_delete:
        try:
            logger.warning('{0} start remove {1}:{2}/{3}'.format(module, config.get('address'), name_bucket, keyname))
            key = bucket.get_key(keyname)
            if config.get('simulate'):
                logger.warning('{0} simulate remove {1}:{2}/{3}'.format(module, config.get('address'), name_bucket, keyname))
            else:
                key.delete()
                logger.warning('{0} success remove {1}:{2}/{3}'.format(module, config.get('address'), name_bucket, keyname))
        except Exception as err:
            logger.critical('error remove backup {1}:{2}/{3}: {4}'.format(module, config.get('address'), name_bucket, keyname, str(err)))
    return


def clean_backup(config):
    name_bucket = config.get('name_bucket')
    logger = config.get('logger')
    module = '[{0}/{1}]'.format(__name__, clean_backup.__name__)
    bucket = mds.get_bucket(**config)
    backups_mds = mds.list_keys(**config)
    backups_unique = {}
    backups_duplicates = []
    for keyname in backups_mds:
        if keyname.find("basebackups") > -1:
            rgxp = re.compile('([a-zA-Z]+[0-9]*)_?(\S+)?/basebackups.*_(\d+)T(\d+)Z/*')
            walg_backup = rgxp.findall(keyname)
            if len(walg_backup) == 0:
                continue
            instance, _, sdate, _ = walg_backup[0]
        else:
            instance, sdate, _ = keyname.split('_', 2)
        if not backups_unique.has_key(instance):
            backups_unique[instance] = {}
        if backups_unique[instance].has_key(sdate):
            backups_duplicates.append(keyname)
        else:
            backups_unique[instance][sdate] = keyname

    deleted_list = backups_duplicates if config.get('remove_duplicates') else []
    if deleted_list:
        logger.critical('{0} removed duplicates: {1}'.format(module, deleted_list))
    today = datetime.date.today()
    for instance in backups_unique:
        if len(backups_unique[instance]) <= int(ROTATE_DAYS):
            if config['force']:
                logger.critical('{0} force clean backups {1}. Count unique backups per day <= {2}. We are deleted old backups anyway'.format(module, instance, ROTATE_DAYS))
            else:
                logger.critical('{0} skip clean backups {1}. Count unique backups per day <= {2}'.format(module, instance, ROTATE_DAYS))
                continue
        for sdate in backups_unique[instance]:
            backup_date = datetime.datetime.strptime(sdate, '%Y%m%d').date()
            delta = today - backup_date
            if int(delta.days) > int(ROTATE_DAYS):
                deleted_list.append(backups_unique[instance][sdate])
                logger.critical('{0} removed old backups {1}_{2}'.format(module, instance, sdate))
    config['name_backups'] = deleted_list
    logger.debug('{0} remove backups: {1}'.format(module, config['name_backups']))
    remove_backup(config)
    return
