#!/usr/bin/perl -w

# $Id$

=head1 NAME

    generate-mysql-grant-statements.pl -- генерация запросов "grant <priv> on <table> to <user>@<host>" по относительно простому конфигу

=head1 DESCRIPTION

    generate-mysql-grant-statements.pl --db-host ppctest-ts-mysql -i ppcdata1 -c /etc/mysql-grants/direct.yaml| mysql -uroot -S /var/run/mysqld.ppcdata1/mysqld.sock

    Читает файл с правилами для генерации грантов и генерирует sql-запросы для указанного хоста и инстанса mysql

    -c, --conf <file> 
        файл с правилами для генерации

    -i, --instance ppcdata1|ppcdict|...
        инстанс, для которого надо сгенерировать гранты

    --db-host ppctest-ts-mysql
        хост, для которого надо сгенерировать гранты

    --all
        генерировать правила для всех хостов и инстансов
        Полезно, чтобы сравнить правила до и после какой-нибудь доработки

=head1 FORMAT

Конфиг -- файл с yaml-хешом.
На верхнем уровне -- ключи definitions и rules.

В definitions определяются именованные наборы (set) прав, хостов, инстансов. На них можно потом ссылаться из правил. 

В полях db_host и client_host можно ссылаться на кондукторные группы: conductor/<имя группы>

Правила -- массив хешей, пример правила:

  - name: dev rbac
    db_host: 
      - set/direct_dev_db 
      - ppctest-ts1-mysql.yandex.ru
      - conductor/direct_nd_dev_db
    instance: rbac2
    grant:
      - SUPER
      - SELECT
    on: '*.*'
    user: rbac
    client_host: 
      - set/direct_ppcdev 
      - ppcdev2.yandex.ru
      - conductor/direct_nd_dev
    password: '*BA72E...'
    password: '*BA72E...'  или так: literal_password: 'sales'
    
Пароль может быть задан либо хешом (поле password), либо своим буквальным зачением (поле literal_password)

=cut

use strict;
use warnings;

use Getopt::Long;
use Data::Dumper;
use YAML;
use List::MoreUtils qw/ uniq /;

use Yandex::Conductor;

use utf8;
use open ':std' => ':utf8';

run() unless caller();

sub run
{
    my %O = %{parse_options()};
    my $conf = YAML::LoadFile($O{conf_file});

    prepare_rules($conf); 

    if ($O{all}){
        my @db_hosts = uniq map {@{$_->{db_host}}} @{$conf->{rules}}; 
        @db_hosts = sort @db_hosts;
        my @instances = uniq map {@{$_->{instance}}} @{$conf->{rules}}; 
        @instances = sort @instances;
        for my $h (@db_hosts){
            for my $ins (@instances){
                print "### $h $ins\n";
                my $statements = generate_statements($conf->{rules}, $h, $ins);
                print join "", map {"$_\n"} (@$statements);
                print "\n\n";
            }
        }
    } else {
        my $statements = generate_statements($conf->{rules}, $O{db_host}, $O{instance});
        print join "", map {"$_\n"} (@$statements);
    }

    exit; 
}

=head2 prepare_rules

    Обрабатываются поля 
        db_host
        instance
        grant
        client_host

    1. Если скаляр -- делаем ссылку на массив
    2. Если какой-то эл-т массива -- 'set/...', 
       заменяем на соотв. список из definitions
    3. Если db_host или client_host -- 'conductor/nnn', 
       заменяем на соотв. кондукторную группу

=cut
sub prepare_rules
{
    my ($conf) = @_;

    for my $r (@{$conf->{rules}}){
        for my $f (qw/db_host instance grant client_host/){
            $r->{$f} = [$r->{$f}] unless ref $r->{$f} eq 'ARRAY';
            $r->{$f} = [map {$_=~m!^set/(.*)! ? @{$conf->{definitions}->{$f}->{$1}} : $_} @{$r->{$f}}];
        }
        for my $f (qw/db_host client_host/){
            $r->{$f} = [map {$_=~m!^conductor/(.*)! ? @{conductor_groups2hosts($1)} : $_} @{$r->{$f}}];
        }
    }

    return '';
}


sub generate_statements
{
    my ($rules, $db_host, $instance) = @_;

    my @s;

    for my $rule (@$rules){
        next unless grep { $_ eq $db_host  } @{$rule->{db_host}};
        next unless grep { $_ eq $instance } @{$rule->{instance}};
        for my $client_host (@{$rule->{client_host}}){
            my $priv = join( ', ', @{$rule->{grant}});
            my $identified = "";
            if ($rule->{use_pre_57_grant_syntax}) {
                # для rbac, где 5.5 и IDENTIFIED WITH 'mysql_native_password' AS ...  не работает, хотя и есть в документации
                if ($rule->{password}){
                    $identified = " IDENTIFIED BY PASSWORD '$rule->{password}'";
                } elsif ( $rule->{literal_password} ){
                    $identified = " IDENTIFIED BY '$rule->{literal_password}'";
                }
            } else {
                if ($rule->{password}){
                    $identified = " IDENTIFIED WITH 'mysql_native_password' AS '$rule->{password}'";
                } elsif ( $rule->{literal_password} ){
                    $identified = " IDENTIFIED WITH 'mysql_native_password' BY '$rule->{literal_password}'";
                }
            }
            push @s, "GRANT $priv ON $rule->{on} TO '$rule->{user}'\@'$client_host'$identified;";
        }
    }

    push @s, "flush privileges;";

    return \@s;
}


sub parse_options
{
    my %O;
    GetOptions(
        "h|help"       => sub {system("podselect -section NAME -section DESCRIPTION $0 | pod2text-utf8 >&2"); exit 0;},
        "all"          => \$O{all},
        "db-host=s"       => \$O{db_host},
        "i|instance=s" => \$O{instance},
        'c|conf=s'    => \$O{conf_file},
    ) || die "can't parse options";

    die "missing --conf parameter" unless $O{conf_file};
    if (!$O{all}){
        die "missing --db-host parameter" unless $O{db_host};
        die "missing --instance parameter" unless $O{instance};
    }

    return \%O;
}
