#!/usr/bin/env python2
# -*- encoding: utf-8 -*-

import argparse
import os
import logging
import json
import time
import multiprocessing as mp
import sys
import MySQLdb

from copy import deepcopy

#CONFIG_MYSQL = ['/etc/yandex-direct/db-config.json']
CONFIG_MYSQL = ['/root/test1.json']
EXCLUDE_INSTANCES = ['rbac2', 'ppclog']

def startLoging(level, log_file=None):
    logger = logging.getLogger('steam logs to console')
    logger.setLevel(level=getattr(logging, level))
    # create console handler and set level to LEVEL
    ch = logging.StreamHandler() if log_file is None else logging.FileHandler(
        log_file, mode='a')
    ch.setLevel(level=getattr(logging, level))
    # create formatter
    formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
    # add formatter to ch
    ch.setFormatter(formatter)
    # add ch to logger
    logger.addHandler(ch)
    return logger

def parseInstances(raw, result=list(), saved=dict()):
    '''Принимает на вход словарь, ищет вхождения searching_key и выводит список с набором хэшей.
       Например:
           [ {u'port': 9451, u'instance': u'ppcdata11', u'host': u'host11.yandex.ru', u'user': u'user1', u'pass': {u'file': u'/etc/tokens/mysql'}},
             {u'port': 9450, u'instance': u'ppcdata10', u'host': u'host10.yandex.ru', u'user': u'user1', u'pass': {u'file': u'/etc/tokens/mysql'}}
           ]
    '''
    saved = saved.copy()
    searching_keys = ['instance', 'host', 'port', 'pass', 'user']
    excluding_keys = ['extra_users']
    dict_list = []
    good_keys = set(raw.keys())-set(excluding_keys)
    for value in raw:
        if value in searching_keys:
            saved[value] = raw[value]
        elif isinstance(raw[value], dict):
            dict_list.append(value)

    if len(saved) < len(searching_keys):
        for i in dict_list:
            parseInstances(raw[i], result, saved)
    else:
        logger.debug('[parseInstances] {0}'.format(saved))
        result.append(saved)
    return result

def readFile(f1le):
    try:
        passwd = open(f1le, 'r').read().strip()
    except Exception as err:
        passwd = ''
        logger.critical(err)
    return passwd

def readPasswd(config):
    '''Принимает на вход список словарей, находит в ней структуры вида {'pass': {'file': '/etc/secret'}},
       пытается прочитать и перезаписывает ключ 'pass'. Если произошла ошибка чтения файла, подставляется
       пустая строка.
    '''
    pass_files = [value['pass']['file'] for value in config if isinstance(value['pass'], dict) and
                                                                         value['pass'].has_key('file')]
    pass_files = tuple(set(pass_files))
    passwd_list = dict([(i, readFile(i)) for i in pass_files])
    logger.debug('[readPasswd] {0}'.format(passwd_list))
    for i in config:
        if isinstance(i['pass'], dict) and i['pass'].has_key('file'):
            i['pass'] = passwd_list[i['pass']['file']]
    return

def readConfig(config_mysql, cnf_data, conn_saved):
    '''Принимает на вход путь до конфига mysql, на выходе список с словарями. В каждом указывается порт,
       инстанс, хост, юзер и пароль. Пример вывода указан в описании parseInstances().
    '''
    mtime_current = int(os.stat(config_mysql).st_mtime) if os.path.exists(config_mysql) else 0
    if cnf_data.has_key(config_mysql):
        mtime_last = cnf_data[config_mysql]['mtime']
        if mtime_last == mtime_current:
            logger.debug('config dont changes. Using saving')
            return

    for i in conn_saved:
        try:
            logger.debug('[readConfig] close connect {0}'.format(i))
            conn_saved[i].close()
        except Exception as err:
            logger.error('[readConfig] {0}'.format(err))

    try:
        insts = []
        cnf_data[config_mysql] = {}
        logger.debug('read new config')
        with open(config_mysql, 'rb') as fd:
            raw = json.load(fd)
        parseInstances(raw, insts)
        insts = [i for i in insts if i['instance'] not in EXCLUDE_INSTANCES]
        readPasswd(insts)
        cnf_data[config_mysql]['data'] = insts
        cnf_data[config_mysql]['mtime'] = mtime_current
    except Exception as err:
        logger.critical('[readConfig] {0}'.format(err))
    return

def runQuery(queue):
    results = ()
    while 1:
        try:
            requests, cnf, db, connects = queue.get()
            host = cnf['host'][0] if isinstance(cnf['host'], list) else cnf['host']
            connect_id = host + '_' + str(cnf['port'])
            logger.info('[runQuery] start request {0}'.format(connect_id))
            try:
                reconnect = connects[connect_id].ping() if connects.has_key(connect_id) else 1
            except Exception as err:
                logger.error('[reconnect] {0}'.format(err))
                reconnect = 1

            if reconnect:
                logger.info('[reconnect] create new connection')
                db = MySQLdb.connect(user=cnf['user'], passwd=cnf['pass'],
                                    host=host, port=int(cnf['port']), db=db, connect_timeout=10)
                db.autocommit(True)
                connects[connect_id] = db
            else:
                logger.info('[reconnect] use current connection')
                db = connects.get(connect_id)
            cur = db.cursor()
            for req in requests:
                try:
                    cur.execute("SET SESSION lock_wait_timeout=10")
                    cur.execute(req)
                    result = cur.fetchall()
                except Exception as err:
                    msg = '{0} for config {1}/{2}/{3}'.format(str(err), cnf['instance'], host, cnf['port'])
                    logger.critical(msg)
            cur.close()
            logger.debug('[runQuery] {1}:{2} {0}'.format(results, host, cnf['port']))
        except Exception as err:
            logger.critical('[runQuery] {0}: {1}'.format(connect_id, err))
        finally:
            queue.task_done()
    return

def runQueryDBs(requests, cnf_name, db='', conn_saved={}, cnf_data={}):
    allprocs = []
    q = mp.JoinableQueue()
    try:
        for i in range(5):
            proc = mp.Process(target=runQuery, args=(q,))
            proc.start()
            allprocs.append(proc)
        while 1:
            readConfig(cnf_name, cnf_data, conn_saved)
            for i in cnf_data[cnf_name]['data']:
                q.put((requests, i, db, conn_saved))
            time.sleep(1)

    except Exception as err:
        logger.error('[runQueryDBs] {0}'.format(err))
        raise
    finally:
        for proc in allprocs:
            proc.terminate()
            logger.critical("{0} process terminated".format(proc))

def initialize(configs):
    requests = ['CREATE DATABASE IF NOT EXISTS heartbeat']
    [runQueryDBs(requests, cnf) for cnf in configs]
    # для java синхронизаторов оказалось важно знать параметр USE <db> в запросах типа CREATE
    requests = ['CREATE TABLE IF NOT EXISTS heartbeat.heartbeat (timestamp int, hostname varchar(255) NOT NULL, PRIMARY KEY (hostname)) ENGINE=InnoDB DEFAULT CHARSET=utf8']
    [runQueryDBs(requests, cnf, db='heartbeat') for cnf in configs]


def run(configs):
        conn_saved, cnf_saved = {}, {}
        requests = ['REPLACE INTO heartbeat.heartbeat SET hostname="master", timestamp=UNIX_TIMESTAMP()']
        [runQueryDBs(requests, cnf, db='heartbeat', conn_saved=conn_saved, cnf_data=cnf_saved) for cnf in configs]
        time.sleep(1)

if __name__ == '__main__':
    m = argparse.ArgumentParser(description='active cheack mysql replication lag')
    m.add_argument('--debug', '-d', action='store_true',
                   help="debug mode")
    m.add_argument('--initialize', action='store_true',
                   help="создать структуру db/tables в БД")
    m.add_argument('--configs', '-c', nargs='*', type=str, action='store',
                   default=CONFIG_MYSQL,
                   help="список конфигов json для коннекта в БД")
    m.add_argument('--no-fork', action='store_true', dest="nofork",
                   help="запустить программу в основном процессе, а не дочерним")
    m.add_argument('--log-file', '-l', action='store', type=str, default=None,
                   dest="logfile", help="путь к файлу для логов")
    options = m.parse_args()
    global logger
    level = 'DEBUG' if options.debug else 'INFO'
    logger = startLoging(level, options.logfile)
    if options.initialize:
        initialize(options.configs)
    else:
        if options.nofork:
            run(options.configs)
        else:
            fpid = os.fork()
            if fpid == 0:
                run(options.configs)
            else:
                sys.exit(0)
