#!/usr/bin/env python
# -*- coding: utf8 -*-

description = """
    Копирование реплики mysql

    Тестировался и отлаживался на ppcdata-х, к другим Direct-style базам применение надо аккуратно проверять.

    В конце напишет одну или две команды, которые останется выполнить (смотреть TODO)

    Примеры:
    На машину с одним инстансом привезти данные с мастера:
    dt-replica-clone.py -s ppcdata15-01f.yandex.ru

    Если инстансов несколько -- надо указать нужный через -i
    dt-replica-clone.py -s ppcdata15-01f.yandex.ru -i ppcdata15

    Если источник -- реплика, можно это указать через -t:
    dt-replica-clone.py -s ppcdata15-01f.yandex.ru -t replica

    Может отправлять Ямб-сообщение "успешно завершился", логин адресата передать через -l:
    dt-replica-clone.py -s ppcdata15-01f.yandex.ru -l lena-san

    Можно ничего не делать, а посмотреть список действий:
    dt-replica-clone.py -s ppcdata15-01f.yandex.ru -n

    Можно пропустить некторые шаги:
    dt-replica-clone.py -s ppcdata15-01f.yandex.ru -x apt-get-update

    Можно все вместе:
    dt-replica-clone.py -s ppcdata15-01f.yandex.ru -i ppcdata15 -l lena-san -x apt-get-update -v -t replica

"""

import sys
sys.path.insert(0, '/opt/direct-py/startrek-python-client-sni-fix' )

from subprocess import check_call, check_output
import re
import argparse
import os
import pwd
import requests
import yaml
import time


from requests.packages.urllib3.exceptions import InsecureRequestWarning
requests.packages.urllib3.disable_warnings(InsecureRequestWarning)

"""
TODO
  - объединить локальное и удаленное выполнение ('localhost')
  + проверить gtid
  - проверить, источник -- мастер или реплика
  + stop
  + rm
  + mkdir
  + возможность пропускать шаги
  + возможность начинать не с начала или заканчивать раньше
  - ?? проверять, что текущая реплика -- не кольцевая?
"""

actions = {}
order_of_actions = []
required_packages = [ 'mbuffer', 'percona-xtrabackup-24', 'qpress', 'yandex-du-lm' ]
# global state; данные, передающиеся между шагами (self, если бы это был класс)
GS = {}

################################################################################

def init_actions():
    global actions
    global order_of_actions
    # в этом списке все действия строго в том порядке, как должны выполняться
    # этот список удобно и безопасно поддерживать при доработках
    # во время работы скрипта удобнее дикт и отдельно список-порядок
    # их делаем ниже из этого массива
    ordered_actions = [
            {
                'name': 'check-tmux',
                'code': do_check_tmux,
                },
            {
                'name': 'sleep',
                'code': do_sleep,
                },
            {
                'name': 'check-find-mysql-instance',
                'code': do_check_find_mysql_instance,
                },
            {
                'name': 'apt-get-update',
                'code': do_apt_get_update,
                },
            {
                'name': 'install-packages',
                'code': do_install_packages,
                },
            {
                'name': 'compare-packages',
                'code': do_compare_packages,
                },
            {
                'name': 'check-disk-space',
                'code': do_check_disk_space,
                },
            {
                'name': 'check-gtid',
                'code': do_check_gtid,
                },
            {
                'name': 'get-replication-params',
                'code': do_get_replication_params,
                },
            {
                'name': 'stop-mysql',
                'code': do_stop_mysql,
                },
            {
                'name': 'remove-old-data',
                'code': do_remove_old_data,
                },
            {
                'name': 'check-mysql-datadir',
                'code': do_check_mysql_datadir
            },
            {
                'name': 'copy-data',
                'code': do_copy_data,
                },
            {
                'name': 'innobackupex-decompress',
                'code': do_innobackupex_decompress,
                },
            {
                'name': 'remove-qp-files',
                'code': do_remove_qp_files,
                },
            {
                'name': 'innobackupex-apply-log',
                'code': do_innobackupex_apply_log,
                },
            {
                'name': 'chown-data-dir',
                'code': do_chown_data_dir,
                },
            {
                'name': 'fix-uuid',
                'code': do_fix_uuid,
                },
            {
                'name': 'mysql-start-skip-slave',
                'code': do_mysql_start_skip_slave,
                },
            {
                'name': 'mysql-set-position',
                'code': do_mysql_set_position,
                },
            {
                'name': 'mysql-change-master',
                'code': do_mysql_change_master,
                },
            {
                'name': 'remove-xtrabackup-files',
                'code': do_remove_xtrabackup_files,
                },
#            {
#                'name': '',
#                'code': ,
#                },
            ]
            # здесь перерабатываем список в дикт "название -- свойства"
            # и отдельно заполняем список названий действий в правильном порядке
    for a in ordered_actions:
        name = a.pop('name')
        actions[name] = a
        order_of_actions.append(name)
    return


################################################################################
def die(message=''):
    sys.stderr.write("%s\n" % message)
    exit(1)


def send_message(to, msg):
    if not isinstance(to, basestring) or not re.match(r'^[a-z0-9\-_]+$', to):
        die("bad login for notifications: '%s'" % to);
    payload = {'to': to, 'msg': msg}
    r = requests.get('https://direct-dev.yandex-team.ru/yamb_send', params=payload, verify=False, timeout=20)
    return

def get_ssh_login():
    # при запуске из-под sudo предполагаем, что у пользователя нет прав залогиниться root'ом
    if 'SUDO_USER' in os.environ and os.environ['SUDO_USER']:
        return os.environ['SUDO_USER']
    else:
        return pwd.getpwuid(os.getuid()).pw_name

# выполнить на удаленной машине команды, вернуть ничего
def remote_exec(host, script):
    ssh_login = get_ssh_login()
    if ssh_login != 'root':
        script = 'sudo ' + script
    cmd="""ssh -q %s@%s <<'EOF' "bash -s"
%s
EOF
""" % (ssh_login, host, script)
    if GS['verbose']:
        print "going to to run:\n%s\n" % (cmd)
    check_call(cmd, shell=True)
    return


# выполнить на удаленной машине команды, вернуть stdout
def remote_qx(host, script):
    ssh_login = get_ssh_login()
    if ssh_login != 'root':
        script = 'sudo ' + script
    cmd="""ssh -q %s@%s <<'EOF' "bash -s"
%s
EOF
""" % (ssh_login, host, script)
    if GS['verbose']:
        print "going to to run:\n%s\n" % (cmd)
    output = check_output(cmd, shell=True)
    return output.rstrip()


def local_exec(script):
    if GS['verbose']:
        print "going to to run:\n%s\n" % (script)
    check_call(script, shell=True)
    return


def local_qx(script):
    if GS['verbose']:
        print "going to to run:\n%s\n" % (script)
    output = check_output(script, shell=True)
    return output.rstrip()
################################################################################


def parse_options():
    global GS
    parser = argparse.ArgumentParser(add_help=False)
    parser.add_argument("-s", "--source", dest="source_host", help="source host", type=str)
    parser.add_argument("-t", "--source-type", dest="source_type", help="source type: master|replica", type=str, choices=['master', 'replica'], default='master')
    parser.add_argument("-b", "--begin-with", dest="action_start", help="action to begin", type=str)
    parser.add_argument("-e", "--end-with", dest="action_stop", help="action to end", type=str)
    parser.add_argument("-i", "--instance", dest="mysql_instance", help="mysql instance", type=str)
    parser.add_argument("-x", "--skip", dest="skip", default=[], help="actions to skip", type=str, action='append')
    parser.add_argument("-h", "--help", dest="help", help="справка", action="store_true")
    parser.add_argument("-v", "--verbose", dest="verbose", help="verbose mode", action="store_true")
    parser.add_argument("-n", "--dry-run", dest="dry_run", help="dry run", action="store_true")
    parser.add_argument("-l", "--notify-login", dest="notify_login", help="login to send notifications to", type=str)
    opts, extra = parser.parse_known_args()

    if opts.help:
        print description
        print parser.format_help()
        exit(0)

    if len(extra) > 0:
        die("unexpected parameters, stop")

    GS['verbose'] = opts.verbose
    GS['instance'] = opts.mysql_instance
    GS['source_type'] = opts.source_type

    # валидация source_host
    if not opts.source_host:
        die("expecting source host (-s ...)")
    if not re.match(r'^[a-z0-9\-\.]+$', opts.source_host):
        die("bad source host name: %s" % opts.source_host)
    GS['source_host'] = opts.source_host

    # валидируем порядок действий
    start_index=0
    if opts.action_start:
        try:
            start_index = order_of_actions.index(opts.action_start)
        except ValueError:
            die("unknown action in -b (%s)\n\npossible actions: %s" %
                    (opts.action_start, ", ".join(order_of_actions))
                    )
    stop_index = len(order_of_actions)
    if opts.action_stop:
        try:
            stop_index = order_of_actions.index(opts.action_stop)
        except ValueError:
            die("unknown action in -e (%s)\n\npossible actions: %s" %
                    (opts.action_stop, ", ".join(order_of_actions))
                    )
    if stop_index < start_index:
        die("impossible order of action to start and action to stop (%s, %s)\n\npossible actions: %s" %
                (opts.action_start, opts.action_stop, ", ".join(order_of_actions))
                )
    actions_to_do = order_of_actions[start_index:stop_index+1]

    can_skip = ", ".join(actions_to_do)
    for a in opts.skip:
        try:
            actions_to_do.remove(a)
        except ValueError:
            die("impossible or unknown action to skip: %s\n\npossible actions to skip: %s" % (a, can_skip))

    GS['actions_to_do'] = actions_to_do

    return opts


def print_plan():
    print "%s" % GS
    print "Going to do:\n"
    for a in order_of_actions:
        if a in GS['actions_to_do']:
            print " * %s" % a
        else:
            print "[SKIP] %s" % a
    print "\nuse -b, -e and -x to change actions to perform\n"
    print "source_type: %s\n" % GS['source_type']
    return


def validate_instance():
    if not re.match(r'^[a-z0-9_.-]+$', GS['instance']):
        die("bad instance name: '%s'" % GS['instance'])
    return


################################################################################

def do_check_tmux():
    has_tmux = 'TMUX_PANE' in os.environ and os.environ['TMUX_PANE']
    if not has_tmux:
        # Если запущен из-под sudo, переменные окружения не экспортируются в процесс. Проверим ещё родительский
        with open('/proc/%s/environ' % os.getppid()) as f:
            recs = f.read().rstrip().split('\0')
        # запись должна получиться одна
        tmux_pane_recs = [r for r in recs if r.startswith('TMUX_PANE=')]
        if tmux_pane_recs:
            tmux_pane_value = tmux_pane_recs[0].split('=', 1)[1]
            has_tmux = (tmux_pane_value != '')
    if not has_tmux:
        die("ERROR: expecting tmux (checking via $TMUX_PANE for this or parent process)")
    return

def do_sleep():
    time.sleep(10)
    return

def do_check_find_mysql_instance():
    if not 'instance' in GS or not GS['instance']:
        instances = re.split("\n+", local_qx("lm --complete"))
        if len(instances) < 1:
            die("ERROR: can't find any local mysql instances (see lm --complete)")
        elif len(instances) > 1:
            die("ERROR: found multiple local mysql instances (%s), use -i to specify target" % ", ".join(instances))
        GS['instance'] = instances[0]
    validate_instance()
    # если такого инстанса нет -- упадет
    local_exec("ls -d /opt/mysql.%s" % GS['instance'])
    return

def do_check_gtid():
    validate_instance()
    remote_status = remote_qx(GS['source_host'], "lm %s status" % GS['instance'])
    if not re.search(r'gtid_mode: ON', remote_status):
        die("ERROR: can't find 'gtid_mode: ON' in status message\n%s" % remote_status)
    if GS['source_type'] != 'master' and not re.search(r'Slave.*:[0-9]+:1$', remote_status):
        die("ERROR: expecting Auto_Position: 1\n%s" % remote_status)
    return

def do_get_replication_params():
    validate_instance()
    GS['rplcat_user'] = remote_qx(GS['source_host'], "head -n 5 /opt/mysql.%s/data/master.info |tail -n 1" % GS['instance'])
    GS['rplcat_passwd'] = remote_qx(GS['source_host'], "head -n 6 /opt/mysql.%s/data/master.info |tail -n 1" % GS['instance'])
    GS['master_port'] = remote_qx(GS['source_host'], "head -n 7 /opt/mysql.%s/data/master.info |tail -n 1" % GS['instance'])
    print "replication: %s %s %s" % ( GS['rplcat_user'], GS['rplcat_passwd'], GS['master_port'] )
    return

def do_apt_get_update():
    cmd = "apt-get update"
    local_exec(cmd)
    remote_exec(GS['source_host'], cmd)
    return

def do_install_packages():
    cmd = "apt-get install -y %s" % (" ".join(required_packages))
    local_exec(cmd)
    remote_exec(GS['source_host'], cmd)
    return

def do_compare_packages():
    for p in required_packages:
        dpkg_query = "dpkg-query -W -f '${db:Status-Abbrev} ${binary:Package} ${Version}\\n' %s" % p
        local_state = local_qx(dpkg_query)
        remote_state = remote_qx(GS['source_host'], dpkg_query)
        if not re.match("ii ", local_state):
            die("unexpected state of package %s on local host: %s" % (p, local_state))
        if not re.match("ii ", remote_state):
            die("unexpected state of package %s on remote host: %s" % (p, remote_state))
        if local_state != remote_state:
            die("state of package %s differs on local and remote hosts: '%s' vs. '%s'" % (p, local_state, remote_state))
    return

def do_check_disk_space():
    validate_instance()
    source_data_size_str = remote_qx(GS['source_host'], "du -s /opt/mysql.%s/data" % GS['instance'])
    source_data_size_g = int( re.split("\s+", source_data_size_str)[0] )/ 1024 **2

    # закомментированный вариант задумывался как более правильный, но делает странное
    #local_space_str = local_qx("df -k --output=avail /opt/mysql.%s/data | tail -n1" % GS['instance'])
    #local_space_g = int(local_space_str.strip()) / 1024 ** 2
    local_space_str = local_qx("df /opt/mysql.%s/data |head -n 2|tail -n 1" % GS['instance'])
    local_space_g = int(re.split("\s+", local_space_str)[1]) / 1024 ** 2


    if local_space_g < 100:
        die("bad local_space_g")

    if source_data_size_g < 100:
        die("bad source_data_size_g")

    if local_space_g < source_data_size_g * 1.4:
        die("ERROR: local space (%s GB) seems insufficient for data (%s GB) + archive (+25..30%%)" % (local_space_g, source_data_size_g))
    return

def do_stop_mysql():
    validate_instance()
    cmd = "lm %s server-stop -f" % GS['instance']
    local_exec(cmd)
    return

def do_remove_old_data():
    validate_instance()
    path = "/opt/mysql.%s" % GS['instance']
    os.chdir(path)
    cmd = "find . -not -name 'bin-logs' -not -name 'relay-logs' -not -name 'data' -print -delete"
    local_exec(cmd)
    return

def do_check_mysql_datadir():
    validate_instance()
    datadir_path = "/opt/mysql.%s" % GS['instance']
    os.chdir(datadir_path)
    files_str = local_qx("ls -R -1 .")
    files = re.split("\n", files_str)
    if len(files_str) > 400:
        files_str = files_str[:400]+"\n..."
    if not files == [ '.:', 'bin-logs', 'data', 'relay-logs', '', './bin-logs:', '', './data:', '', './relay-logs:', ]:
        die("ERROR: unexpected files/directories in mysql datadir %s:\n%s" % (datadir_path, files_str))
    return

def do_copy_data():
    validate_instance()
    path = "/opt/mysql.%s/data" % GS['instance']
    os.chdir(path)
    if GS['source_type'] == 'replica':
        mbuffer_rate_limit = '-r 80M'
    else:
        mbuffer_rate_limit = '-r 40M'
    ssh_login = get_ssh_login()
    innobackupex_cmd = 'innobackupex'
    if ssh_login != 'root':
        innobackupex_cmd = 'sudo ' + innobackupex_cmd
    cmd = """set -o pipefail; ssh -q %s@%s <<'EOF' "INCTANCE='%s' bash -s" | mbuffer -q | xbstream -x -v
    set -o pipefail
    cd /opt/mysql.$INCTANCE/data
    %s --defaults-file=/etc/mysql/$INCTANCE.cnf --socket=/var/run/mysqld.$INCTANCE/mysqld.sock --user=root --stream=xbstream --compress --compress-threads=8 --parallel=2 --binlog-info=ON --slave-info --safe-slave-backup ./ |mbuffer %s -q
EOF
""" % (ssh_login, GS['source_host'], GS['instance'], innobackupex_cmd, mbuffer_rate_limit)
    local_exec(cmd)

    return

def do_innobackupex_decompress():
    validate_instance()
    path = "/opt/mysql.%s/data" % GS['instance']
    os.chdir(path)
    cmd="innobackupex --decompress --parallel 16 ."
    local_exec(cmd)
    return

def do_remove_qp_files():
    validate_instance()
    path = "/opt/mysql.%s/data" % GS['instance']
    os.chdir(path)
    cmd="find . -name '*.qp' -delete"
    local_exec(cmd)
    return

def do_innobackupex_apply_log():
    validate_instance()
    path = "/opt/mysql.%s/data" % GS['instance']
    os.chdir(path)
    cmd = "innobackupex --apply-log --use-memory=96G ."
    local_exec(cmd)
    return

def do_remove_xtrabackup_files():
    validate_instance()
    path = "/opt/mysql.%s/data" % GS['instance']
    os.chdir(path)
    cmd = "sudo rm -v xtrabackup_* backup-my.cnf"
    print "TODO:\ncd %s && %s" % (path, cmd)
    return
    local_exec(cmd)
    return

def do_chown_data_dir():
    validate_instance()
    cmd = "chown mysql: -R /opt/mysql.%s" % GS['instance']
    local_exec(cmd)
    return

def do_fix_uuid():
    validate_instance()
    cmd="lm %s fix-uuid -f" % GS['instance']
    local_exec(cmd)
    return

def do_mysql_start_skip_slave():
    validate_instance()
    cmd = "lm %s server-start --skip-slave-start -f" % GS['instance']
    local_exec(cmd)
    return

def do_mysql_set_position():
    validate_instance()
    cmd = "lm %s status-gtid" % GS['instance']
    local_exec(cmd)
    cmd = "cat xtrabackup_binlog_info | perl -pe 's/\\n//' | awk '{ print $NF }'"
    gtid_exec = local_qx(cmd)
    # '1a6a311b-c553-427b-c2c7-6f785045bfa1:1,4c1c7ed9-ec88-38f7-73df-52e439b1caa3:1-41643828,bc55e9e9-ed96-3c5b-d974-9fddea303e03:1-888560685,ff8263ea-6f13-8c2f-32d7-683fb2268bd4:1-257883138'
    lm_cmd = "reset master; set global gtid_purged='%s'" % gtid_exec
    cmd = 'lm %s mysql "%s"'% (GS['instance'], lm_cmd)
    if not re.match(r'^[a-z0-9\-\.:,]+$', gtid_exec):
        die("ERROR: very suspicious value for gtid_purged: '%s'\ncheck everything twice, manually perform reset master & set global gtid_purged, and then change-master & remove remaining xtrabackup files" % gtid_exec)
    local_exec(cmd)
    return


def do_mysql_change_master():
    validate_instance()
    suggested_cmd = "lm %s change-master <....yandex.ru>:%s:auto:auto:%s:%s" % (GS['instance'], GS['master_port'], GS['rplcat_user'], GS['rplcat_passwd'])
    print "TODO: %s" % suggested_cmd
    return

################################################################################

def run():
    init_actions()
    opts = parse_options()

    print_plan()

    if opts.dry_run:
        exit(0)

    if opts.notify_login:
        send_message(opts.notify_login, "\n%s: going to start" % os.path.basename(__file__))

    for a in GS['actions_to_do']:
        print "### action: %s" % a
        actions[a]['code']()

    print "SUCCESS"
    if opts.notify_login:
        send_message(opts.notify_login, "\n%s: SUCCESS" % os.path.basename(__file__))
    exit(0)

if __name__ == '__main__':
    run()
