#!/usr/bin/env python
# *-* encoding: utf-8 *-*

''' Программа для сверки различий между базами MySQL ppclogs.
'''

import getpass
import json
import logging
import os
import threading
import MySQLdb
from optparse import OptionParser
from threading import Thread

SERVERS = [ {'host': 'ppclogs03e.yandex.ru', 'port': 3309},
            {'host': 'ppclogs03f.yandex.ru', 'port': 3309},
            {'host': 'ppclogs03i.yandex.ru', 'port': 3309} ]
USER = "adiuser"
PASSWORD = ""
DB = "ppclog"

def getLogSetup(level):
    ''' Функция для задания уровня логирования. Параметр level
        может принимать значения INFO, WARNING, CRITICAL. В выводе
        возвращается интерфейс для логирования.
    '''
    logger = logging.getLogger('strem logs to console')
    logger.setLevel(level=getattr(logging, level))
    # create cinsole header and set level
    ch = logging.StreamHandler()
    ch.setLevel(level=getattr(logging, level))
    # create formatter
    formatter = logging.Formatter('%(asctime)s %(name)s %(levelname)s %(message)s')
    # and formatter to ch
    ch.setFormatter(formatter)
    # add ch to logger
    logger.addHandler(ch)
    return logger

def parseArgs():
    ''' Парсит аргументы программы. Возвращает кортеж с ссылками
        на значения у флагов и обычных параметров, ссылку на
        логгирование. Пример: (opts, args, logger)
    '''
    USAGE = "usage: %prog --help"
    VERSION = '1.0'

    parser = OptionParser(usage=USAGE, version=VERSION)

    parser.add_option( "-d", "--debug",
                       action="store_true",
                       dest="debug", help="debug mode" )
    parser.add_option( "-v", "--verbose",
                       action="store_true",
                       dest="verbose", help="verbose mode")
    parser.add_option( "-p", "--password",
                       action="store", default="",
                       dest="password", help="password")
    parser.add_option( "-u", "--user",
                       action="store", default="",
                       dest="user", help="user")
    parser.add_option( "-b", "--db",
                       action="store", default="ppclog",
                       dest="db", help="database")
    parser.add_option( "-c", "--config",
                       action="store", default="/etc/yandex-direct/db-config.json",
                       dest="config", help="config")

    (opts, args) = parser.parse_args()

    if opts.debug:
        logger = getLogSetup("DEBUG")
    elif opts.verbose:
        logger = getLogSetup("INFO")
    else:
        logger = getLogSetup("CRITICAL")
    return opts, args, logger

def connectDB(host="localhost", user="root", port=None,
              password="", db="mysql", unix_socket=None):
    ''' Функция для коннекта к Mysql. Возвращает курсор и
        ссылку на подключение к БД: (cursor, dbh)
    '''
    logger.debug("connect {0}:{1} {2} {3}".format(host, port, user, db))
    if unix_socket:
        dbh = MySQLdb.connect(host="localhost",
                             user=user,
                             unix_socket=unix_socket,
                             passwd=password,
                             db=db)
    else:
        dbh = MySQLdb.connect(host=host,
                            user=user,
                            port=port,
                            passwd=password,
                            db=db)
    cursor = dbh.cursor()
    return cursor, dbh

def fetchDB(cursor, command):
    ''' Функция для выполнения запросов к MySQL.
        Возаращает список результатов.
    '''
    cursor.execute(command)
    return [ i for i in cursor.fetchall() ]

def closeDB(dbh):
    dbh.close()
    return

def showTablesStatus(cursor, db):
    ''' Функция получает список таблиц. Вывод представляет кортеж:
          database name TABLE_SCHEMA
          table name    TABLE_NAME
          engine        ENGINE
          row format    ROW_FORMAT
          count rows    TABLE_ROWS
          data length   DATA_LENGTH
          create time   CREATE_TIME
          update time   UPDATE_TIME
    '''
    sql = '''SELECT TABLE_SCHEMA, TABLE_NAME, ENGINE,\
             ROW_FORMAT, TABLE_ROWS, DATA_LENGTH, CREATE_TIME,\
             UPDATE_TIME FROM information_schema.TABLES\
             WHERE TABLE_SCHEMA='{0}' AND UPDATE_TIME<=CURDATE()\
          '''.format(db)
    logger.debug("start execute '{0}'".format(sql))
    cursor.execute(sql)
    return [ i for i in cursor.fetchall() ]

def getMetaTables(i, host, port, user, db, password, result):
    ''' Функция идет в БД и получает список метоинформации по таблицам.
        Результат представляет собой словарь вида:
            {'server_name': [ ['name_table1', 'engine1', 'table_size1' ...],
                              ['name_table2', 'engine2', 'table_size2' ...]]
    '''
    logger.debug("run thread {0}".format(i))
    try:
        cursor, dbh = connectDB(host=host, user=user, db=db,
                                port=port, password=password)
    except Exception as err:
        logger.critical("error connect: {0}".format(str(err)))
        return
    try:
        result[host] = showTablesStatus(cursor, db)
        logger.debug(host)
    except Exception as err:
        logger.critical("execute error: {0}".format(str(err)))
    try:
        closeDB(dbh)
    except Exception as err:
        pass
    return

def getTableDicts(meta):
    ''' Создает два словаря: таблицы-колическтво строк,
                             таблицы-тип таблицы.
        Результат представет в виде кортежа двух словарей типа:
        {'table_name': {'server_name1': 'count_rows1',
                        'server_name2': 'count_rows2}}
        {'table_name': {'server_name1': 'engine1',
                        'server_name2': 'engine2}}
    '''
    cnt_tables = dict()
    engine_tables = dict()
    for host in meta:
        logger.debug("parse tables to host:{0}".format(host))
        for data in meta[host]:
            full_name = '.'.join([data[0], data[1]])
            if not cnt_tables.has_key(full_name):
                cnt_tables[full_name] = dict()
            if not engine_tables.has_key(full_name):
                engine_tables[full_name] = dict()
            cnt_tables[full_name][host] = data[4]
            engine_tables[full_name][host] = data[2]
    return cnt_tables, engine_tables

def showDiffTables(cnt_tables, cnt_servers):
    ''' Функция составляет список отсутствующих таблиц, строк, метаинформации.
        Выводит в виде кортежа 3 словаря: (lost_tables, lost_rows, lost_meta).
        Внешний вид каждого словаря:
        {'table_name1': {'server_name1': 'count_rows1',
                         'server_name2': 'count_rows2'},
         'table_name2': {'server_name1': 'count_rows1',
                         'server_name2': 'count_rows2'}}
    '''
    lost_tables, lost_rows, lost_meta = dict(), dict(), dict()
    for name in cnt_tables:
        cnt_rows = cnt_tables[name].values()
        if all(v is None for v in cnt_rows):
            lost_meta.update({name: cnt_tables[name]})
            continue
        if len(cnt_tables[name]) != cnt_servers:
            lost_tables.update({name: cnt_tables[name]})
        if sum(cnt_rows)/(len(cnt_rows)*1.0) != float(cnt_rows[0]):
            lost_rows.update({name: cnt_tables[name]})
    if len(lost_tables)>0:
        logger.debug("lost tables {0}\n".format(lost_tables))
    if len(lost_rows)>0:
        logger.debug("lost rows {0}\n".format(lost_rows))
    if len(lost_meta)>0:
        logger.debug("lost meta {0}\n".format(lost_meta))
    return lost_tables, lost_rows, lost_meta

def foundLost(lost, servers):
    ''' Функция составляет список возможных доноров для копирования дынных.
        Выводит словарь вида:
        {'table_name1': {'donors': list(),
                         'patients': list()}}
    '''
    donor_list = dict()
    for table in lost:
        max_cnt = max(lost[table].values())
        donors = [ i for i in lost[table].keys() if lost[table][i] == max_cnt ]
        patients = list(set(servers)-set(donors))
        donor_list[table] = {'donors': donors, 'patients': patients}
    logger.debug("donor list: {0}\n\n".format(donor_list))
    return donor_list

def printFullRestoreCmd(donor_list, engine_tables):
    restore_myisam, restore_others = dict(), dict()
    def printMYISAM():
        for _patient in restore_myisam:
            print "\n=={0}==".format(_patient)
            for _donor in restore_myisam[_patient]:
                _files = [ "{0}:/opt/mysql.ppclog/{1}.*".format(_donor, _table.replace('.', '/'))
                                    for _table in restore_myisam[_patient][_donor]]
                print "rsync -Pav {0} /opt/mysql.ppclog/ppclog/;".format(" ".join(_files))
    def printOthers():
        for _patient in restore_other:
            print "\n=={0}==".format(_patient)
            for _donor in restore_other[_patient]:
                for _table in restore_other[_patient][_donor]:
                    print "mysqldump -h {0} /tmp/{1};".format(_donor, _table.replace('.', '/'))

    for table in donor_list:
        for patient in donor_list[table]['patients']:
            donor = donor_list[table]['donors'][0]
            if engine_tables[table][donor] in ['MyISAM', 'ARCHIVE']:
                if not restore_myisam.has_key(patient):
                    restore_myisam[patient] = dict()
                if not restore_myisam[patient].has_key(donor):
                    restore_myisam[patient][donor] = list()
                restore_myisam[patient][donor].append(table)
            else:
                if not restore_others.has_key(patient):
                    restore_others[patient] = dict()
                if not restore_others[patient].has_key(donor):
                    restore_others[patient][donor] = list()
                restore_others[patient][donor].append(table)
    if restore_myisam or restore_others:
        print "для восстновления консистентности потребуется запустить с машин"
        if len(restore_myisam) > 0: printMYISAM()
        if len(restore_others) > 0: printOthers()

def readConfig(config):
    ''' Функция для обработки аргументов подключения к БД.
        Пытаемся прочитать json конфиг, при отсутсвии используем default значения.
        При успользовании аргументов(--user, --password), последние перекрывают config значения.
        Возвращает кортеж значений: user, passwd, db.
    '''
    user, passwd, db = USER, PASSWORD, DB
    conf = dict()
    if os.path.exists(opts.config) and os.path.isfile(opts.config):
        with open(opts.config) as fd:
            try:
                raw = json.load(fd)
                if raw.has_key('db_config'):
                    conf = raw['db_config']
                logger.debug("config\n{0}".format(conf))
            except Exception as err:
                logger.debug("error parse json config {0}: {1}".format(opts.config, str(err)))
            if conf.has_key('user'):
                user=conf.get('user')
            if conf.has_key('pass'):
                passwd=conf.get('pass')
    user = opts.user if opts.user else user
    passwd = opts.password if opts.password else passwd
    if not passwd:
        passwd = getpass.getpass("enter password for {0}:".format(user))
    logger.debug("user: {0}, password: {1}, db: {2}".format(user, passwd, db))
    return user, passwd, db

def main():
    logger.debug("start debug mode")
    user, password, db = readConfig(opts.config)
    list_servers = [ i['host'] for i in SERVERS ]
    meta = dict()
    threads = [Thread(target=getMetaTables, args=[i, item['host'], item['port'], user, db,
                                          password, meta]) for i, item in enumerate(SERVERS)]
    for t in threads:
        t.start()
    for i in threads:
        i.join()
    cnt_tables, engine_tables = getTableDicts(meta)
    cnt_servers = len(list_servers)
    lost_tables, lost_rows, lost_meta = showDiffTables(cnt_tables, cnt_servers)
    donor_list = foundLost(lost_tables, list_servers)
    printFullRestoreCmd(donor_list, engine_tables)
    donor_list = foundLost(lost_rows, list_servers)
    printFullRestoreCmd(donor_list, engine_tables)

if __name__ == '__main__':
    opts, args, logger = parseArgs()
    main()
