#!/usr/bin/perl -w

=head1 NAME

    mysql_pack_tables.pl

=head1 DESCRIPTION

    Упаковка(myisampack) и архивация(ENGINE=ARCHIVE) старых таблиц.
    В директории /etc/mysql/pack_tables/ должны лежать конфиги в формате YAML,
    с информацией, в каких базах, какие таблички пропускать через myisampack,
    каким менять storage engine на ARCHIVE, а какие удалять.

    Находим первое правило, подходящее по database/name_re, остальные правила 
    к этой таблице не пытаемся применять.

    Если из имени(name_re) матчится строка из 8-ми символов - считаем датой в формате YYYYMMDD
    Если из имени матчится строка из 6-ми символов - считаем датой в формате YYYYMM
    
=head1 CONFIG EXAMPLE

    ---
    socket: /var/run/mysqld.ppc/mysqld.sock
    user: root
    password: secret
    rules:
      -
        database: ppclog
        name_re: .*_(\d{8})
        pack_after: 7d
        archive_after: 6M
      -
        database: ppclog
        name_re: logsuggestion_(\d{8})
        drop_after: 1M
      -
        database: ppclog_bs
        name_re: .*_(\d{8})
        pack_after: 7d
        archive_after: 6M

=cut

use strict;

use Pid::File::Flock;
use POSIX qw/strftime/;
use File::Touch qw/touch/;
use Getopt::Long;
use YAML::Syck;
use DateTime;
use Time::HiRes;

use DBI;
use DBD::mysql;

use Yandex::DateTime;
use Yandex::Shell;

my $DEFAULT_CONFIG_DIR = '/etc/mysql/pack_tables';
# этот файл будем тачить при успешном завершении
my $SUCCESS_FILE = '/var/cache/mysqlpack/last-success.time';

# Чтение конфига
my @CONFIG_DIRS;
my @CONFIG_FILES;
my $DEBUG = $ENV{DEBUG};
my $UNTIL;
my $SLEEP_COEF = 1;
my $TABLE_PREFIX = '';
my $LOCKNAME;
GetOptions(
    "until=s" => \$UNTIL,
    "table-prefix=s" => \$TABLE_PREFIX,
    "sleep-coef=f" => \$SLEEP_COEF,
    "config=s" => \@CONFIG_FILES,
    "config-dir=s" => \@CONFIG_DIRS,
    "success-file=s" => \$SUCCESS_FILE,
    "lockname=s" => \$LOCKNAME,
    "debug" => \$DEBUG,
    );
if (!@CONFIG_FILES && !@CONFIG_DIRS) {
    push @CONFIG_DIRS, $DEFAULT_CONFIG_DIR;
}

die "Incorrect until param: '$UNTIL'" if $UNTIL && $UNTIL !~ /^([01]\d|2[0123]):[0-5]\d$/;

my @actions = (
    [drop => \&drop_table],
    [archive => \&archive_table],
    [pack => \&pack_table],
    [remove_partitioning => \&remove_partitioning],
    );

# Получаем конфиги
for my $cfg_dir (@CONFIG_DIRS) {
    opendir(my $cfg_dir_fh, $cfg_dir) || die "Can;t open directory $cfg_dir: $!";
    push @CONFIG_FILES, grep {-e} map {"$cfg_dir/$_"} grep {/^[\w\-]+$/} readdir($cfg_dir_fh);
    closedir($cfg_dir_fh) || die "Can't close $cfg_dir: $!";
}

Pid::File::Flock->new(name => $LOCKNAME, debug => $DEBUG);

# Начинаем обрабатывать конфиги
for my $cfg_file (@CONFIG_FILES) {
    l("Start process $cfg_file\n");
    my $cfg = YAML::Syck::LoadFile($cfg_file);
    process_cfg($cfg);
}
touch($SUCCESS_FILE);

# обработка обного конфига
sub process_cfg {
    my ($cfg) = @_;
    my $dbh = DBI->connect("DBI:mysql:;mysql_socket=$cfg->{socket}", $cfg->{user} || 'root', $cfg->{password} || '',  {RaiseError => 1});
    $cfg->{myisamchk_opts} ||= ["--myisam-sort-buffer-size=512M"];

    # Группируем правила по базам
    my %db_rules;
    for my $r (@{$cfg->{rules}}) {
        for my $a (@actions) {
            next if !defined $r->{"$a->[0]_after"};
            $r->{"_$a->[0]_border"} = date_border($r->{"$a->[0]_after"});
            l("Border: ".$r->{"$a->[0]_after"}." -> ".$r->{"_$a->[0]_border"});
        }
        for my $db (split /\s*,\s*/, $r->{database}) {
            push @{$db_rules{$db}}, $r;
        }
    }
    my $prev_t;
    for my $database (sort keys %db_rules) {
        l("Start use $database");
        die "Incorrect database name: '$database'" if $database !~ /^\w+$/;
        $dbh->do("USE $database");
        $dbh->do("SET SQL_LOG_BIN = 0");
        for my $tbl (@{$dbh->selectcol_arrayref("SHOW TABLES")}) {
            next if index($tbl, $TABLE_PREFIX) != 0;
            if ($UNTIL && strftime("%H:%M", localtime) ge $UNTIL) {
                l("Exit due to --until param");
                exit 0;
            } 

            if ($SLEEP_COEF > 0) {
                Time::HiRes::sleep((Time::HiRes::time() - $prev_t) * $SLEEP_COEF) if $prev_t;
                $prev_t = Time::HiRes::time();
            }

            l(" Start process $tbl");
            my ($type, $partitioned) = table_type($dbh, $tbl);
            if ($type eq 'crashed') {
                l("  skip CRASHED $tbl");
                next;
            }

            for my $r (@{$db_rules{$database}}) {
                if ($tbl =~ /^(?:$r->{name_re})$/) {
                    l("  name matched with $r->{name_re}");
                    my $tbl_date = $1;
                    if (length($tbl_date) == 6) {
                        # 31-ое число есть не в каждом месяце, но сравнение лексикографическое,
                        # поэтому нам не важно
                        $tbl_date .= '31';
                    }
                    for my $a (@actions) {
                        if (defined $r->{"_$a->[0]_border"} && $tbl_date lt $r->{"_$a->[0]_border"}) {
                            l("  rule '$a->[0]' matched ($tbl_date lt ".$r->{"_$a->[0]_border"}.")");
                            $a->[1]->($dbh, $database, $tbl, $cfg);
                            last;
                        }
                    }
                    last;
                }
            }
        }
    }
}

# удаление таблички
sub drop_table {
    my ($dbh, $database, $tbl) = @_;
    l("  start drop table");
    $dbh->do("DROP TABLE $tbl");
}

# упаковка таблички
sub pack_table {
    my ($dbh, $database, $tbl, $cfg) = @_;
    my ($type, $partitioned) = table_type($dbh, $tbl);
    if ($partitioned) {
        l('  remove partitioning');
        $dbh->do("ALTER TABLE $tbl REMOVE PARTITIONING");
    }
    if ($type eq 'myisam-packed' || $type eq 'myisam-small' || $type eq 'archive') {
        l('  skip');
        return;
    } elsif ($type ne 'myisam') {
        l('  convert to myisam');
        $dbh->do("ALTER TABLE $tbl ENGINE=MyISAM");
    }
    my (undef, $tmp_dir) = $dbh->selectrow_array("show variables like 'tmpdir'");
    my (undef, $data_dir) = $dbh->selectrow_array("show variables like 'datadir'");
    l("  start myisampack");
    $dbh->do("FLUSH TABLE $tbl");
    yash_system('myisampack', '-T' => $tmp_dir, '-s', "$data_dir/$database/$tbl.MYI");
    l("  start myisamchk");
    yash_system('myisamchk', @{$cfg->{myisamchk_opts}}, '-rqs', '-t' => $tmp_dir, "$data_dir/$database/$tbl.MYI");
    l("  myisamchk finished");
    $dbh->do("FLUSH TABLE $tbl");
}

# удаление партиций
sub remove_partitioning {
    my ($dbh, $database, $tbl) = @_;
    my ($type, $partitioned) = table_type($dbh, $tbl);
    if ($partitioned) {
        l('  remove partitioning');
        $dbh->do("ALTER TABLE $tbl REMOVE PARTITIONING");
    } else {
        l('  skip');
    }
}

# архивирование таблички
sub archive_table {
    my ($dbh, $database, $tbl) = @_;
    my ($type, $partitioned) = table_type($dbh, $tbl);
    if ($type eq 'archive') {
        l('  skip');
        return;
    }
    l("  archive table $tbl");
    my $newtbl = $tbl."_pack_tmp";
    $dbh->do("DROP TABLE IF EXISTS $newtbl");
    $dbh->do("CREATE TABLE $newtbl ENGINE=ARCHIVE AS SELECT * FROM $tbl");
    $dbh->do("RENAME TABLE $tbl TO ${tbl}_old, $newtbl TO $tbl, ${tbl}_old TO $newtbl");
    $dbh->do("DROP TABLE $newtbl");
}

# простое логгирование в STDOUT
sub l {
    print join(" ", strftime("%Y-%m-%d:%H:%M:%S", localtime), @_), "\n" if $DEBUG;
}

# тип таблицы: myisam, myisam-packed, innodb, archive, ...
sub table_type {
    my ($dbh, $tbl) = @_;
    my $status = $dbh->selectrow_hashref("SHOW TABLE STATUS LIKE ?", {}, $tbl);
    my $partitioned = $status->{Create_options} && $status->{Create_options} =~ /partitioned/;
    my $type;
    if (!defined $status->{Engine} && $status->{Comment} && $status->{Comment} =~ /is marked as crashed/) {
        $type = 'crashed';
    } elsif ($status->{Engine} eq 'MyISAM') {
        if ($status->{Row_format} eq 'Compressed') {
            $type = 'myisam-packed';
        } elsif ($status->{Data_length} < 1 * 1024 * 1024) {
            $type = 'myisam-small';
        } else {
            $type = 'myisam';
        }
    } else {
        $type = lc($status->{Engine});
    }
    return ($type, $partitioned);
}

# получить отформатированную дату
sub date_border {
    my ($duration, $fmt) = @_;
    return undef if !defined $duration || !$duration;
    return DateTime->now(time_zone => "local")->subtract(duration($duration))->strftime($fmt || '%Y%m%d');
}

