#!/usr/bin/perl

use Direct::Modern;

use Digest::MD5 'md5_hex';
use File::Slurp 'slurp';
use FindBin '$Bin';
use JSON 'from_json';
use List::Util 'min';
use LWP::Simple qw();

use Yandex::DBTools;
use Yandex::DBShards;

use my_inc '/var/www/ppc.yandex.ru', for => 'protected';

BEGIN {
    $ENV{SETTINGS_LOCAL_SUFFIX} = 'ROProd';
};

use PrimitivesIds;
use Settings;

my $SOURCE_URL = 'https://proxy.sandbox.yandex-team.ru/236205018';
my $FILENAME = 'yandex_bid_sample.tsv';

if ( ! -e $FILENAME || md5_hex( slurp($FILENAME) ) ne '312ec995eed588fcecf5f7705e2bc953' ) {
    say "downloading $FILENAME...";
    LWP::Simple::mirror($SOURCE_URL, "$Bin/$FILENAME");
    say "...done";
}

open my $fh, '<', $FILENAME;

my $bids;
while(<$fh>) {
    chomp;
    next if /^account_id/;

    my %rec;
    @rec{ qw( account_id campaign_id adgroup_id keyword_id current_bid bid ) } = split /\t/;
    $rec{bid} += 0;

    push @$bids, \%rec;
}

close $fh;

open my $result_fh, '>', 'result.tsv';
say $result_fh join( "\t", qw( campaign_id adgroup_id keyword_id bid rarely_served db_bid db_bid_context bs_bid bs_bid_context ) );

if ( ! -d "$Bin/audit" ) {
    mkdir "$Bin/audit";
}

my @adgroup_ids = map { $_->{adgroup_id} } @$bids;
my $rarely_served_map = get_hash_sql( PPC( pid => \@adgroup_ids ),
    [ 'select pid, is_bs_rarely_loaded from phrases', where => { pid => SHARD_IDS } ] );

foreach my $rec (@$bids) {
    my $db_rows = get_all_sql( PPC( cid => $rec->{campaign_id} ), [
        'select campaigns.OrderID, banners.BannerID, bids.PhraseID, bids.price as db_bid, bids.price_context as db_bid_context',
        'from campaigns',
        'inner join phrases on phrases.cid = campaigns.cid',
        'inner join banners on banners.pid = phrases.pid',
        'inner join bids on bids.pid = phrases.pid',
        where => {
            'campaigns.cid' => $rec->{campaign_id},
            'phrases.pid' => $rec->{adgroup_id},
            'bids.id' => $rec->{keyword_id},
        }
    ] );

    my ( $db_bid, $db_bid_context, $bs_bid, $bs_bid_context );

    if ( $db_rows && @$db_rows ) {
        my ($first_row) = @$db_rows;
        ( $db_bid, $db_bid_context ) = @$first_row{ qw( db_bid db_bid_context ) };

        if ( my @bs_banner_ids = grep { defined $_ && $_ > 0 } map { $_->{BannerID} } @$db_rows ) {
            my $bs_banner_id = min @bs_banner_ids;

            my $audit_result_filename = "$Bin/audit/banner-id-$bs_banner_id.json";
            if ( ! -e $audit_result_filename ) {
                say "Downloading audit for bs_banner_id=$bs_banner_id...";
                LWP::Simple::mirror( "http://yabs.yandex.ru/audit?banner-id=$bs_banner_id", $audit_result_filename );
            }

            my $audit_content = from_json( slurp($audit_result_filename) );
            for my $banner ( @{ $audit_content->[0]->{Banners} } ) {
                for my $phrase ( @{ $banner->{Phrases} } ) {
                    next unless $phrase->{PhraseID} == $first_row->{PhraseID};
                    $bs_bid = $phrase->{Price} / 1000000;
                    $bs_bid_context = $phrase->{ContextPrice} / 1000000;
                }
            }

        }
    }

    my $rarely_served = $rarely_served_map->{ $rec->{adgroup_id} };

    say $result_fh join( "\t", map { $_ // 'MISSING' }
        ( @$rec{ qw( campaign_id adgroup_id keyword_id bid ) }, $rarely_served, $db_bid, $db_bid_context, $bs_bid, $bs_bid_context ) );
}

sub key {
    my ($rec) = @_;
    return join( ':', @$rec{ qw( campaign_id adgroup_id keyword_id ) } );
}
