#!/usr/bin/perl

use Direct::Modern;
use FindBin;
use lib $FindBin::Bin . '/../../../protected';
use lib $FindBin::Bin . '/../../../perl/settings';
use lib $FindBin::Bin . '/../../../perl/rbac-elementary';

use List::MoreUtils qw/uniq/;

use Settings;
use ScriptHelper;
use RBAC2::Extended;
use RBACDirect;
use RBACElementary;
use Primitives;
use PrimitivesIds;
use Currency::Rate;

use Yandex::Clickhouse;
use Yandex::DBTools;
use Yandex::DBShards;

=head1 DESCRIPTION

    Cкрипт выгружает TSV-отчет по отключениям автопополнения ОС

    Параметры:
    --from YYYY-MM-DD
        начиная с какой даты смотреть логи
    --to YYYY-MM-DD
        по какую дату смотреть лог
    --host - (опционально) хост кликхауса, где лежат логи

    LOG_TEE=1 perl ./DIRECT-65280-autopay-turn-offs-report.pl --from 2017-01-12 --to 2017-04-12

=cut

extract_script_params(
    'from=s' => \my $date_from,
    'to=s' => \my $date_to,
    'host' => \my $host,
);

my $rbac = RBAC2::Extended->get_singleton(1);

my $clh = Yandex::Clickhouse->new(
    host => defined $host ? $host : 'ppchouse02k.yandex.ru',
    user => 'readonly',
);
$clh->query_format('TabSeparatedWithNames');

my $logs = $clh->query(
    $clh->format(["SELECT `log_time`, `cluid`
                     FROM `ppclog_cmd` 
                    WHERE",
                    {log_date__date__ge => $date_from,
                     log_date__date__le => $date_to,
                     log_time__datetime__ge => "$date_from 00:00:00",
                     log_time__datetime__le => "$date_to 23:59:59",
                     cmd => 'ajaxSaveAutopaySettings'},
                     "AND like(param, '%_\"autopay_mode_\":_\"none_\"%')",
                      "ORDER BY `log_time` ASC"] )
)->tsv(names => 1);
$_->{cluid} = eval($_->{cluid})->[0] for @$logs;

my @uids = uniq map { $_->{cluid} } @$logs;
my $user_agencyid = get_hash_sql(PPC(uid => \@uids), ['select u.uid, r.agency_client_id 
                                                        from users u
                                                        join agency_client_relations r on r.client_client_id = u.ClientID
                                                       where ', {
                                                               'u.uid' => SHARD_IDS
                                                            }]);
my $agency_rep_uid = rbac_get_chief_reps_of_agencies($rbac, [uniq values %$user_agencyid]);
my $agency_rep_login = get_uid2login(uid => [uniq map { @$_ } values %$agency_rep_uid ]);

my $user_managers = rbac_get_managers_of_clients($rbac, \@uids);
my $manager_login = get_uid2login(uid => [uniq map { @$_ } values %$user_managers ]);

my $cluid_info = get_hashes_hash_sql(PPC(uid => \@uids), ['select u.uid, u.login, cl.create_date client_create_date, IFNULL(cl.work_currency, "YND_FIXED") currency, IFNULL(sum(s.sum), 0) sum
                                                      from users u 
                                                      join clients cl using(ClientID)
                                                      join campaigns c on c.uid = u.uid and c.OrderID > 0
                                                      left join bs_order_target_stat s on s.OrderID = c.OrderID and s.stat_date >= ? and s.stat_date <= ?', 
                                                     where => {'u.uid' => SHARD_IDS},
                                                     'group by u.uid'], $date_from, $date_to);
for my $info (values %$cluid_info) {
    if ($info->{currency} ne 'YND_FIXED') {
        $info->{sum} = convert_currency($info->{sum}, $info->{currency}, 'YND_FIXED', with_nds => 1);
    }
    $info->{sum} = sprintf('%.2f', $info->{sum});
}

for my $row (@$logs) {
    $cluid_info->{$row->{cluid}}->{agency_login} = $agency_rep_login->{$agency_rep_uid->{$user_agencyid->{$row->{cluid}} // 0} // 0} if $user_agencyid->{$row->{cluid}};
    $cluid_info->{$row->{cluid}}->{manager_login} = join ', ', map { $manager_login->{$_} // ()} @{ $user_managers->{$row->{cluid}} } if $user_managers->{$row->{cluid}};
    $cluid_info->{$row->{cluid}}->{turn_offs_qty} ++;
    $cluid_info->{$row->{cluid}}->{first_turn_off} //= $row->{log_time};
}

my @fields = qw/login agency_login manager_login turn_offs_qty client_create_date first_turn_off sum/;
print join("\t", @fields) . "\n";
for my $cluid (sort keys %$cluid_info) {
    print join("\t", map { $cluid_info->{$cluid}->{$_} // '' } @fields) . "\n";
}
