package Plack::App::Alive;

# $Id$

=head1 NAME

    Plack::App::Alive

=head1 DESCRIPTION

    Проверить возможность коннекта к основным БД
    и заполненность разделов на сервере.

    Параметры конструктора:
    partitions - массив дирректорий для проверки места
    db_names - названия баз для проверки возможности законнектиться

=cut

use strict;
use warnings;

use Filesys::Df qw/df/;

use parent qw/Plack::Component/;
use Plack::Util::Accessor qw(partitions db_names custom_check);

use Yandex::SendMail;

use Yandex::DBTools;

use utf8;

=head2 $FREE_SPACE_BORDER

    какая часть диска должна быть свободна (1%)

=cut
our $FREE_SPACE_BORDER ||= 0.01;

=head2 $FREE_INODES_BORDER

    какая часть inode должны быть свободны (1%)

=cut
our $FREE_INODES_BORDER ||= 0.01;

=head2 $SEND_ALERT

    нужно ли отправлять письмо о проблемах (по умолчанию - нужно)

=cut
our $SEND_ALERT ||= 1;

=head2 call

    plack callback

=cut
sub call {
    my ($self, $env) = @_;

    my @errors;

    # проверяем заполненность диска
    for my $path (@{$self->partitions || []}) {
        my $df = df($path, 1);
        if (!$df) {
            push @errors, {test => "df:$path", error => "no df data"};
            next;
        }
        my $space_proc = $df->{bfree} / ($df->{blocks}||1);
        if ($space_proc < $FREE_SPACE_BORDER) {
            push @errors, {test => "df:space:$path", error => "not enough space: $space_proc < $FREE_SPACE_BORDER"};
        }
        my $inodes_proc = $df->{ffree} / ($df->{files}||1);
        if ($inodes_proc < $FREE_INODES_BORDER) {
            push @errors, {test => "df:inodes:$path", error => "not enough inodes number: $inodes_proc < $FREE_INODES_BORDER"};
        }
    }

    for my $dbname (@{$self->db_names || []}) {
        if (! eval { Yandex::DBTools::soft_reconnect_db($dbname); } ) {
            push @errors, {test => "db:$dbname", error => $@};
        }
    }

    while (my ($check_name, $check_sub) = each %{$self->custom_check || {}} ) {
        next if eval { $check_sub->() };
        push @errors, {test => "custom:$check_name", error => $@};
    }

    if (!@errors) {
        return [200, ['Content-Type' => 'text/plain'], ['ok']];
    } else {
        if ($SEND_ALERT) {
            my $letter_text = join '', map {"$_->{test} - $_->{error}\n"} @errors;
            send_alert($letter_text, "alive check");
        }

        my $resp = "errors: ".join(", ", map {$_->{test}} @errors);
        return [500, ['Content-Type' => 'text/plain'], [$resp]];
    }
}

1;
