package Plack::CaptchaChecker;

=head1 NAME

    Plack::CaptchaChecker

=head1 DESCRIPTION

    Проверка необходимости показа капчи, с сохранением стейта в memcached

=cut

use strict;
use warnings;

use Digest::CRC qw/crc32/;
use Params::Validate qw/validate CODEREF/;

use Plack::UTF8Request;
use Yandex::HashUtils;
use Yandex::Memcached;

use utf8;

=head2 Plack::CaptchaChecker->new(prefix => "xxx", md_servers => ...)

    Обязательные параметры:
      prefix - строка - префикс ключей=> 1,
      md_servers - memcached сервера, в формате Yandex::Memcached
    Опциональные:
      captcha_interval, captcha_max_req, captcha_freq, captcha_incr
        - за captcha_interval секунд пользователь может совершить captcha_max_req запросов,
          после этого начинаем показывать капчу на каждый captcha_freq запрос
          если передан captcha_incr, то каждый запрос считается за captcha_incr запросов (по умолчанию captcha_incr=1)
      captcha_limits_sub - ссылка на функцию, которая принимает env и возвращает ссылку на хэш,
        с переопределениями captcha_interval, ...
      captcha_key_sub - ссылка функцию, получающую env, и возвращающую ключ, по которому считаем 
        статистику (по умолчанию - ip адрес)
    

=cut
sub new {
    my $class = shift;
    my $self  = $class->_init(@_);

    return bless $self, $class;
}


sub _init {
    my $class = shift;
    my %p = validate(@_, {
        prefix => 1,
        md_servers => 1,
        captcha_interval => 0, captcha_max_req => 0, captcha_freq => 0, captcha_incr => {default => 1},
        captcha_limits_sub => {type => CODEREF, optional => 1},
        captcha_key_sub => {type => CODEREF, optional => 1},
        }
    );

    my $self = \%p;

    $self->{_default_limits} = {};
    $self->{_default_limits}->{$_} = delete $self->{$_} for qw/captcha_interval captcha_max_req captcha_freq/;

    $self->{md} = Yandex::Memcached->new(servers => $self->{md_servers});

    return $self;
}


sub key {
    my ($self, $env) = @_;
    
    # по умолчанию - ограничиваем число запросов на ip
    my $ident = $self->{captcha_key_sub} 
        ? $self->{captcha_key_sub}->($env) 
        : Plack::UTF8Request->new($env)->address;

    # для размазывания времени блокировки у разных клиентов(key-ев)
    my $time_shift = crc32($ident) % $self->{_check_limits}->{captcha_interval};

    my $div_time = int( (time - $time_shift) / $self->{_check_limits}->{captcha_interval} );

    return $self->{prefix}."/$div_time/".$ident;
}

# увеличение счётчика, возвращаем 1 - нужно показать капчу, 0 - не нужно
sub check {
    my ($self, $env) = @_;

    my $limits = hash_merge {}, $self->{_default_limits};
    hash_merge $limits, $self->{captcha_limits_sub}->($env) if $self->{captcha_limits_sub};
    $self->{_check_limits} = $limits;

    my $key = $self->{_check_key} = $self->key($env);

    my $incr_ret = $self->{md}->incr($key, $self->{captcha_incr});
    if (!defined $incr_ret) {
        # memcached недоступен
        return 0;
    } elsif (!$incr_ret) {
        # такого ключа ещё нет
        $self->{md}->add($key, $self->{captcha_incr}, $limits->{captcha_interval});
        return 0;
    } else {
        return $incr_ret > $limits->{captcha_max_req};
    }
}

# колбэк, вызываемый при успешном распозновании капчи
sub on_recognition {
    my ($self, $env) = @_;

    if (my $key = $self->{_check_key}) {
        my $limits = $self->{_check_limits};
        $self->{md}->set($key, $limits->{captcha_max_req} - $limits->{captcha_freq} + 1, $limits->{captcha_interval});
    }
}

1;
