package Plack::Middleware::ACL;
use base qw/Plack::Middleware/;

# $Id$

=head1 NAME

    Plack::Middleware::ACL

=head1 SYNONPSIS

    enable 'ACL', rules => Yandex::LiveFile::YAML->new(filename => '...');

    # если в rules есть запись типа "networks:<filename>", указывающая на файл со списком хостов, нужно указать директорию, в которой этот файл нужно искать
    enable 'ACL', rules => ..., networks_dir => '/var/www/ppc.yandex.ru/protected/data/networks';

    # можно писать ошибки в лог (объект Yandex::Log)
    enable 'ACL', rules => ..., error_log => [log_file_name => 'intapi_check_acl_errors.log', ...];  # в массиве аргументы Yandex::Log->new

=head1 DESCRIPTION

    Проверка ip клиента на соответствие acl. 

    Полагается на $env->{cmd}, выставляемый Plack::Middleware::DetermineCmd

    В параметре rules ожидается объект, который с помощью метода ->data отдает ссылку на хеш вида
        rule => [whitelist]

    В $env->{cmd_params}->{allow_to} ожидается список правил (rule) и/или хостов

    Если проверка не прошла успешно -- клиенту отдается 403 Forbidden

=cut


use strict;
use warnings;

use List::Util qw/any/;
use Net::Patricia;
use NetAddr::IP;
use Net::IPv6Addr;
use Plack::Request;
use Plack::Util::Accessor qw/rules networks_dir error_log/;
use Socket qw/getnameinfo getaddrinfo NI_NUMERICHOST/;

use Yandex::Log;

use utf8;

our $NETWORKS_DIR;
our $error_log;

=head2 call

    Коллбэк для Plack::Middleware

=cut

sub call {
    my ($self, $env) = @_;
    $error_log //= Yandex::Log->new(@{ $self->error_log // [no_log=>1] });

    $NETWORKS_DIR = $self->networks_dir;

    if (exists $env->{cmd_params} && exists $env->{cmd_params}->{allow_to}) {
        my $allow_to = $env->{cmd_params}->{allow_to};
        my $rules = $self->rules
            ? ($self->rules->data or die "can't load the rules")
            : {};
        my $whitelist = [];
        for my $item (ref $allow_to ? @$allow_to : $allow_to) {
            if (my $hosts = $rules->{$item}) {
                push @$whitelist, @$hosts;
            } else {
                push @$whitelist, $item;
            }
        }

        my $ip = Plack::Request->new($env)->address;
        unless (acl_ok($ip, $whitelist)) {
            return [403, ['Content-Type' => 'text/plain'], ["Forbidden"]];
        }
    }
    
    # Дополнительная работа: разрешаем без-tvm-ные запросы по спискам skip_tvm2_check
    # формат записей в skip_tvm2_check аналогичен allow_to
    # важно, чтобы этот Middleware отработал раньше, чем CheckTVM2Ticket
    $env->{skip_tvm} = 0;
    if (exists $env->{cmd_params} && exists $env->{cmd_params}->{skip_tvm2_check}) {
        my $whitelist = $env->{cmd_params}->{skip_tvm2_check};
        my $ip = Plack::Request->new($env)->address;
        if (acl_ok($ip, $whitelist)) {
            $env->{skip_tvm} = 1;
        }
    }

    return $self->app->($env);
}


=head2 acl_ok($ip, $whitelist)

    Возвращаем 1, если $whitelist пуст, либо $ip проходит проверку
    В остальных случаях возвращаем 0

=cut

sub acl_ok {
    my ($ip, $whitelist) = @_;

    return 1 unless @$whitelist;
 
    my $err;
    my @addrs;
    ($err, @addrs) = getaddrinfo($ip, 0);
    if ($err) {
        warn $err;
        return 0;
    }
    my $addr_bin = $addrs[0]->{addr};

    my $hostname;
    ($err, $hostname) = getnameinfo($addr_bin);
    if ($err) {
        warn $err;
        return 0;
    }

    my @res;
    ($err, @res) = getaddrinfo($hostname, 0);
    if ($err) {
        warn $err;
        return 0;
    }
    my $reverse_dns_ok;
    for my $addr (@res) {
        if ($addr_bin eq $addr->{addr}) {
            $reverse_dns_ok = 1;
            last;
        }
    }

    my $ip_obj = NetAddr::IP->new($ip);
    for my $item (@$whitelist) {
        if ($item =~ /^networks:(.*)/) {
            my $file = $1;
            if (defined $NETWORKS_DIR) {
                my $ok;
                eval { $ok = is_ip_in_list($ip, "$NETWORKS_DIR/$file"); 1 } or $error_log->out($@);
                return 1 if $ok;
            } else {
                $error_log->out("found rule '$item' in ACL, but networks_dir is not defined");
            }
        } elsif ($item =~ /^[a-z0-9-\.]+$/ && $item =~ /[a-z]/) {
            # $item -- доменное имя

            return 1 if $reverse_dns_ok && $hostname eq $item;
        } elsif (ref $item eq 'Regexp') {
            # $item -- регулярка с доменным именем

            return 1 if $reverse_dns_ok && $hostname =~ /$item/;
        } else {
            # $item -- подсеть или отдельный ip

            my $item_obj = NetAddr::IP->new($item);
            next unless $item_obj->version eq $ip_obj->version; # не сравниваем IPv4-адреса и IPv6-адреса друг с другом

            # для отдельных адресов $item_obj считается подсетью с маской только из единичных битов, так что ->within работает и как сравнение двух адресов
            return 1 if $ip_obj->within($item_obj);
        }
    }

    return 0;
}

=head2 is_ip_in_list

** Функция скопирована из svn+ssh://svn.yandex.ru/direct/trunk/protected/IpTools.pm@158320 **
** с заменой 'print STDERR' на $error_log->out **
** Если нужно тут что-то поправить, посмотрите, нужно ли это сделать в IpTools. **

Параметры: 
1. IP-адрес (строка)
2. Файл с IP-адресами и масками сетей (полный путь)

Проверяет, записан ли данный адрес в файле (явно или через маску сети)

Для каждого файла создаем 2 объекта Net::Patricia, все эти объекты и времена модификации файлов 
храним в хеше:
%patricia_cache = (
   'имя файла' => {
                   mtime => <время изменения файла>,
                   checkers => [<проверяющая функция>, ...],
                  },
   .....
); 

Если с последнего обращения время модификации файла изменилось -- перечитываем файл.

ipv6-ready: да; обрабатывает сети и адреса
hbf-ready: да

=cut

{
    my %patricia_cache;

sub is_ip_in_list
{
    my ($ip_addr, $ip_file_name) = @_;

    #проверяем, что для файла $ip_file_name уже есть запись. Если ее нет -- делаем запись с mtime=0
    my $check = $patricia_cache{$ip_file_name} //= {mtime => 0};

    my $cur_mtime = (stat($ip_file_name))[9];

    if ($cur_mtime > $check->{mtime}
        || !defined $check->{checkers}
        ) {
        # разбираем файл, записываем в Patricia'вские объекты: отдельно ipv4, отдельно ipv6 (Patricia работает отдельно с теми и другими)
        # history: в качестве альтернатив предлагались: NetAddr::IP[::Lite] (lena-san) и Net::CIDR[::Lite] (pankovpv)
        my $patricia = new Net::Patricia;
        my $patricia_v6 = new Net::Patricia AF_INET6;
        my @checkers = (
            sub {scalar $patricia->match_string(shift)},
            sub {scalar $patricia_v6->match_string(shift)},
            );
        open my $fh, "<", $ip_file_name or die "Error: $!";
        while (my $network = <$fh>) {
            $network =~ s/#.*$//; # remove comments
                $network =~ s/\s+//g; # remove spaces
            next if length($network) == 0;
            if ($network =~ /^ \d{1,3} (\.\d{1,3}){3} ( \/ \d+ )? $/x) {
                # похоже на ipv4
                eval {
                    $patricia->add_string($network, 1);
                };
                if ($@) {
                    $error_log->out("ERROR in parse $ip_file_name: \"$network\"\n$@\n");
                }
            } elsif ( $network =~ /^ [0-9a-f:]+ (\/[0-9]+)? $/x  ) {
                # похоже на ipv6
                eval {
                    $patricia_v6->add_string($network, 1);
                };
                if ($@){
                    $error_log->out("ERROR in parse $ip_file_name: (ipv6) \"$network\"\n$@\n");
                }
            } elsif ( my ($proj, $net) = $network =~ /^ ([0-9a-f]+) @ ([0-9a-f:]+ (\/[0-9]+)?) $/x  ) {
                # похоже на host-based-firewall
                eval {
                    my $proj_id = hex $proj;
                    if ($proj_id == 0 || $proj_id >= 2 ** 32) {
                        die "Incorrect proj: $proj";
                    }
                    my $p = new Net::Patricia AF_INET6;
                    $p->add_string($net, 1);
                    push @checkers, sub { my $ip = shift; return $p->match_string($ip) && _hbf_proj_id($ip) == $proj_id; };
                };
                if ($@){
                    $error_log->out("ERROR in parse $ip_file_name: (ipv6) \"$network\"\n$@\n");
                }
            } else {
                # ни на что не похоже
                $error_log->out("ERROR in parse $ip_file_name: \"$network\"\n");
            }
        }
        close $fh;
        $check->{checkers} = \@checkers;
        $check->{mtime} = $cur_mtime;
    }

    return scalar any(sub {eval {$_->($ip_addr)}}, @{$check->{checkers}});
}
}

# описание формата ip https://wiki.yandex-team.ru/NOC/newnetwork/hbf/project_id/
sub _hbf_proj_id {
    my $ip = shift;
    my @parts = eval { Net::IPv6Addr->new($ip)->to_array(); };
    return undef unless @parts;
    return hex("$parts[4]$parts[5]");
}

1;
