package PortBlocker;
use strict;
use warnings;
use feature 'state';

=head1 NAME

PortBlocker

=head1 DESCRIPTION

Модуль для кода, который совместно используют portblocker.pl и portblocker-control.pl. Сейчас
это функции для работы с файлом с информацией о том, какие файлы конфигурации включены
($ENABLED_CONFFILES_STATE_PATH).

=cut

use parent 'Exporter';

our @EXPORT = qw(
    $CONFIG_GLOB
    get_enabled_conffiles_hash
    save_enabled_conffiles_hash

    get_configuration
    get_conffile_ports
);

use File::Basename;
use File::Slurp 'write_file';
use YAML 'LoadFile';

use Yandex::ListUtils;

=head1 CONFIGURATION

=head2 $CONFIG_GLOB

Где portblocker берёт файлы конфигурации.

=cut

our $CONFIG_GLOB = '/etc/portblocker/*.conf';

=head2 $ENABLED_CONFFILES_STATE_PATH

Где хранится информация о статусе: какие из конфиг-файлов включены и обрабатываются командой --set, а какие нет.

=cut

my $ENABLED_CONFFILES_STATE_PATH = '/var/spool/portblocker/enabled-conffiles.yaml';

=head1 SUBROUTINES/METHODS

=head2 get_enabled_conffiles_hash

=cut

sub get_enabled_conffiles_hash {
    state $enabled_conffiles_hash;

    return $enabled_conffiles_hash if $enabled_conffiles_hash;

    my $enabled_conffiles_config = {};
    if ( -e $ENABLED_CONFFILES_STATE_PATH ) {
        $enabled_conffiles_config = LoadFile($ENABLED_CONFFILES_STATE_PATH);
    }

    $enabled_conffiles_hash = {};
    for my $conffile ( glob( $CONFIG_GLOB) ) {
        my $conffile_base = basename($conffile);
        $enabled_conffiles_hash->{$conffile_base} = $enabled_conffiles_config->{$conffile_base} // 1;
    }

    return $enabled_conffiles_hash;
}

=head2 save_enabled_conffiles_hash

=cut

sub save_enabled_conffiles_hash {
    my ($enabled_conffiles_hash) = @_;
    write_file( $ENABLED_CONFFILES_STATE_PATH, { atomic => 1 }, YAML::Dump( $enabled_conffiles_hash ) );
}

# булевское значение: все файлы конфигурации загружаются только 1 раз
my $conffiles_loaded;

# массив правил; TODO: уточнить формат
my $config;

# { 17401 => [ qw( 1.conf 2.conf ) ], 17501 => ['1.conf'], ... }
# если какой-то порт определён в нескольких файлах, это ошибка
my %port_defined_in_conffiles;

# { '1.conf' => [ 17401, 17501, 17601 ], ... }
my %conffile_ports;

=head2 load_conffiles

=cut

sub load_conffiles {
    return if $conffiles_loaded;

    $config = [];

    for my $conffile ( glob($CONFIG_GLOB) ) {
        my $config_from_file = YAML::LoadFile($conffile);

        my $conffile_base = basename($conffile);

        $conffile_ports{$conffile_base} = [];

        my $rules = $config_from_file->{rules};
        my $lists = $config_from_file->{lists};

        for my $rule (@$rules) {
            $rule->{conffile} = $conffile_base;

            # немного хитрая логика: внутри правила можно задать ports (массив) и/или port (одно значение)
            # если заданы оба, значение внутри port записывается в конец массива ports
            # если задано только port, в итоге инициализируется ports = [ $port ]
            # в любом случае в конце концов в структуре в памяти остаётся только ports, port удаляется
            my $ports = $rule->{ports} || [];
            push @$ports, $rule->{port} if $rule->{port};

            $rule->{ports} = $ports;
            delete $rule->{port};

            for my $port (@$ports) {
                $port_defined_in_conffiles{$port} ||= [];
                push @{ $port_defined_in_conffiles{$port} }, $conffile;
            }

            push @{ $conffile_ports{$conffile_base} }, @$ports;

            if ( my $listname = delete $rule->{'allow-list'} ) {
                $rule->{allow} = $lists->{$listname};
            }
        }

        push @$config, @$rules;
    }

    my $port_collision = 0;
    for my $port ( nsort keys %port_defined_in_conffiles ) {
        my $conffiles = $port_defined_in_conffiles{$port};

        if ( @$conffiles > 1 ) {
            $port_collision = 1;
            print STDERR "Rules for port $port are defined in multiple conffiles: " . join( ', ', @$conffiles ) . "\n";
        }
    }

    die "Port collision(s) detected, quitting.\n" if $port_collision;

    $conffiles_loaded = 1;
}

=head2 get_configuration

=cut

sub get_configuration {
    load_conffiles();
    return $config;
}

sub get_conffile_ports {
    my ($conffile) = @_;
    load_conffiles();
    return $conffile_ports{$conffile};
}

1;
