# -*- coding: utf-8 -*-

import threading
import traceback
import time
import os

from kazoo.retry import KazooRetry
from kazoo.client import KazooClient, KazooState, KeeperState
from kazoo.exceptions import LockTimeout
from kazoo.recipe.lock import Lock
from kazoo.recipe.watchers import DataWatch, ChildrenWatch

from ppcinv.zk_storage import zk_get_host_path, zk_get_fqdn_path, zk_load_host
from ppcinv.helpers import fs_update_host

import logging
logging.getLogger(__name__).addHandler(logging.NullHandler())


# возможные ситуации:
# * хост изолирован по сети, или одна из нод зк (к которой был активный коннект) изолирована по сети - переподключаемся с zk_retry, через 2x iteration_period - с нуля
# * подключение к ноде zk в пределах дц, дц изолирован по сети - аналогично предыдущему пункту, лок пропадает примерно через zk_timeout/2
# * восстановилось подключение к кластеру после пропадания лока с него - получаем lock_watch (либо session_expired) и reconnect
# * кратковременные дропы по сети в пределах zk_timeout без разрыва соединения - не замечаем, либо reconnect без потери сесиии и лока
# * хост изолирован по сети во время lock.acquire - получаем SessionExpiredError, реконнект, hook_lock_lost не вызывается


class PPCInvDelivery(object):
    # iteration_period желательно делать больше zk_timeout
    def __init__(self, zk_hosts, fqdn, store_to, zk_timeout=10, zk_lock_timeout=5, iteration_period=30, reconnect_period=3600, restart_period=3600*4, remove_if_older=180):
        self.zk_hosts = zk_hosts
        self.zk_timeout = zk_timeout
        self.iteration_period = iteration_period
        self.zk_lock_timeout = zk_lock_timeout
        self.reconnect_period = reconnect_period
        self.restart_period = restart_period
        self.remove_if_older = remove_if_older

        self.fqdn = fqdn
        self.store_to = store_to
        self.logger = logging.getLogger(__name__)
        self.lock_data = self.fqdn + ' ' + str(os.getpid())
        self.reconnect_event = threading.Event()
        self.zk = None
        self.init_time = time.time()


    def zk_init(self):
        if isinstance(self.zk, KazooClient):
            self.zk.stop()
        # неизвестно, как мы сюда попали, спим, чтобы zk-кластер нас точно забыл
        time.sleep(self.iteration_period + self.zk_timeout)

        self.zk = None
        self.lock = None
        self.host_path = None
        self.host_data = None
        self.raw_host_path = None
        self.raw_host_data = None
        self.connect_time = 0
        # сетевые проблемы с одной из zk-нод скорее всего не приведут к потере лока - будет переподключение к рабочей ноде, если успеет за iteration_period
        zk_retry = { 'max_tries': 3, 'delay': 1, 'max_jitter': 1, 'backoff': 1, 'ignore_expire': False }
        self.zk = KazooClient(hosts=self.zk_hosts, timeout=self.zk_timeout, connection_retry=zk_retry, command_retry=zk_retry)
        self.zk.add_listener(self.connection_listener)


    def run(self):
        self.logger.info('Start delivery service')

        while True:
            # перезапускаем скрипт целиком по таймауту
            if time.time() - self.init_time > self.restart_period:
                self.logger.info('Stop delivery service - restart_period reached')
                break
            self.check_host_age()

            if not self.zk or (isinstance(self.zk, KazooClient) and self.zk.state != KazooState.CONNECTED):
                try:
                    self.one_iteration()
                except Exception as e:
                    self.logger.error('Exception in delivery iteration: %s - %s' % (type(e), e))
                    # logging.debug(traceback.format_exc())
                    if isinstance(self.zk, KazooClient): 
                        # что-то пошло не так, но подключение могли установить - отключаемся
                        self.zk.stop()
                    continue

            if self.reconnect_event.wait(self.iteration_period):
                self.logger.info('Reconnect - reconnect_event received')
                # переподключаемся с нуля при сомнениях валидности лока в zk
                self.zk.stop()
                self.reconnect_event.clear()
            elif (isinstance(self.zk, KazooClient) and self.zk.state == KazooState.SUSPENDED):
                self.logger.info('Lock may be lost, sleep one more iteration timeout and do full reconnect')
            elif (isinstance(self.zk, KazooClient) and self.zk.state == KazooState.CONNECTED):
                self.update_host(touch_only=True)
                # переподключаемся с нуля по таймауту, на всякий случай
                if time.time() - self.connect_time > self.reconnect_period:
                    self.logger.info('Reconnect - reconnect_period reached')
                    self.zk.stop()


    def check_host_age(self):
        try:
            mtime = os.stat(self.store_to).st_mtime
            if time.time() - mtime > self.remove_if_older:
                self.logger.warning('File is too old')
                self.invalidate_host()
        except Exception as e:
            logging.debug('Cannot check host metadata file age: %s - %s' % (type(e), e))
            pass


    def one_iteration(self):
        self.zk_init()
        self.logger.info('New iteration started')
        self.zk.start()

        self.logger.info('Get host path for ' + self.fqdn)
        self.host_path, self.raw_host_path = zk_get_host_path(self.zk, self.fqdn)
        self.zk.DataWatch(zk_get_fqdn_path(self.fqdn), self.fqdn_watch)

        self.logger.info('Acquire lock on ' + self.host_path)
        self.lock = self.zk.Lock(self.host_path, self.lock_data)

        # если не смогли взять лок при работающем зукипере - удаляем файл и выходим
        # в остальных случаях выкидывает по Exception и делаем новую итерацию
        if not self.lock_host():
            self.logger.critical('Lock is already acquired by another host!')
            self.invalidate_host()
            raise Exception('Cannot acquire lock')
        self.zk.DataWatch(self.lock.path + '/' + self.lock.node, self.lock_watch)

        self.host_data, self.raw_host_data = zk_load_host(self.zk, self.host_path)
        self.update_host()
        self.zk.DataWatch(self.host_path, self.host_watch)


    def update_host(self, touch_only=False):
        if touch_only:
            try:
                os.utime(self.store_to, None)
                return
            except Exception as e:
                self.logger.error('Cannot update file mtime: %s - %s' % (type(e), e))

        self.logger.info('Save new host data to ' + self.store_to)
        fs_update_host(self.host_data, os.path.dirname(self.store_to), os.path.basename(self.store_to))
        self.logger.info('Host metadata saved')


    def invalidate_host(self):
        self.logger.critical('Remove host metadata file ' + self.store_to)
        try:
            os.remove(self.store_to)
        except Exception as e:
            self.logger.error('Cannot remove file: %s - %s' % (type(e), e))


    def lock_host(self):
        try:
            self.lock.acquire(blocking=True, timeout=self.zk_lock_timeout)
        except LockTimeout:
            try:
                children = self.zk.get_children(self.lock.path)
                for c in children:
                    ld, _ = self.zk.get(self.lock.path + '/' + c)
                self.logger.info('%s is already locked by %s %s' % (self.lock.path, c, ld))
            except Exception as e:
                self.logger.error('Cannot get lock holder: %s - %s' % (type(e), e))
                pass

            return False
        except:
            raise

        return True


    def connection_listener(self, new_state):
        self.logger.debug('ZK state changed to %s, %s' % (self.zk.state, self.zk.client_state))
        # suspended state поймаем в run()
        if new_state == KazooState.LOST:
            self.logger.warning('ZK connection lost, reconnect')
            self.reconnect_event.set()
        elif new_state == KazooState.CONNECTED:
            self.connect_time = time.time()


    # поменялись данныe хоста - обновляем или invalidate_host (если нода пропала)
    def host_watch(self, *args):
        data, stat, event = args
        self.logger.debug('Host data watcher: %s' % (args,))

        if data and data != self.raw_host_data:
            self.logger.info('Host data changed from "%s" to "%s"' % (self.raw_host_data, data))
            self.zk.handler.spawn(self.update_host())
        elif not data:
            self.logger.critical('Host data is empty!')
            self.invalidate_host()
            self.reconnect_event.set()


    # поменялся host_path в fqdn_node - что-то очень странно, начинаем все заново (если нода изчезла - еще и invalidate_host)
    def fqdn_watch(self, *args):
        data, stat, event = args
        self.logger.debug('Host path watcher: %s' % (args,))
        if data and data != self.raw_host_path:
            self.logger.critical('Host path changed from "%s" to "%s", reconnect' % (self.raw_host_path, data))
            self.reconnect_event.set()
        elif not data:
            self.logger.critical('Cannot find host path - empty or absent fqdn node!')
            self.invalidate_host()
            self.reconnect_event.set()


    def lock_watch(self, *args):
        data, stat, event = args
        self.logger.debug('Lock watch: %s' % (args,))
        if data != self.lock_data:
            self.logger.warning('Lock watch event received: %s, reconnect' % (event,))
            self.reconnect_event.set()
