# -*- coding: utf-8 -*-

from math import floor
from ppcinv.helpers import *

def conductor_host_to_group(fqdn):
    return get_json_from_url('https://c.yandex-team.ru/api/hosts/%s?format=json' % fqdn)[0]['group']


def conductor_get_group_id(name):
    return int(get_json_from_url('https://c.yandex-team.ru/api/groups/%s?format=json' % (name,))[0]['id'])


def bot_get_host_id(fqdn):
    jres = get_json_from_url('https://bot.yandex-team.ru/api/consistof.php?name=%s&format=json' % (fqdn,))
    try:
        id = int(jres['data']['instance_number'])
    except:
        logging.getLogger(__name__).critical('Cannot get BOT id: ' + json.dumps(jres))
        raise
    return id


def parse_gencfg_host_card(instance_conf, host_conf, logger):
    # hostname должен быть всегда
    bot_fqdn = instance_conf['hostname']
    domain = instance_conf['domain']
    slot = int(instance_conf['port'])

    # что делать при отсутствии остальных полей - пока дефолтим
    dc = instance_conf.get('dc')
    loc = instance_conf.get('location')

    porto_limits = instance_conf.get('porto_limits', {})
    # округляем до ближайшего целого
    cpu_cores_guarantee = int(floor(porto_limits.get('cpu_cores_guarantee', -1) + 0.5))
    cpu_cores_limit = int(floor(porto_limits.get('cpu_cores_limit', -1) + 0.5))
    # почему-то в gencfg все инстансы разные, округляем ram до gb
    memory_guarantee_gb = porto_limits.get('memory_guarantee', -1024**3)
    memory_guarantee_gb = int(floor(memory_guarantee_gb / 1024**3 + 0.5))

    host_res = instance_conf.get('host_resources', {})
    host_mem_gb = host_res.get('memory', -1)
    host_cpu_cores = host_res.get('ncpu', -1)

    host_conf['bot_fqdn'] = bot_fqdn
    host_conf['location'] = loc
    host_conf['root_dc'] = dc
    host_conf['cpu_cores_guarantee'] = cpu_cores_guarantee
    host_conf['cpu_cores_limit'] = cpu_cores_limit
    host_conf['memory_guarantee_gb'] = memory_guarantee_gb
    host_conf['host_cpu_cores'] = host_cpu_cores
    host_conf['host_memory_gb'] = host_mem_gb

    fqdn = None
    old_launcher_fqdn = None
    hbf_backbone_fqdn = None
    # пытаемся получить старый launcher fqdn
    try:
        m = re.match(r'(\w+-\d+)' + domain, bot_fqdn)
        host_prefix = m.group(1)
        old_launcher_fqdn = host_prefix + '-' + str(slot) + '.vm' + domain
        host_conf['gencfg_old_launcher_fqdn'] = old_launcher_fqdn
        logger.debug('Old launcher.py fqdn: ' + old_launcher_fqdn)
        fqdn_to_ip(old_launcher_fqdn)
    except Exception as e:
        old_launcher_fqdn = None
        logger.debug('Cannot validate old launcher.py fqdn: %s %s' % (type(e), e))

    # пытаемся получить новый hbf-fqdn
    if instance_conf.get('hbf', {}).get('mtn_ready'):
        try:
            hbf_backbone_fqdn = instance_conf['hbf']['interfaces']['backbone']['hostname']
            logger.debug('HBF backbone fqdn: ' + hbf_backbone_fqdn)
            host_conf['gencfg_hbf_backbone_fqdn'] = hbf_backbone_fqdn
            fqdn_to_ip(hbf_backbone_fqdn)
        except Exception as e:
            hbf_backbone_fqdn = None
            logger.debug('Cannot validate hbf backbone fqdn: %s %s' % (type(e), e))

    host_conf['ready'] = True
    if hbf_backbone_fqdn and not old_launcher_fqdn:
        fqdn = hbf_backbone_fqdn
    elif old_launcher_fqdn and not hbf_backbone_fqdn:
        fqdn = old_launcher_fqdn
    elif hbf_backbone_fqdn and old_launcher_fqdn:
        fqdn = old_launcher_fqdn
        if 'a_itype_ppcvm' in instance_conf.get('tags', []):
            fqdn = hbf_backbone_fqdn
    else:
        host_conf['ready'] = False

    host_conf['fqdn'] = fqdn
    logger.debug('Selected fqdn: %s' % (fqdn,))

    return host_conf


def gencfg_group_to_host_data(gencfg_group_name, gencfg_release='trunk'):
    hosts = []
    logger = logging.getLogger(__name__)
 
    group_conf = get_json_from_url('http://api.gencfg.yandex-team.ru/%s/searcherlookup/groups/%s/instances' % (gencfg_release, gencfg_group_name))

    for instance_conf in group_conf.get('instances', []):
        host_conf = {
            'gencfg_group_name': gencfg_group_name,
            'gencfg_release': gencfg_release,
            'gencfg_hbf_backbone_fqdn': None,
            'gencfg_old_launcher_fqdn': None,
            'fqdn': None,
            'bot_fqdn': None,
            'location': None,
            'root_dc': None,
            'cpu_cores_guarantee': None,
            'cpu_cores_limit': None,
            'memory_guarantee_gb': None,
            'ready': False,
            'pos_in_cluster': None,
        }
        parse_gencfg_host_card(instance_conf, host_conf, logger)

        # всегда добавляем хост, даже если что-то не распарсили
        hosts.append(host_conf)

    return hosts


def host_update(host_to_update, update_from_host):
    added_keys = set(update_from_host.keys()) - set(host_to_update.keys())
    deleted_keys = set(host_to_update.keys()) - set(update_from_host.keys())
    updated_keys = set(update_from_host.keys()) & set(host_to_update.keys())

    changed = bool(deleted_keys)
    for k in deleted_keys:
        host_to_update.pop(k)
    for k in added_keys:
        host_to_update[k] = update_from_host[k]

    for k in list(updated_keys):
        if host_to_update[k] != update_from_host[k]:
            if k == 'pos_in_cluster':
                if update_from_host[k] is not None:
                    logging.getLogger(__name__).critical("Position changed for host %s from %s to %s" % (host_to_update['fqdn'], host_to_update['pos_in_cluster'], update_from_host['pos_in_cluster']))
                else:
                    continue
 
            host_to_update[k] = update_from_host[k]
            changed = True

    return changed


def cluster_hosts_get_merge_actions(new_hosts_data, old_hosts_data):
    fqdn_to_new_host = { x['fqdn']: x for x in new_hosts_data }
    fqdn_to_old_host = { x['fqdn']: x for x in old_hosts_data }

    added_fqdns = set(fqdn_to_new_host.keys()) - set(fqdn_to_old_host.keys())
    deleted_fqdns = set(fqdn_to_old_host.keys()) - set(fqdn_to_new_host.keys())
    updated_fqdns = set(fqdn_to_new_host.keys()) & set(fqdn_to_old_host.keys())
    changed = bool(added_fqdns) or bool(deleted_fqdns)

    remaining_positions = [ x['pos_in_cluster'] for x in old_hosts_data if x['fqdn'] not in deleted_fqdns ]
    pos_gen = host_position_generator(remaining_positions)

    for fqdn in sorted(added_fqdns):
        host_conf = fqdn_to_new_host[fqdn]
        host_conf['pos_in_cluster'] = pos_gen.next()

    added_hosts = [ fqdn_to_new_host[x] for x in added_fqdns ]
    deleted_hosts = [ fqdn_to_old_host[x] for x in deleted_fqdns ]

    updated_hosts = []
    for fqdn in sorted(updated_fqdns):
        if host_update(fqdn_to_old_host[fqdn], fqdn_to_new_host[fqdn]):
            updated_hosts.append(fqdn_to_old_host[fqdn])
            changed = True

    return { 'add': added_hosts,
             'update': updated_hosts,
             'delete': deleted_hosts,
             'changed': changed
           }


def nanny_service_get_gencfg_groups(nanny_service_name):
    # инстансы и ревизии gencfg в няне
    # curl -vvv -H "Accept: application/json" -H "Content-type: application/json" -H "Authorization: OAuth ololo' -X GET https://nanny.yandex-team.ru/v2/services/direct_export_sas/ | jq . | less

    NANNY_URL = 'https://nanny.yandex-team.ru'
    with open(os.path.expanduser('~/.nanny-token'), 'r') as f:
        OAUTH_TOKEN = f.read().strip()
    YNDX_ROOT_CA = os.path.expanduser('~/.yndx/allCAs.pem')

    session = requests.Session()
    session.headers['Authorization'] = 'OAuth {}'.format(OAUTH_TOKEN)
    session.headers['Accept'] = 'application/json'
    session.headers['Content-Type'] = 'application/json'

    url = '/'.join((NANNY_URL, 'v2', 'services', nanny_service_name))

    req = {
        'id': nanny_service_name,
        # for post requests
    }

    result = session.get(url, data=json.dumps(req), verify=YNDX_ROOT_CA).json()
    groups_list = result['runtime_attrs']['content']['instances']['extended_gencfg_groups']['groups']
    # [ {u'release': u'tags/stable-100-r39', u'name': u'SAS_DIRECT_EXPORT' ... }, ... ]
    groups_data = [ { 'release': x['release'], 'name': x['name'] } for x in groups_list ]

    return groups_data


def platform_api_v1_get_json(*args):
    PLATFORM_URL = 'https://platform.yandex-team.ru'
    with open(os.path.expanduser('~/.platform-token'), 'r') as f:
        OAUTH_TOKEN = f.read().strip()

    session = requests.Session()
    session.headers['Authorization'] = 'OAuth {}'.format(OAUTH_TOKEN)
    session.headers['Accept'] = 'application/json'
    session.headers['Content-Type'] = 'application/json'

    parts = (PLATFORM_URL, 'api', 'v1') + args
    url = '/'.join(parts)

    # на платформе пока внешний сертификат
    result = session.get(url).json()

    return result


'''
Проходит по всем сервисам конкретного Platform environment с Nanny движком и выдает список Nanny serviceId
'''
def platform_env_nanny_get_services(platform_env_id):
    # список всех nanny-сервисов в платформе
    # curl -s -H "Authorization: OAuth `cat ~/.platform-token`" -L http://platform.yandex-team.ru/api/v1/project/direct | jq '.applications[].environments[] | select (.engine == "nanny") | .objectId'
    # nanny-сервисы окружения:
    # curl -s -H "Authorization: OAuth `cat ~/.platform-token`" -L http://platform.yandex-team.ru/api/v1/nanny/direct.direct-export.prod | jq '.services[].serviceId'

    data = platform_api_v1_get_json('nanny', platform_env_id)

    # {u'engine': u'nanny', u'status': u'', u'name': u'prod', u'objectId': u'direct.direct-java-logviewer.prod', u'author': u'dspushkin', u'admin': True, u'projectName': u'direct', u'comment': u'', u'version': 1504293759671, u'services': [{u'serviceId': u'direct_java_logviewer_sas', u'serviceName': u'direct_java_logviewer_sas', u'description': u'direct java logviewer in sas', u'since': 1501011773443, u'summary': u'ONLINE'}, {u'serviceId': u'direct_java_logviewer_vla', u'serviceName': u'direct_java_logviewer_vla', u'description': u'direct java intapi in vla', u'since': 1501011616311, u'summary': u'ONLINE'}], u'creationDate': u'2017-09-01T22:22:39.671+0300', u'serviceStatuses': {u'ONLINE': 2}, u'statusMessage': u'', u'applicationName': u'direct-java-logviewer'}
    assert data['engine'] == 'nanny'
    data = [ x['serviceId'] for x in data['services'] ]

    return data


def platform_get_projects():
    PLATFORM_BAD_PROJECTS = ['junk']

    data = platform_api_v1_get_json('project')
    # [{"projectName":"advq",
    data = [ x['projectName'] for x in data if x['projectName'] not in PLATFORM_BAD_PROJECTS ]

    return data


# функции с суффиксом _data возвращают dict с внутренними (ppcinv) представлениями данных из внешних источников
def platform_project_get_environments_data(project_name):
    envs_data = []

    # {"applications":[{"objectId":"direct.commander","environments":[{"objectId":"direct.commander.public","admin":false,"engine":"qloud"
    apps = platform_api_v1_get_json('project', project_name)
    apps = apps['applications']

    for app in apps:
        app_path = app['objectId']

        for env in app['environments']:
            env_data = {
                'platform_project_name': project_name,
                'platform_app_path': app_path,
                'platform_environment_path': env['objectId'],
                'platform_environment_engine': env['engine'],
            }
            envs_data.append(env_data)

    return envs_data


def platform_get_all_environments_data():
    all_envs = []
    projects = platform_get_projects()
    for p in projects:
        envs = platform_project_get_environments_data(p)
        all_envs.extend(envs)

    return all_envs


def platform_nanny_gencfg_to_hosts_data(cluster_conf):
    hosts = []
    nanny_services = platform_env_nanny_get_services(cluster_conf['platform_environment_path'])

    for service in nanny_services:
        gencfg_groups = nanny_service_get_gencfg_groups(service)

        for group in gencfg_groups:
            group['guest_name'] = group['name']

            # TODO: fast-resync (by gencfg release change)
            hosts_data = gencfg_group_to_host_data(group['guest_name'], group['release'])
            for host_conf in hosts_data:
                host_conf.update(cluster_conf)
                if host_conf['fqdn'] is None:
                    fake_fqdn = get_cluster_path_list(host_conf)
                    fake_fqdn.extend(['fake', host_conf['bot_fqdn']])
                    host_conf['fqdn'] = '.'.join(fake_fqdn)
                host_conf['nanny_service_name'] = service
                host_conf['conductor_group_name'] = None
                host_conf['conductor_group_id'] = None
                try:
                    cgroup = conductor_host_to_group(host_conf['fqdn'])
                    cgid = conductor_get_group_id(cgroup)
                    host_conf['conductor_group_name'] = cgroup
                    host_conf['conductor_group_id'] = cgid
                except:
                    host_conf['ready'] = False
 
                host_conf['bot_id'] = None
                try:
                    bot_id = bot_get_host_id(host_conf['bot_fqdn'])
                    host_conf['bot_id'] = bot_id
                except:
                    host_conf['ready'] = False

            hosts.extend(hosts_data)

    return hosts
