# -*- coding: utf-8 -*-

import fcntl
from contextlib import contextmanager
import json
import os
import requests
import re
import sys
import shutil

from socket import getaddrinfo, gethostname, gaierror, getnameinfo, \
                    AF_UNSPEC, SOCK_STREAM, AI_ADDRCONFIG, AI_CANONNAME

import logging
import logging.handlers
logging.getLogger(__name__).addHandler(logging.NullHandler())


# [] -> 0, 1, 2, ...
# [0] -> 1, 2, ...
# [1, 3] -> 0, 2, 4, ...
def host_position_generator(positions):
    positions = sorted([-1] + positions)
    for i in range(0, len(positions)):
        pos = positions[i]
        while i >= len(positions) - 1 or pos + 1 < positions[i + 1]:
            pos += 1
            yield pos


def check_host_alias(host_alias, cluster_conf):
    # check cluster_conf specific alias params
    return str(int(host_alias))


def get_host_alias(host_data):
    return '%d' % (host_data['pos_in_cluster'],)


def get_cluster_path_list(host_data):
    return [host_data['project_name'], host_data['ya_environment_name'], host_data['cluster_name']]


def get_host_path_list(host_data):
    return get_cluster_path_list(host_data) + [get_host_alias(host_data)]


def get_json_from_url(url):
    res = None
    try:
        res = requests.get(url)
        jres = res.json()
    except Exception as e:
        errstr = 'Cannot get json data from %s: %s. Server response type: %s, str: %s.' % (url, str(e), type(res), str(res))

        if isinstance(res, requests.models.Response):
            try:
                errstr += ' Response text: ' + res.text
            except Exception as e:
                errstr += ' Cannot get response text: ' + str(e)

        logging.getLogger(__name__).debug(errstr)
        raise

    return jres


def rmtree_full(*args):
    files = []
    # сначала рекурсивно удалим директории
    for path in args: 
        if os.path.exists(path) and os.path.isfile(path) or os.path.islink(path):
            files.append(path)
        elif os.path.exists(path) and os.path.isdir(path):
            shutil.rmtree(path)
    # и потом остатки файлов
    for path in files:
        if os.path.exists(path):
            os.remove(path)


def fs_update_host(host_data, path, name=None):
    name = name if name else get_host_alias(host_data)
    logger = logging.getLogger(__name__)

    store_to = os.path.join(path, name)
    try:
        if not os.path.exists(path):
            os.makedirs(path)
        f = open(store_to + '.tmp', 'w')
        json.dump(host_data, f, sort_keys=True, indent=4, separators=(',', ': '))
        # make sure that all data is on disk
        # see http://stackoverflow.com/questions/7433057/is-rename-without-fsync-safe
        f.flush()
        os.fsync(f.fileno()) 
        f.close()
        os.rename(store_to + '.tmp', store_to)
    except Exception as e:
        logger.critical('Cannot save host metadata to %s: %s' % (store_to, e))
        raise


def init_root_logger(args):
    try:
        args.log_level = int(logging.getLevelName(args.log_level))
    except:
        args.log_level = logging.ERROR

    logger = logging.getLogger()
    logger.setLevel(args.log_level)
    handler = None
    if args.log_file:
        handler = logging.handlers.TimedRotatingFileHandler(args.log_file, when='D', backupCount=14)
    else:
        handler = logging.StreamHandler()

    handler.setLevel(args.log_level)
    handler.setFormatter(logging.Formatter('[%(asctime)s]\t%(levelname)-8s\t'+str(os.getpid())+'\t%(threadName)-25s\t%(name)-25s\t%(message)s'))
    logger.addHandler(handler)
    return logger


def fqdn_to_ip(fqdn):
    ip_set = set()
    flags = 0 # AI_ADDRCONFIG if host_name == MY_FQDN else 0
    res = getaddrinfo(fqdn, None, AF_UNSPEC, SOCK_STREAM, 0, flags)

    for addr in res:
        ip_set.add(addr[4][0])
    return list(ip_set)


@contextmanager
def flock_nb(fd):
    logger = logging.getLogger(__name__)
    try:
        fcntl.flock(fd, fcntl.LOCK_EX|fcntl.LOCK_NB)
        logger.debug('Flock acquired: ' + fd.name)
        yield
    finally:
        fcntl.flock(fd, fcntl.LOCK_UN)
        logger.debug('Flock released: ' + fd.name)
