# -*- coding: utf-8 -*-

from ppcinv.helpers import *
from kazoo.client import KazooClient

import logging
logging.getLogger(__name__).addHandler(logging.NullHandler())

ZK_FQDNS_DB_PATH = '/ppcinv/fqdns'
ZK_HOSTS_DB_PATH = '/ppcinv/hosts'


def zk_join_path(*args):
    path = []
    for node in args:
        node_parts = [ x.encode('utf8') for x in node.strip().strip('/').split('/') if x ]
        path.extend(node_parts)

    return '/' + '/'.join(path)


def zk_update_hosts(hosts_data, zk_hosts, prefix='/ppcinv'):
    if not hosts_data:
        return

    logger = logging.getLogger(__name__)
    zk = KazooClient(hosts=zk_hosts)
    zk.start()
    zk.ensure_path(ZK_FQDNS_DB_PATH)

    for host_data in hosts_data:
        cluster_path_list = get_cluster_path_list(host_data)
        # /ppcinv/direct/production/api5_java
        cluster_path = zk_join_path(ZK_HOSTS_DB_PATH, *cluster_path_list)

        host_path_list = get_host_path_list(host_data)
        # /ppcinv/direct/production/api5_java/13
        host_path = zk_join_path(ZK_HOSTS_DB_PATH, *host_path_list)

        # /ppcinv/fqdns/sas1-1923.vm...
        fqdn_path = zk_join_path(ZK_FQDNS_DB_PATH, host_data['fqdn'])

        data = json.dumps(host_data, separators=(',', ':'), ensure_ascii=True)
        zk.ensure_path(cluster_path)
        trx = zk.transaction()
        # записываем путь '/ppcinv/direct/production/api5_java' в '/ppcinv/sas1-1923.vm...', чтобы хосты могли найти себя
        if zk.exists(fqdn_path):
            trx.set(fqdn_path, host_path)
        else:
            trx.create(fqdn_path, host_path)

        # записываем json хоста в '/ppcinv/direct/production/api5_java/13'
        if zk.exists(host_path):
            trx.set(host_path, data)
        else:
            trx.create(host_path, data)
        trx.commit()


def zk_delete_hosts(hosts_data, zk_hosts, node='/ppcinv'):
    if not hosts_data:
        return

    logger = logging.getLogger(__name__)
    zk = KazooClient(hosts=zk_hosts)
    zk.start()

    for host_data in hosts_data:
        host_path_list = get_host_path_list(host_data)
        host_path = zk_join_path(ZK_HOSTS_DB_PATH, *host_path_list)
        fqdn_path = zk_join_path(ZK_FQDNS_DB_PATH, host_data['fqdn'])

        trx = zk.transaction()
        trx.delete(fqdn_path)
        # TODO: что будет, если удалить залоченную ноду?
        trx.delete(host_path)
        # TODO: удалять пустые директории
        trx.commit()


def zk_load_hosts(cluster_conf, zk_hosts):
    logger = logging.getLogger(__name__)
    zk = KazooClient(hosts=zk_hosts)
    zk.start()

    cluster_path = zk_join_path(ZK_HOSTS_DB_PATH, *get_cluster_path_list(cluster_conf))
    hosts_data = []
    if not zk.exists(cluster_path):
        return hosts_data

    for host_alias in zk.get_children(cluster_path):
        host_path = zk_join_path(cluster_path, host_alias)
        logger.debug('Read host alias ' + host_path)

        host_data, _ = zk.get(host_path)
        logger.debug('host data: ' + host_data)
        hosts_data.append(json.loads(host_data))

    return hosts_data


def zk_merge_hosts(merge_actions, zk_hosts):
    zk_delete_hosts(merge_actions['delete'], zk_hosts)
    zk_update_hosts(merge_actions['add'] + merge_actions['update'], zk_hosts)


def zk_get_fqdn_path(fqdn):
    return zk_join_path(ZK_FQDNS_DB_PATH, fqdn)


def zk_get_host_path(zk, fqdn):
    logger = logging.getLogger(__name__)
    try:
        raw_host_path, _ = zk.get(zk_join_path(ZK_FQDNS_DB_PATH, fqdn))
        host_path = raw_host_path.strip()
        logger.debug('Host %s metadata path: %s' % (fqdn, host_path))
    except Exception as e:
        logger.critical('Cannot find host metadata path for %s: %s' % (fqdn, e))
        raise
    return host_path, raw_host_path


def zk_load_host(zk, host_path):
    logger = logging.getLogger(__name__)
    try:
        raw_host_data, _ = zk.get(host_path)
        logger.debug('Raw host data: ' + raw_host_data)
        host_data = json.loads(raw_host_data)
    except Exception as e:
        logger.critical('Cannot fetch host metadata: ' + str(e))
        raise
    logger.debug('Host data: %s' % (host_data,))
    return host_data, raw_host_data
