#! /usr/bin/perl

use strict; 
use warnings;

=head1 DESCRIPTION

    Вычисляет отношение (ratio) между соответствующими элементами числовых серий, заданных в двух файлах.

    На вход получает два файла (одинаковой структуры). 
    Каждая строчка файла разбивается на поля, разделенные пробельными символами. 
    Одно из полей отмечается как ключевое (параметр -k), одно или несколько полей отмечаются для сравнения (-c). 
    Для каждого ключа берется запись из первого файла и запись из второго, 
    соответствующие подлежащие сравнению поля из первой и второй записи делятся друг на друга.
    А именно, второе значение делится на первое, мнемоника "во сколько раз показатель увеличился/уменьшился от первого файла ко второму".
    Ключ должен быть уникальным внутри файла. 

    Скрипт хорошо подходит для обработки данных, полученных yandex-profile-mem-growth.pl

    Сортировку результата по нужной колонке удобно делать через |sort [-n] [-k NN]; вместо -n можно -g -- все нечисловые строки окажутся в начале. 

    Нумерация полей начинается с 1 (мнемоника: как в sort).

   
    Параметры

    -h, --help
        вывод справки

    -k, --key <num>
        ключевое поле

    -c, --compare <num>[,<num>]
        какие поля подлежат сравнению
        -c 2 -c 4 эквиваентно -c 2,4
        по умолчанию -c 1

    -v, --verbose
        включает расширенный вывод; 
        если задан ключ -o, то -v игнорируется

    -v2, --verbose2
        включает расширенный вывод с разностями вместо отношений; 
        если задан ключ -o, то -v2 игнорируется

    -o, --out <шаблон> 
        выводить статистику в соответствии с указанным шаблоном
        При выводе каждой строки в шаблоне делаются замены: 
        $v1 -- значение и первого файла
        $v2 -- значение из второго файла
        $r -- отношение $v2 / $v1
        $d -- разность $v2 - $v1
        \t -- символ табуляции
        По умолчанию: 
         * просто значение отношения ("краткий вывод"), 
         * если указан ключ -v, то дополнительно выводятся соответствующие значения из первого и второго файлов ("расширенный вывод")

    -d, --delim <str>
        разделитель отношений друг от друга и от метки строки
        по умолчанию пробел для краткого формата, а для расширенного - см. вывод с --debug

    --debug
        перед статистикой будет напечатан текущий шаблон форматирования (полезно, если хочется немного поменять умолчальный шаблон)


    Примеры

    > zcat /mnt/direct-logs/ppcsoap01d/ppc.yandex.ru/201111/profile.log.20111117.gz |grep SOAP- |yandex-profile-mem-growth.pl -d -f 13:24 -t 14:05 | sort -n -k 5 > 20111117_ok
    > zcat /mnt/direct-logs/ppcsoap01d/ppc.yandex.ru/201111/profile.log.20111117.gz |grep SOAP- |yandex-profile-mem-growth.pl -d -f 14:24 -t 15:05 | sort -n -k 5 > 20111117_nook
    > tail -n 5 20111117_nook                                                        
    0.03    =       69      /       2723    SOAP-JSON/GetReportList
    0.16    =       437     /       2781    SOAP-JSON/GetCampaignsParams
    0.03    =       90      /       2878    SOAP-JSON/PingAPI
    0.04    =       123     /       2944    SOAP-API/UpdatePrices
    0.93    =       3433    /       3694    SOAP-JSON/GetBannerPhrasesFilter
    
    # отношение суммарного прироста по памяти по методам, краткий вывод, сортировка по отношению:
    > yandex-profile-cmp-stat.pl -c 3 -k 6 20111117_ok 20111117_nook |sort -g 

    # то же самое, но с подробной информацией (значения из первой и второй серий) 
    > yandex-profile-cmp-stat.pl -c 3 -k 6 -v 20111117_ok 20111117_nook |sort -g

    # то же самое, но сортировка по значениям из первой серии
    > yandex-profile-cmp-stat.pl -c 3 -k 6 -v 20111117_ok 20111117_nook |sort -g -k 5

    # отношение двух пар серий: средний и суммарный прирост по памяти
    > yandex-profile-cmp-stat.pl -c 1,3 -k 6 -v 20111117_ok 20111117_nook|sort -g
    > yandex-profile-cmp-stat.pl -c 1 -c 3 -k 6 -v 20111117_ok 20111117_nook|sort -g

=cut

use Getopt::Long;
use Data::Dumper;
use File::Slurp;
use List::MoreUtils qw/uniq pairwise/;

run() unless caller();


sub run
{
    my $opt = parse_options();

    my $stat = calc_stat($opt);

    print_stat($stat, $opt);
}


# Разбирает параметры, подставляет умолчания
sub parse_options
{
    my %O = (
    );

    GetOptions (
        "h|help"       => \&usage,

        "v|verbose"    => \$O{verbose},
        "v2|verbose2"  => \$O{verbose2},
        "c|compare=s@" => \$O{to_compare},
        "k|key=i"      => \$O{key},
        "o|out=s"      => \$O{out},
        "d|delim=s"    => \$O{delimeter},

        "debug"        => \$O{debug},
    );


    if ($O{verbose}){ 
        $O{out} = '$r\t=\t$v2\t/\t$v1' unless defined $O{out};
        $O{delimeter} = "\t; " unless defined $O{delimeter}; 
    } elsif ($O{verbose2}){
        $O{out} = '$d\t=\t$v2\t-\t$v1' unless defined $O{out};
        $O{delimeter} = "\t; " unless defined $O{delimeter}; 
    } else {
        $O{out} = '$r' unless defined $O{out};
        $O{delimeter} = ' ' unless defined $O{delimeter}; 
    }

    $O{to_compare} ||= [1];
    
    @{$O{to_compare}} = split ",", join(',', @{$O{to_compare}});

    die "key field should be specified" unless $O{key};

    die "incorrect number of input files (exactly 2 expected)" unless @ARGV == 2;

    $O{file1} = $ARGV[0];
    $O{file2} = $ARGV[1];

    return \%O;
}


# Печатает usage-сообщение
sub usage {
    system("podselect -section NAME -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8 >&2");
    exit(1);
}


# чтение файлов, подсчет отношений
sub calc_stat
{
    my ($opt) = @_;

    my %RAW_STAT;
    for my $f (qw/file1 file2/){
        my @lines = split "\n", read_file($opt->{$f});
        for my $l (@lines){
            my @fields = split " ", $l;
            unshift @fields, '';
            my $fields_count = @fields;
            my $key = $fields[$opt->{key}];
            die if exists $RAW_STAT{$f}->{$key};
            my @values_to_compare = @fields[@{$opt->{to_compare}}];
            $RAW_STAT{$f}->{$key} = \@values_to_compare;
        }
    }

    my @keys = uniq map { keys %{$RAW_STAT{$_}} } qw/file1 file2/;

    my %STAT;
    for my $key (@keys){
        $STAT{$key}->{values1} = $RAW_STAT{file1}->{$key} || []; 
        $STAT{$key}->{values2} = $RAW_STAT{file2}->{$key} || []; 
        $STAT{$key}->{cmp} = [pairwise { 
            my ($v1, $v2) = ($a, $b);
            $v1 = '-' unless defined $a; 
            $v2 = '-' unless defined $b;
            my $r =  $v1 eq '-' || $v2 eq '-' || $v1 == 0 ? '-' : sprintf( "%.2f", ($v2/$v1));
            my $d =  $v1 eq '-' || $v2 eq '-' ? '-' : sprintf( "%.2f", ($v2 - $v1));
            {v1 => $v1, v2 => $v2, r => $r, d => $d}
        } @{$STAT{$key}->{values1}}, @{$STAT{$key}->{values2}}];
    }

    return \%STAT;
}


# вывод
sub print_stat
{
    my ($stat, $opt) = @_;

    print "current output pattern: '$opt->{out}', delimeter: '$opt->{delimeter}'\n" if $opt->{debug};
    for my $key (sort keys %$stat){
        my $h = $stat->{$key};
        my @cmp_strs = map { 
            my $str = $opt->{out};
            $str =~ s/\$v1/$_->{v1}/g;
            $str =~ s/\$v2/$_->{v2}/g;
            $str =~ s/\$r/$_->{r}/g;
            $str =~ s/\$d/$_->{d}/g;
            $str =~ s/\\t/\t/g;
            $str;
        } @{$stat->{$key}->{cmp}||[]};

        print join($opt->{delimeter}, @cmp_strs, $key)."\n";
    }

    return;
}

