#! /usr/bin/perl

use strict; 
use warnings;

=head1 DESCRIPTION

    Читает Direct-style profile.log'и, выводит статистику залоггированного прироста процессов по памяти. 
    Читает либо STDIN, либо файлы, перечисленные в параметрах (=поведение perl'ового <>).
    Предполагается, что логи в пределах одних суток (учитывается только отметка времени, не дата).

    Сортировку результата по нужной колонке удобно делать через |sort [-n] [-k NN] 

    Параметры

    -h, --help
        вывод справки

    -n, --minutes
        выводить прирост памяти в разбивке по минутам

    -d, --methods
        выводить прирост памяти в разбивке по методам

    -f, --from <hh:mm>
    -t, --to <hh:mm>
        выводить статистику соответственно начиная и заканчивая указанным времнем
        по умолчанию -- 00:00 и 23:59 

    -o, --out <шаблон> 
        выводить статистику в соответствии с указанным шаблоном
        При выводе каждой строки в шаблоне делаются замены: 
        $m -- название метода 
        $s -- суммарный прирост памяти
        $c -- количество вызовов метода/методов
        $a -- средний прирост памяти на один вызов ($s/$c)
        \t -- символ табуляции

    --debug
        перед статистикой будет напечатан текущий шаблон форматирования (полезно, если хочется немного поменять умолчальный шаблон)


    Примеры

    Группировка по методам, время 14:24-15:05, сортировка по среднему:
    cat profile.log.20111117 | yandex-profile-mem-growth.pl -d -f 14:24 -t 15:05 |sort -n

    Группировка по минутам за 14:24-15:05
    cat profile.log.20111117 | yandex-profile-mem-growth.pl -n -f 14:24 -t 15:05

    Группировка по методам за 15:00-16:00, логи с ppcback*, сортировка по суммарному приросту
    yandex-profile-mem-growth.pl -f 15:00 -t 16:00 -d /mnt/direct-logs/ppcback*/ppc.yandex.ru/protected/logs/profile.log.20120110 |sort -n -k 3

    Суммарный рост по минутам за 15:00-16:00, с ppcback* (свой собственный шаблон вывода)
    yandex-profile-mem-growth.pl -f 15:00 -t 16:00 -n -o '$t $s' /mnt/direct-logs/ppcback01*/ppc.yandex.ru/protected/logs/profile.log.20120110

=cut

use Getopt::Long;
use Data::Dumper;
use JSON;

run() unless caller();


sub run
{
    my $opt = parse_options();

    my $raw_stat = read_raw_stat($opt);

    my $stat = calc_stat($raw_stat, $opt);

    print_stat($stat, $opt);
}


# Разбирает параметры, подставляет умолчания
sub parse_options
{
    my %O = (
        from => '00:00',
        to   => '23:59',
    );

    GetOptions (
        "h|help"    => \&usage,
        "n|minutes" => \$O{minutes},
        "d|methods" => \$O{methods},
        "f|from=s"  => \$O{from},
        "t|to=s"    => \$O{to},
        "o|out=s"   => \$O{out},
        "debug"     => \$O{debug},
    );

    $O{out}  = "\$a\\t=\\t\$s\\t/\\t\$c\\t\$m"     if !$O{out} && !$O{minutes} &&  $O{methods};
    $O{out}  = "\$t \$s / \$c = \$a"     if !$O{out} &&  $O{minutes} && !$O{methods};
    $O{out}  = "\$t \$m \$s / \$c = \$a" if !$O{out};

    return \%O;
}


# Печатает usage-сообщение
sub usage {
    system("podselect -section NAME -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8 >&2");
    exit(1);
}


# чтение лога, подсчет простой поминутно-пометодной статистики
sub read_raw_stat
{
    my ($opt) = @_;

    my %RAW_STAT;

    while(my $str = <>){
        # старый профайлинг
        #2011-11-17      00:00:01        SOAP-JSON/GetClientsList        -       total:0.143,u:0.060,s:0.000,cu:0.000,cs:0.000,mem:0
        #my (undef, $time, $method, undef, $stat) = split " ", $str, 5;
        #$time =~ s/:\d+$//;
        #$stat =~ s/^.*mem:(-?[\d]+).*$/$1/;

        my $d = eval{from_json($str)}; 
        next if $@; 
        my $time = $d->[1]; #"2015-08-31 21:00:05.271512"
        $time =~ s/.* ([0-9]{2}:[0-9]{2}).*/$1/;
        next if $time lt $opt->{from} || $time gt $opt->{to};
        my $method = $d->[5];
        my $stat = $d->[14]->{times}->{mem};

        $RAW_STAT{$method}->{$time}->{mem} += $stat;
        $RAW_STAT{$method}->{$time}->{count}++;
    }

    return \%RAW_STAT;
}


# фильтрация по from и to, аггрегация + нули для всех методов/минут без данных
sub calc_stat
{
    my ($raw_stat, $opt) = @_;

    my @methods = keys %$raw_stat;

    my %STAT;
    for my $h (0 .. 23){
        for my $min (0 .. 59){
            my $time = sprintf "%02d:%02d", $h, $min;
            next if $time lt $opt->{from} || $time gt $opt->{to};
            my $time_key = $opt->{minutes} ? $time : "$opt->{from}-$opt->{to}";
            for my $method (@methods){
                my $method_key = $opt->{methods} ? $method : "all_methods";
                for my $k (qw/mem count/){
                    $STAT{$time_key}->{$method_key}->{$k} ||= 0;
                    $STAT{$time_key}->{$method_key}->{$k} += $raw_stat->{$method}->{$time}->{$k} || 0;
                }
            }
        }
    }

    return \%STAT;
}


# вывод 
sub print_stat
{
    my ($stat, $opt) = @_;

    print "current output pattern: '$opt->{out}'\n" if $opt->{debug};
    for my $t (sort keys %$stat){
        for my $m (sort keys %{$stat->{$t}}){
            my $s = $stat->{$t}->{$m}->{mem};
            my $c = $stat->{$t}->{$m}->{count};
            my $a = $c != 0 ? sprintf "%.2f", $s/$c : "-";
            my $ready_line = $opt->{out};
            $ready_line =~ s/\\t/\t/g;
            $ready_line =~ s/\$s/$s/g;
            $ready_line =~ s/\$t/$t/g;
            $ready_line =~ s/\$m/$m/g;
            $ready_line =~ s/\$c/$c/g;
            $ready_line =~ s/\$a/$a/g;

            print "$ready_line\n";
        }
    }
}

