use strict;
use warnings;

package ProjectSpecific;


# $Id$


=head1 NAME

ProjectSpecific -- полезные данные/функции для работы со Стартреком, svn и т.п.

=head1 DESCRIPTION

Полезные данные/функции для удобной работы со Стартреком/svn, специфические для конкретной инсталляции Стартрека или проекта

=cut

use Sys::Hostname;
use Sys::Hostname::FQDN qw/fqdn/;
use Readonly;
use YAML;

use base qw/Exporter/;
our @EXPORT_OK = qw/

    svn_url
    version2rev
    version2svnpath

    default_assignee
    human_readable_project_name
    jira_limtest_ticket_project

    releaser_dbh_config

    tabula_url
    tabula_location
    tabula_testupdate_url

    get_upload_notify_email_to
    get_upload_notify_email_from

    get_jabber_connection_settings

    get_startrek_client_token

    parse_release_summary
    make_release_summary
    change_release_summary

    get_db_conf_aliases
    get_db_conf_file

    get_review_board_server_url

    get_sign_comment
/;

use utf8;
use feature 'state';

our $BASIC_PATH ||= "/etc/project_specific/basic_properties.yaml";
our $EXTENDED_PATH ||= "/etc/project_specific/extended_properties.yaml";

our $PROJECT;
$PROJECT ||= get_default_project();

sub get_default_project
{
    my $p; 
    # переменные окружения
    $p ||= $ENV{PROJECT_FOR_SVN_JIRA_ETC}; ## deprecated, не надо ею пользоваться
    $p ||= $ENV{PROJECT_SPECIFIC};  ## OK, можно пользоваться
    if ($p) {
        return $p;
    }
    # Файловая система: /etc/project_specific/default_project
    my $file = "/etc/project_specific/default_project";
    if ( -f $file ){
        open(my $fh, "<", $file) or die "cannot open < $file: $!";
        my $s = <$fh>;
        $s =~ s/\s+//g;
        return $s;
    }
    return undef;
}

sub get_data {
    my @path = @_;
    state $conf;
    if (!$conf) {
        $conf = YAML::LoadFile($BASIC_PATH);
        if ( -f $EXTENDED_PATH ){
            my $ext_conf = YAML::LoadFile($EXTENDED_PATH);
            merge_conf($conf, $ext_conf, "");
        }
    }
    my $node = $conf;
    while (@path) {
        $node = $node->{ shift(@path) };
    }
    Readonly my $res => $node;
    return $res;
}


sub merge_conf
{
    my ($basic, $ext, $path) = @_;

    if ( ref $basic ne "HASH" || ref $ext ne "HASH" ){
        die "can't merge $basic, $ext";
    }

    for my $k ( sort keys %$ext ){
        if (!exists $basic->{$k}){
            $basic->{$k} = $ext->{$k};
        } elsif ( ref $basic->{$k} ne 'HASH' ){

            # если значения совпадают, ничего не делаем
            my $eq = 1;
            if (!ref $basic->{$k}) {
                $eq = 0 if $basic->{$k} ne $ext->{$k};
            } elsif (ref $basic->{$k} eq 'ARRAY' && ref $ext->{$k} eq 'ARRAY') {
                my @a = @{ $basic->{$k} };
                my @b = @{ $ext->{$k} };
                $eq = 0 if scalar @a != scalar @b;
                for my $i (0..$#a) {
                    last if $eq == 0;
                    $eq = 0 if ($a[$i] ne $b[$i]);
                }
            } else {
                $eq = 0;
            }
            next if $eq;

            die "redefenition of '$path/$k', stop";
        } else {
            merge_conf( $basic->{$k}, $ext->{$k}, "$path/$k" );
        }
    }

    return;
}

sub get_project_data {
    return get_data(@_, $PROJECT) // die "unknown project $PROJECT";
}

#................................................
# SVN
#................................................

=head2 svn_url

    $trunk_svn_url = svn_url('trunk');
    $my_branch_svn_url = svn_url(branch => 'bem-dev-step-zero');
    $release_svn_url = svn_url(release => 12345);
    $limtest_svn_url = svn_url(limtest => 'new_grant_requests', 62240, 62761);

=cut

sub svn_url 
{
    my $alias;
    my $subdir;

    if (@_ == 0) {
        $alias = 'root';
    } elsif (@_ == 1) {
        ($alias) = @_;
    } else {
        if ($_[0] eq 'branch') {
            $alias = 'branches';
            $subdir = $_[1];
        } elsif ($_[0] eq 'release') {
            $alias = 'releases';
            $subdir = "release-$_[1]";  # "release-$release_base_rev"
        } elsif ($_[0] eq 'limtest') {
            $alias = 'limtest';
            $subdir = "$_[1]-$_[2]-$_[3]";  # "$branch-$release_base_rev-$head_rev"
        } else {
            die "svn_url: don't know how to process argument '$_[0]'";
        }
    }

    my $svn_path = get_data('svn_path_for_project',$PROJECT, $alias) // get_data(svn_path => $alias);
    die "svn_url: unknown alias '$alias'" unless defined $svn_path;
    if ($subdir) {
        $svn_path .= "/$subdir";
    }
    return get_project_data('svn_url') . $svn_path;
}

=head2 version2rev

    из номера версии выбирает номер ревизии
    ревизия -- базовая (если $options{rev} eq 'base'; 1.1234.1279-1 => 1234)
    или последняя (если $options{rev} eq 'last'; 1.1234.1279-1 => 1279)

    version2rev($version, rev => base)
    version2rev($version, rev => last)

=cut

sub version2rev
{
    my ($version, %options) = @_;

    $options{name} ||= '';
    $options{rev}  ||= '';

    if ($options{rev} eq 'last'){
        die "incorrect $options{name} version '$version'" unless $version =~ /^(?:1\.)?(?:\d{2,8}\.)?(\d{2,8})(?:-\d)?$/;
        return $1;
    } elsif ($options{rev} eq 'base'){
        die "incorrect $options{name} version '$version'" unless $version =~ /^(?:1\.)?(\d{2,8})(?:\.\d{2,8})?(?:-\d)?$/;
        return $1;
    }

    die "version2rev: incorrect parameters";
}



=head2 version2svnpath



=cut

sub version2svnpath
{
    my ($version) = @_;

    my $base_rev = version2rev($version, rev => 'base');
    my $last_rev = version2rev($version, rev => 'last');
    
    my $svn_path = $base_rev eq $last_rev ? svn_url("trunk") : svn_url(release => $base_rev);

    return $svn_path;
}

=head2 default_assignee

    Вызывается, если при создании тикета не указан исполнитель (assignee)

    Параметры: 
        Джира-объект

    Результат: Джира-логин исполнителя

=cut

sub default_assignee
{
    my $jira_obj = shift;

    return $jira_obj->{'.easy_login'};
}

=head2 jira_limtest_ticket_project

    Возвращает названия проекта в джире, в котором надо создавать тикет на выкладывание кода на среду ограниченного тестирования.

=cut

sub jira_limtest_ticket_project
{
    return get_project_data('jira_limtest_ticket_project');
}

sub releaser_dbh_config
{
    my $data = get_project_data('releaser_dbh_config');
    if (ref $data->{pass} eq 'HASH') {
        my $file = $data->{pass}->{file};
        open my $fh, '<', $file or die "Can't open $file: $!";
        $data->{pass} = <$fh>;
        chomp $data->{pass};
    }
    return $data;
}

sub tabula_url
{
    return get_project_data('tabula_url');
}

sub tabula_location
{
    my ($loc) = @_;

    my $path = (get_data(tabula_location => $loc) || die "unknown location $loc");

    return tabula_url().$path;
}


our %TABULA_TESTUPDATE_SERVERS = (
    Direct => 'testserver=test1&testserver=test-sand',
    Geocontext => 'testserver=test1',
    Directmod => 'testserver=test1',
);
sub tabula_testupdate_url
{
    my $version = shift or die "empty version";
    my $ts = get_project_data('tabula_testupdate_servers');
    return sprintf("%s/steady?%s&version=%s", tabula_location("testupdate"), $ts, $version);
}

=head2 get_upload_notify_email_to

    Возвращает адрес, на который надо отправлять уведомления для админов о том, что надо что-то выложить. 

=cut

sub get_upload_notify_email_to
{
    return get_project_data('upload_notify_email_to');
}


=head2 get_upload_notify_email_from

    Возвращает адрес, который надо поставить в поле From уведомления для админов о том, что надо что-то выложить. 

=cut

sub get_upload_notify_email_from
{
    return get_project_data('upload_notify_email_from');
}

=head2 get_jabber_connection_settings

    Возвращает ссылку на хеш с параметрами соединения с сервером джаббера, используемыми для отправки уведомлений.
    В хеше буду ключи:
        username
        password
        componentname
        xmpp_server_hostname
        xmpp_server_port

=cut

sub get_jabber_connection_settings
{
    return get_project_data('jabber_connection_settings');
}

=head2 get_startrek_client_token

    Получить OAuth-токен для доступа к Стартреку.
    Именованные параметры:
        
        file    -- имя файла, из которого нужно прочитать токен, по умолчанию $HOME/.startrek_client_token.
                   Если файла с таким именем не существует, токен запрашивается на oauth.yandex-team.ru и сохраняется во вновь созданный файл с таким именем; при этом с клавиатуры запрашивается пароль пользователя.

=cut

sub get_startrek_client_token {
    my %O = @_;
    my $file = $O{file} || "$ENV{HOME}/.startrek_client_token";

    my $token;
    if (-e $file) {
        open(my $fh, '<', $file) or die "Cannot open $file for reading: $!";
        chomp($token = <$fh>);
        if (!$token) {
            die "File $file is empty!";
        }
        close $fh;
    } else {
        my $login = [getpwuid($<)]->[0] or die "Cannot get user login";
        my $password;
        print "Please enter your domain password: ";
        system("stty -echo");
        chop($password=<STDIN>);
        print "\n";
        system("stty echo");

        require Yandex::OAuth;

        $Yandex::Log::LOG_ROOT = '/';
        $Yandex::OAuth::OAUTH_URL = 'https://oauth.yandex-team.ru';
        %Yandex::OAuth::OAUTH_LOG_SETTINGS = ( log_file_name => '/dev/null' );
        my $response = Yandex::OAuth::oa_method(
            '/token',
            {
                grant_type => 'password',
                username   => $login,
                password   => $password,
            },
            request_method => 'POST',
            header => {
                'Authorization' => "Basic " . get_project_data('startrek_client_auth'),
                'Content-Type'  => "application/x-www-urlencoded",
            },
        );
        $token = $response->{access_token};

        if (open(my $fh, '>', $file)) {
            print $fh $token;
            close $fh;
            chmod 0400, $file;
            print "Your Startrek OAuth token saved to $file\n";
        } else {
            warn "Cannot open $file for writing: $!";
        }
    }
    return $token;
}

sub make_last_releases_query {
    my %O = @_;
    my @query_parts = ("Queue: " . get_project_data('default_tracker_queue'), "Type: release");
    $O{component} ||= eval { get_project_data('startrek_release_component') };
    if ($O{component}) {
        push @query_parts, qq/Components: "$O{component}"/;
    }
    push @query_parts, '"Sort by": Key DESC';
    return join(' ', @query_parts);
}

=head2 parse_release_summary

=cut

sub parse_release_summary {
    my ($summary) = @_;
    my $re = get_project_data('release_summary_regexp');

    # сейчас для Модерации всегда будет $migration_title == undef
    my ($name, $version, $migration_title) = $summary =~ /$re/;
    $migration_title ||= '';
    $migration_title =~ s/^\s*\+\s*//;

    my $details = {
        version => $version,
        name => $name,
        has_migration => $migration_title ? 1 : 0,
    };
    $details->{migration_title} = $migration_title if $migration_title;

    return $details;
}

=head2 make_release_summary

=cut

sub make_release_summary {
    my %O = @_;
    my $format = get_project_data('release_summary_format');

    my $summary = sprintf $format, $O{name}, $O{version};
    my $section_migration = $O{has_migration} ? ($O{migration_title} || "инструкции по выкладке") : "";
    $summary .= " + $section_migration" if $section_migration;

    return $summary;
}

=head2 change_release_summary

=cut

sub change_release_summary
{
    my ($summary, %O) = @_;

    my $details = parse_release_summary($summary);

    $details->{$_} = $O{$_} for keys %O;
    my $new_summary = make_release_summary(%$details);

    return $new_summary;
}

=head2 beta_url

=cut

sub beta_url {
    my ($port) = @_;
    my $num = int( ($port - 7000) / 1000 );
    my $url = get_project_data('beta_url');
    for ($url) {
        s/\$PORT(?![A-Z])/$port/g;
        s/\$NUM(?![A-Z])/$num/g;
    }
    return $url;
}


=head2 get_db_conf_aliases

=cut

sub get_db_conf_aliases {
    state $aliases = do {
        my $conf = get_project_data('db_config');
        my %aliases = map {($_ => $conf->{$_}->{aliases} || [])} keys %$conf;
        \%aliases;
    };
    return $aliases;
}


=head2 get_db_conf_file

=cut

sub get_db_conf_file {
    my ($conf_name) = @_;

    state $conf_file = do {
        my $conf = get_project_data('db_config');
        my %conf_file = map {my $c = $conf->{$_}; map {($_ => $c->{conf_file})} ($_, @{$c->{aliases}||[]})} keys %$conf;
        \%conf_file;
    };

    return $conf_file->{$conf_name};
}

=head2 get_review_board_server_url

    Возвращает url, относительно которого нужно заливать diff-файлы на review board. 

=cut

sub get_review_board_server_url
{
    return get_project_data('review_board_server_url');
}

=head2 get_sign_comment

    Возвращает строку с подписью робота для стартрека (для комментариев и тд) 

=cut


sub get_sign_comment
{
    my $script_name = shift;
    my $hostname = `hostname -f`;
    return "----\nСкрипт $script_name с машины $hostname";
}

1;
