#!/usr/bin/python
# -*- coding: utf-8 -*-
# vim: set expandtab:tabstop=4:softtabstop=4:shiftwidth=4

from tempfile import NamedTemporaryFile
import unittest
import re
import os
import sys

class ProjectSpecificSvnTest(unittest.TestCase):
    def setUp(self):
        lib_path = os.path.dirname(__file__) + '/../usr/local/lib/python2.7/dist-packages'
        lib_path = os.path.abspath(lib_path)
        if sys.path[0] != lib_path:
            sys.path.insert(0, lib_path)
        from project_specific import ProjectSpecificSettings
        config_text = u'''---
svn_url:
    Direct: 'svn+ssh://s.y.ru/d'
svn_path:
    root: ''
    trunk: '/t'
    branches: '/b'
    releases: '/r'
    limtest: '/l'
'''
        self.tmpfile = NamedTemporaryFile(mode='w+', delete=False)
        self.tmpfile.write(config_text)
        self.tmpfile.close()
        self.settings_obj = ProjectSpecificSettings(basic_path=self.tmpfile.name, project='Direct')
    
    def tearDown(self):
        os.remove(self.tmpfile.name)

    def test_root_url(self):
        self.assertEqual(self.settings_obj.svn_root_url(), 'svn+ssh://s.y.ru/d')
    def test_trunk_url(self):
        self.assertEqual(self.settings_obj.svn_trunk_url(), 'svn+ssh://s.y.ru/d/t')
    def test_branch_url(self):
        self.assertEqual(self.settings_obj.svn_branch_url('my-branch'), 'svn+ssh://s.y.ru/d/b/my-branch')
    def test_release_url(self):
        self.assertEqual(self.settings_obj.svn_release_branch_url('12345'), 'svn+ssh://s.y.ru/d/r/release-12345')
        self.assertEqual(self.settings_obj.svn_release_branch_url(12345), 'svn+ssh://s.y.ru/d/r/release-12345')
    def test_limtest_branch_url(self):
        self.assertEqual(self.settings_obj.svn_limtest_branch_url('my-branch', 123, 456), 'svn+ssh://s.y.ru/d/l/my-branch-123-456')
        self.assertEqual(self.settings_obj.svn_limtest_branch_url('my-branch', '123', 456), 'svn+ssh://s.y.ru/d/l/my-branch-123-456')
        self.assertEqual(self.settings_obj.svn_limtest_branch_url('my-branch', '123', '456'), 'svn+ssh://s.y.ru/d/l/my-branch-123-456')

    def test_root_url_relative(self):
        self.assertEqual(self.settings_obj.svn_root_url(relative=True), '/')
    def test_trunk_url_relative(self):
        self.assertEqual(self.settings_obj.svn_trunk_url(relative=True), '/t')
    def test_branch_url_relative(self):
        self.assertEqual(self.settings_obj.svn_branch_url('my-branch', relative=True), '/b/my-branch')
    def test_release_url_relative(self):
        self.assertEqual(self.settings_obj.svn_release_branch_url('12345', relative=True), '/r/release-12345')
        self.assertEqual(self.settings_obj.svn_release_branch_url(12345, relative=True), '/r/release-12345')
    def test_limtest_branch_url_relative(self):
        self.assertEqual(self.settings_obj.svn_limtest_branch_url('my-branch', 123, 456, relative=True), '/l/my-branch-123-456')
        self.assertEqual(self.settings_obj.svn_limtest_branch_url('my-branch', '123', 456, relative=True), '/l/my-branch-123-456')
        self.assertEqual(self.settings_obj.svn_limtest_branch_url('my-branch', '123', '456', relative=True), '/l/my-branch-123-456')

    def test_branch_url_re(self):
        self.assertEqual(self.settings_obj.svn_branch_url_re(), re.escape('svn+ssh://s.y.ru/d/b/') + '([^/]+)')
    def test_release_url_re(self):
        self.assertEqual(self.settings_obj.svn_release_branch_url_re(), re.escape('svn+ssh://s.y.ru/d/r/release-') + '([0-9]+)')
    def test_limtest_branch_url_re(self):
        self.assertEqual(self.settings_obj.svn_limtest_branch_url_re(), re.escape('svn+ssh://s.y.ru/d/l/') + '([^/]+)-([0-9]+)-([0-9]+)')

class ProjectSpecificRedefinedSvnPathsTest(unittest.TestCase):
    def setUp(self):
        lib_path = os.path.dirname(__file__) + '/../usr/local/lib/python2.7/dist-packages'
        lib_path = os.path.abspath(lib_path)
        if sys.path[0] != lib_path:
            sys.path.insert(0, lib_path)
        from project_specific import ProjectSpecificSettings
        config_text = u'''---
svn_url:
    Direct: 'svn+ssh://a.y.ru/arc'
    Directmod: 'svn+ssh://s.y.ru/d-m'
svn_path:
    root: ''
    trunk: '/t'
    branches: '/b'
    releases: '/r'
    limtest: '/l'
svn_path_for_project:
    Direct:
        trunk: '/trunk/arcadia/direct/perl'
        branches: '/branches/direct/perl'
        releases: '/branches/direct/release/perl'
        limtest: '/branches/direct/release/limtest'
'''
        self.tmpfile = NamedTemporaryFile(mode='w+', delete=False)
        self.tmpfile.write(config_text)
        self.tmpfile.close()
        self.settings_obj_direct = ProjectSpecificSettings(basic_path=self.tmpfile.name, project='Direct')
        self.settings_obj_directmod = ProjectSpecificSettings(basic_path=self.tmpfile.name, project='Directmod')
    
    def tearDown(self):
        os.remove(self.tmpfile.name)

    def test_trunk_url(self):
        self.assertEqual(self.settings_obj_direct.svn_trunk_url(), 'svn+ssh://a.y.ru/arc/trunk/arcadia/direct/perl')
        self.assertEqual(self.settings_obj_directmod.svn_trunk_url(), 'svn+ssh://s.y.ru/d-m/t')
    def test_branch_url(self):
        self.assertEqual(self.settings_obj_direct.svn_branch_url('my-branch'), 'svn+ssh://a.y.ru/arc/branches/direct/perl/my-branch')
        self.assertEqual(self.settings_obj_directmod.svn_branch_url('my-branch'), 'svn+ssh://s.y.ru/d-m/b/my-branch')

class ProjectSpecificArcadiaTest(unittest.TestCase):
    def setUp(self):
        lib_path = os.path.dirname(__file__) + '/../usr/local/lib/python2.7/dist-packages'
        lib_path = os.path.abspath(lib_path)
        if sys.path[0] != lib_path:
            sys.path.insert(0, lib_path)
        from project_specific import ProjectSpecificSettings
        config_text = u'''---
svn_url:
    Direct: 'svn+ssh://arcadia.yandex.ru/arc'
svn_path:
    root: ''
    trunk: '/trunk/arcadia/direct/perl'
    branches: '/branches/direct/task'
    releases: '/branches/direct/release'
    limtest: '/branches/direct/limtest'
'''
        self.tmpfile = NamedTemporaryFile(mode='w+', delete=False)
        self.tmpfile.write(config_text)
        self.tmpfile.close()
        self.settings_obj = ProjectSpecificSettings(basic_path=self.tmpfile.name, project='Direct')
    
    def tearDown(self):
        os.remove(self.tmpfile.name)

    def test_root_url(self):
        self.assertEqual(self.settings_obj.svn_root_url(), 'svn+ssh://arcadia.yandex.ru/arc')
    def test_trunk_url(self):
        self.assertEqual(self.settings_obj.svn_trunk_url(), 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/direct/perl')
    def test_branch_url(self):
        self.assertEqual(self.settings_obj.svn_branch_url('DIRECT-54321'), 'svn+ssh://arcadia.yandex.ru/arc/branches/direct/task/DIRECT-54321')
    # тестов на релизные ветки ещё нет, svn_release_branch_url с Аркадией будет работать по-другому

    def test_root_url_relative(self):
        self.assertEqual(self.settings_obj.svn_root_url(relative=True), '/')
    def test_trunk_url_relative(self):
        self.assertEqual(self.settings_obj.svn_trunk_url(relative=True), '/trunk/arcadia/direct/perl')
    def test_branch_url_relative(self):
        self.assertEqual(self.settings_obj.svn_branch_url('DIRECT-54321', relative=True), '/branches/direct/task/DIRECT-54321')

    def test_branch_url_re(self):
        self.assertEqual(self.settings_obj.svn_branch_url_re(), re.escape('svn+ssh://arcadia.yandex.ru/arc/branches/direct/task/') + '([^/]+)')

if __name__ == '__main__':
    unittest.main()
