#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
Работа с juggler
"""

import json
import re
import socket
import time
import requests

from subprocess import check_output

JUGGLER_API_URL = 'http://localhost:31579/events' # 'http://juggler-push.search.yandex.net/events'
JCLIENT_API_BINARY = '/usr/bin/jclient-api'
CLIENT_NAME = 'direct_juggler'

TIMEOUT = 3
TRIES = 3
RETRY_DELAY = 0.6

NODATA_TO_STATUS = {
    'force_ok': 'OK',
    'force_crit': 'CRIT',
    'skip': None,
}

STATUS_TO_MASK = {
    'OK':   0,
    'WARN': 1,
    'CRIT': 2,
}

MASK_TO_STATUS = {
    0: 'OK',
    1: 'WARN',
    2: 'CRIT',
}

def queue_events(events):
    """
    Отправка событий (в формате load_events) в juggler
    """
    data = {'events': fix_events(events)}
    data['source'] = CLIENT_NAME

    for retry in range(1, TRIES + 1):
        try:
            reply = requests.post(JUGGLER_API_URL, json=data, timeout=TIMEOUT)
            reply.raise_for_status()
            for event_status in reply.json()["events"]:
                if event_status["code"] != 200:
                    raise Exception(event_status["error"])
            break
        except (requests.HTTPError, requests.exceptions.ConnectionError, requests.exceptions.Timeout) as e:
            if retry >= TRIES:
                raise
            time.sleep(retry * RETRY_DELAY)
            continue

    return reply.json()


def aggregate_events(events, aggregator, aggr_service):
    """
    Агрегирует (logic_and/or) события из заданного списка (в формате load_events) в новое событие с сервисом aggr_service
    """
    if not events:
        return None

    aggr_mask = None
    aggr_desc = []
    for event in sorted(events, key=lambda x: x['service']):
        mask = STATUS_TO_MASK[event['status']]
        aggr_desc.append('%s;%d;%s' % (event['service'], mask, event['description']))
        if (aggr_mask is None) or (aggregator == 'logic_or' and mask > aggr_mask) or (aggregator == 'logic_and' and mask < aggr_mask):
            aggr_mask = mask

    event = {'service': aggr_service, 'status': MASK_TO_STATUS[aggr_mask], 'description': 'aggr_desc: ' + ' --- '.join(aggr_desc)}
    return event


def filter_events(events, services, nodata_mode='force_crit', as_regex=False):
    """
    Фильтрует события (в формате load_events) по заданным services (list/tuple of strings).
    Если сервиса нет в events, применяется nodata_mode
    services могут восприниматься как рег. выражения (as_regex).
    """
    assert isinstance(services, list) or isinstance(services, tuple)

    filtered = {}
    for service in services:
        selected = []
        for event in events:
            if (not as_regex and event['service'] == service) or (as_regex and re.match(service, event['service'])):
                selected.append(event)

        if not selected and nodata_mode != 'skip':
            selected.append({'service': service, 'status': NODATA_TO_STATUS[nodata_mode], 'description': 'autogenerated from nodata_mode'})

        for event in selected:
            filtered[event['service']] = event

    return filtered.values()


def load_events():
    """
    Загружает из локального кеша juggler-client события (print-events)
    Возвращает cписок [{'service': ..., 'status': ..., 'description': ..., ...}, ...]
    """
    jclient_events = check_output([JCLIENT_API_BINARY, 'print-events'])
    jclient_events = json.loads(jclient_events)
    return jclient_events


def events_to_json(events, add_source=False):
    """
    json для отправки в juggler-client bundle (не нужно указывать source!), или напрямую в api (желательно указать source)
    """
    data = {'events': fix_events(events)}
    if add_source:
        data['source'] = CLIENT_NAME

    return json.dumps(data)


def fix_events(events):
    """
    Приводим events к виду, который устроит juggler api
    """
    assert isinstance(events, list) or isinstance(events, tuple)

    # тут не нужно никакой специальной логики по добавлению 'host' в ивенты, где он отсутствует, juggler-client добавит сам
    # явные ошибки в полях тоже проверяет сам juggler
    for event in events:
        # description нужен всегда, если его нет - juggler отвечает что-то вроде
        # 200 OK "status: 500, no descr" и не принимает event
        # лучше его неявно проставлять в status
        event['description'] = event.get('description', event['status'])[0:1024]
        # ограничения тоже лучше поправить тут, чем получить ошибку от juggler
        event['service'] = event['service'][0:128]
        if event.get('tags'):
            event['tags'] = [x[0:128] for x in event['tags']]

    return events
