# -*- coding: utf-8 -*-
import urllib

from requests import Session
from requests.exceptions import RequestException

from yamb.errors import (
    CODE_ERROR_MAP,
    REQUESTS_ERROR_MAP,
    STATUS_CODE_ERROR_MAP,
    HTTPError,
)
from yamb.utils import url_join

__all__ = (
    'Bot',
)


DEFAULT_REQUESTS_SESSION = Session()


class Bot(object):
    def __init__(self,
                 token,
                 base_uri='https://bots.yamb.ws.yandex.ru/bot/',
                 requests_session=DEFAULT_REQUESTS_SESSION,
                 verify=True):
        self.token = token
        self.base_uri = base_uri
        self.requests_session = requests_session
        self.verify = verify

    def me(self, timeout=None):
        """
        https://github.yandex-team.ru/pages/yamb/yamb-bots/api/bot/methods/#informatsiia-o-bote
        """
        return self._request('me', timeout=timeout)['data']

    def send_message(self, text, chat_id=None, user_id=None, organization_id=None, timeout=None):
        """
        https://github.yandex-team.ru/pages/yamb/yamb-bots/api/bot/methods/#otpravka-tekstovogo-soobshcheniia
        """
        if not chat_id and not user_id:
            raise ValueError('You must specify "chat_id" or "user_id"')

        if chat_id:
            data = {
                'chat_id': chat_id,
                'text': text,
            }
        else:
            data = {
                'user_id': user_id,
                'text': text,
                'organization_id': organization_id,
            }

        self._request(
            'send-message',
            method='post',
            json=data,
            timeout=timeout
        )

    def updates(self, offset=None, chat_id=None, timeout=None):
        """
        https://github.yandex-team.ru/pages/yamb/yamb-bots/api/bot/methods/#poluchenie-obnovlenii
        """
        return self._request(
            'updates',
            method='get',
            params={
                'offset': offset,
                'chat_id': chat_id,
            },
            timeout=timeout
        )['data']

    def create_private_chat(self, user_id, organization_id, timeout=None):
        """
        https://github.yandex-team.ru/pages/yamb/yamb-bots/api/bot/methods/#sozdanie-privatnogo-chata
        """
        response = self._request(
            'create-private-chat',
            method='post',
            json={
                'user_id': user_id,
                'organization_id': organization_id,
            },
            timeout=timeout,
        )
        return {
            'created': response['status_code'] == 201,
            'result': response['data']
        }

    def create_group_chat(self,
                          organization_id,
                          name,
                          description,
                          administrators,
                          members,
                          timeout=None):
        """
        https://github.yandex-team.ru/pages/yamb/yamb-bots/api/bot/methods/#sozdanie-gruppovogo-chata
        """
        response = self._request(
            'create-group-chat',
            method='post',
            json={
                'organization_id': organization_id,
                'name': name,
                'description': description,
                'administrators': administrators,
                'members': members,
            },
            timeout=timeout,
        )
        return response['data']

    def chat(self, chat_id, timeout=None):
        """
        https://github.yandex-team.ru/pages/yamb/yamb-bots/api/bot/methods/#poluchenie-informatsii-o-chate
        """
        return self._request(
            'chats/%s' % urllib.quote_plus(chat_id),
            method='get',
            timeout=timeout,
        )['data']

    def _request(self,
                 path,
                 method='get',
                 json=None,
                 params=None,
                 timeout=None):
        try:
            response = getattr(self.requests_session, method.lower())(
                url=url_join(
                    force_trailing_slash=True,
                    parts=(self.base_uri, path)
                ),
                json=json,
                params=params,
                timeout=timeout,
                headers={
                    'Authorization': 'Token %s' % self.token
                },
                verify=self.verify,
            )
            response.raise_for_status()
        except RequestException as e:
            error_class = None
            if e.response is not None:
                # trying to get error class by the status code
                error_class = STATUS_CODE_ERROR_MAP.get(e.response.status_code)

                # trying to get error class by the code
                try:
                    code = e.response.json().get('code')
                except ValueError:
                    # For the 500 error we can get html response.
                    # E.g. '<h1>Server Error (500)</h1>'
                    pass
                else:
                    if code:
                        error_class = CODE_ERROR_MAP.get(code) or error_class
            error_class = (
                error_class or REQUESTS_ERROR_MAP.get(type(e)) or HTTPError
            )
            raise error_class(requests_error=e)
        else:
            return {
                'status_code': response.status_code,
                'headers': response.headers,
                'data': response.json(),
            }
