# -*- coding: utf-8 -*-
import inspect

from requests.exceptions import ConnectionError as RequestsConnectionError
from requests.exceptions import ConnectTimeout as RequestsConnectTimeout
from requests.exceptions import ReadTimeout as RequestsReadTimeout


class YambError(Exception):
    """Base Yamb exception.

    Could be used for catching all the Yamb specific exceptions.

    ```python
    try:
        bot.me()
    except YambError as e:
        print 'Yamb specific exception raised'
    ```
    """
    code = None
    status_codes = None
    requests_errors = None
    message = 'Yamb error'
    errors = {}


class HTTPError(YambError):
    """Base exception for errors produced at the HTTP layer.

    These types of exceptions containes additional parameters:

    * `requests_error` - original [requests exception][requests-exception].
    * `errors` - list of errors.

    [requests-exception]: http://docs.python-requests.org/en/master/api/#exceptions
    """  # noqa
    message = 'HTTP error'

    def __init__(self, requests_error):
        # todo: test me
        self.requests_error = requests_error
        if self.requests_error.response is not None:
            try:
                response_data = self.requests_error.response.json()
            except ValueError:
                # For the 500 error we can get html response.
                # E.g. '<h1>Server Error (500)</h1>'
                pass
            else:
                # todo: use new format https://st.yandex-team.ru/YAMB-1992
                message = (
                    response_data.get('detail') or
                    response_data.get('error')
                )
                code = response_data.get('code')
                if message:
                    self.message = message
                if code:
                    self.code = code
                if not message and not code:
                    self.errors = response_data

    def __str__(self):
        return self.__repr__()

    def __repr__(self):
        error = self.message
        if self.errors:
            error = '%s:\n%s' % (
                error,
                '\n'.join([
                    '* "%s" - %s' % (key, ' '.join(value))
                    for key, value in self.errors.items()
                ])
            )
        return error


class AuthError(HTTPError):
    """Errors due to invalid authentication credentials."""
    status_codes = [401]
    message = 'Auth error'


class ForbiddenError(HTTPError):
    status_codes = [403]
    message = 'Forbidden error'


class NotFoundError(HTTPError):
    status_codes = [404]
    message = 'Not found error'


class ValidationError(HTTPError):
    status_codes = [400, 422]
    message = 'Validation error'


class BotIsNotInOrganization(ValidationError):
    message = 'Bot is not in organization'
    code = 'bot_is_not_in_organization'


class UserIsNotInOrganization(ValidationError):
    message = 'User is not in organization'
    code = 'user_is_not_in_organization'


class ConflictError(HTTPError):
    status_codes = [409]
    message = 'Conflict error'


class ConnectionError(HTTPError):
    """A Connection error occurred."""
    requests_errors = [RequestsConnectionError]
    message = 'Connection error'


class Timeout(HTTPError):
    """The request timed out.
    Catching this error will catch both
    [ConnectTimeout](#connecttimeout) and
    [ReadTimeout](#readtimeout) errors.
    """
    message = 'Timeout'


class ConnectTimeout(ConnectionError, Timeout):
    """The request timed out while trying to connect to the remote server.
    Requests that produced this error are safe to retry.
    """
    requests_errors = [RequestsConnectTimeout]
    message = 'Connect timeout'


class ReadTimeout(Timeout):
    """The server did not send any data in the allotted amount of time."""
    requests_errors = [RequestsReadTimeout]
    message = 'Read timeout'


STATUS_CODE_ERROR_MAP = {}
CODE_ERROR_MAP = {}
REQUESTS_ERROR_MAP = {}


# todo: test me
exceptions = [
    i for i in locals().values()
    if inspect.isclass(i) and issubclass(i, YambError)
]
for exception_class in exceptions:
    if exception_class.code:
        CODE_ERROR_MAP[exception_class.code] = exception_class
    if exception_class.status_codes:
        # don't register exception classes with the error codes
        # for the status codes
        if exception_class.code:
            continue
        for status_code in exception_class.status_codes:
            if status_code in STATUS_CODE_ERROR_MAP:
                raise ValueError(
                    (
                        'Trying to register status code %s for '
                        'the %s, but it is already registered for the %s'
                    ) % (
                        status_code,
                        exception_class,
                        STATUS_CODE_ERROR_MAP[status_code],
                    )
                )
            STATUS_CODE_ERROR_MAP[status_code] = exception_class
    if exception_class.requests_errors:
        for requests_error in exception_class.requests_errors:
            if requests_error in REQUESTS_ERROR_MAP:
                raise ValueError(
                    (
                        'Trying to register requests error %s for '
                        'the %s, but it is already registered for the %s'
                    ) % (
                        requests_error,
                        exception_class,
                        REQUESTS_ERROR_MAP[requests_error],
                    )
                )
            REQUESTS_ERROR_MAP[requests_error] = exception_class
