#!/usr/bin/perl

# $Id$

=head1 NAME

    quasi-make.pl

    "Как бы make" -- легкое и perl-центричное определение и выполнение целей и правил в духе make
    В отличие от make, правила достижения цели выполняются всегда целиком, накаких ленивых проверок не делается.

=head1 DESCRIPTION

    Скрипт читает указанные файлы с описанием целей (они же действия) и выполняет все, что перечислено в правиле для указанной цели (действия).
    Пример: 
        quasi-make.pl -f ./dev/SampleRules.pm date
    -- описания действий взять из файла ./dev/SampleRules.pm и выполнить действие date

    Имеет смысл определять shell'ные алиасы или скрипты-обертки с нужным набором правил и дефолтной целью

    Формат правил см. ниже. 

    Специальные действия (цели):
    CHECK -- проверяет консистентность указанного набора правил.
    LIST -- выводит список доступных действий (целей), полезно для генерации шелльных автоподстановок
    help -- показывает справку по quasi-make.pl

    Описания действий:
    Цели (они же действия) не рекомендуется называть в ВЕРХНЕМ_РЕГИСТРЕ -- такие цели могут в будущем использоваться как специальные.
    Цели не должны совпадать по названию с предопределенными специальными целями.

    Файл, указанный в -f, должен называться <что-нибудь>.pm и определять Pelr-пакет с именем <что-нибудь>. 
    В этом пакете должен быть определен метод actions, который не ждет никаких параметров и возвращает ссылку на хеш: 
    {
        # вызов внешней программы
        tags => "ctags-direct.sh",
        # именованная функция
        apache_check => \&apache_check,
        # анонимная функция
        settings_devtest => sub { switch_settings_local("DevTest") },
        # ссылка на другое действие
        devtest => "action:conf_devtest",
        # несколько действий подряд
        conf_devtest => [
            "action:settings_devtest", 
            "action:apache_restart",
        ],
        # смешанный массив
        multiaction => [
            "ctags-direct.sh",
            \&apache_check,
            sub { switch_settings_local("DevTest") },
            "action:conf_devtest",
        ],
        # Самый общий случай -- хеш. Правила выполнения должны быть по ключу todo
        # TODO описание других ключей
        multiaction2 => {
            todo => [
                "ctags-direct.sh",
                \&apache_check,
                sub { switch_settings_local("DevTest") },
                "action:conf_devtest",
            ],
        }
    }

    perl-функции могут дергать выполнение другого действия через функцию perform_action

    Опции

    -h, --help
        справка

    -C, --directory <path>
        перед работой сменить каталог на <path> (аналогично tar-у)

    -f, --file <path>
        файл с определениями действий (целей)
        может быть использована несколько раз: 
        -f Rules1.pm -f Rules2.pm
        Если одно и то же действие определяется в нескольких файлах -- приоритет у файла, указанного позднее (правее в командной строке)
        Важно! Опция -C (сменить рабочий каталог) обрабатывается до -f, так что путь в -f должен быть абсолютный или относительно -C

    -v, --verbose
        "болтливый" режим -- выводить подробную информацию о порядке выполнения действий

    -q, --quiet
        "молчаливый" режим 
        (включен по умолчанию)

    --lock <lock_file_name>
        брать эксклюзивный лок на файл /var/lock/quasi-make/<lock_file_name>
        если лок взят другим процессом, скрипт не дожидается освобождения лока, а сразу падает.

    --die-on-errors
        при неудачном завершении (ненулевой код завершения) действия сразу умирать, а не собирать сообщения об ошибках

    TODO 
    * возможность выполнять несколько действий за раз
        Варианты: 
        -a <action1> -a <action2>
        -m action1 action2 
    * Дефолтное действие
        Варианты: 
        глобальное умолчательное действие DEFAULT
        параметр --default <action> -- тогда его можно задавать в direct-mk 

=head1 COMMENTS

    
=cut

use strict;
use warnings;

use B qw(svref_2object);
use Cwd;
use Fcntl qw/:flock/;

use Getopt::Long;
use Data::Dumper;

use Yandex::MetricsLogSender;

use utf8;
use open ':std' => ':utf8';

our $recursion_level;
our $step;
our $DRY_RUN;
our $DIE_ON_ERRORS;
our @accumulated_errors;
our $LOCK_FILE_DIR = '/var/lock/quasi-make';
our $metrics = new Yandex::MetricsLogSender(
    'quasi-make.action',
    {
        path => Cwd::getcwd(),
    });

our %ACTIONS = (
    LIST => {
        todo => \&list,
        no_metrics => 1,
        quiet => 1,
    },
    CHECK => {
        todo => \&check,
        no_metrics => 1,
        quiet => 1,
    },
    help => {
        todo => \&help,
        no_metrics => 1,
        quiet => 1,
    },
);


{
my $VERBOSE = 0;

sub p{print " "x($recursion_level-1),@_,"\n" if $VERBOSE;}

sub set_verbose{$VERBOSE = shift;}
}

run() unless caller();

sub run
{
    $recursion_level = 0;
    $step = 0;
    my $opts = parse_options();

    if ($opts->{chdir_to}){
        chdir $opts->{chdir_to} or die "can't cd '$opts->{chdir_to}'";
    }

    my $lock_fh;
    my $lock_file_path;
    if ($opts->{lock}) {
        if (!-d $LOCK_FILE_DIR) {
            mkdir $LOCK_FILE_DIR;
            chmod 0777, $LOCK_FILE_DIR;
        }
        $lock_file_path = $LOCK_FILE_DIR . '/' . $opts->{lock};
        open $lock_fh, '>', $lock_file_path or die "can't open $lock_file_path: $!";
        flock($lock_fh, LOCK_EX | LOCK_NB) or die "can't lock $lock_file_path: $!";
    }

    set_verbose($opts->{verbose});
    $DRY_RUN = $opts->{dry_run};
    $DIE_ON_ERRORS = $opts->{die_on_errors};
    collect_actions(%$opts);

    my $primary_action = shift @ARGV;
    $primary_action ||= 'DEFAULT';

    perform_action($primary_action);
    eval {
        $metrics->flush();
        1;
    } or error("Can't flush metrics: $@");
    if ($lock_file_path) {
        close $lock_fh or die "can't close $lock_file_path: $!";
        unlink $lock_file_path or warn "can't unlink $lock_file_path: $!";
    }

    if (@accumulated_errors) {
        warn "$_\n" foreach @accumulated_errors;
        exit scalar(@accumulated_errors);
    }
}

sub perform_action
{
    my ($action) = @_;
    my $action_key = _get_action_key($action);
    $recursion_level++;
    $step++;
    my $start_time = Time::HiRes::time();
    die "too deep recursion, stopped\n" if $recursion_level > 20;
    die "too many actions, stopped\n" if $step > 120;

    die "don't know how to make $action\n" if !exists $ACTIONS{$action_key};
    my $desc = $ACTIONS{$action_key};
    die "wrong description for action $action, stopped" unless ref $desc eq "HASH";

    my $from = exists $desc->{source_file} ? "from file $desc->{source_file}" : "built-in";
    p("action $action ($from)");
    my $alarm_clock = exists $desc->{timeout} ? $desc->{timeout} : 1200;

    my $todo = ref $desc->{todo} eq "ARRAY" ? $desc->{todo} : [$desc->{todo}];
    for my $cmd (@$todo){
        local $SIG{ALRM} = sub {print "$action timed out\n"; kill -9 => $$};
        alarm $alarm_clock;
        execute($cmd);
        alarm 0;
    }

    add_metrics($action_key, Time::HiRes::time() - $start_time);
    $recursion_level--;
}

sub execute
{
    my ($cmd) = @_;
    $step++;

    p("cmd ".(ref $cmd eq "CODE" ? what_sub_is_this($cmd)."()" : $cmd));

    if ( !ref $cmd && (my ($act) = $cmd =~ /^action:(.*)$/ ) ){
        local @ARGV = @ARGV;
        if ($act =~ /\s+/) {
            my @old_argv = @ARGV;
            @ARGV = split /\s+/, $act;
            $act = shift @ARGV;
            push @ARGV, @old_argv;
        }
        perform_action($act);
    } elsif ( !ref $cmd ){
        if (!$DRY_RUN){
            system($cmd);
            my $status = $? >> 8;
            error("command '$cmd' exited with non-zero status $status") if $status != 0;
        }
    } elsif ( ref $cmd eq 'CODE' ){
        $cmd->(\@ARGV) if !$DRY_RUN;
    }  else {
        die;
    }

    return;
}

sub list 
{
    print join "", map { "$_\n" } sort keys %ACTIONS;
}

sub help
{
    usage();
}

sub check
{
    for my $act (sort keys %ACTIONS){
        $recursion_level = 0;
        $step = 0;
        local $DRY_RUN = 1;
        eval {
            perform_action($act);
        };
        die "problem in action $act:\n$@\n" if $@;
    }
    return '';
}


sub parse_options
{
    my %O = (
        verbose => 0,
        dry_run => 0,
        die_on_errors => 0,
    );

    GetOptions(
        "help"                => \&usage,
        "f|file=s@"           => \$O{file},
        "v|verbose"           => \$O{verbose},
        "q|quiet"             => sub { $O{verbose} = 0 },
        "n|dry-run"           => \$O{dry_run},
        "die-on-errors"       => \$O{die_on_errors},
        "lock=s"              => \$O{lock},
        "C|directory=s"       => \$O{chdir_to},
        #"action=s@"           => \$O{action},
        #"m|multi"             => \$O{multi},
    ) or die "can't parse options";

    return \%O;
}


sub collect_actions
{
    my %O = @_;

    my %SPECIAL_ACTION = map {$_ => 1} keys %ACTIONS;

    for my $file ( @{$O{file}} ){
        #p($file);
        my ($package) = ($file =~ m!^(?:.*/|)(\w+)\.pm$!);
        #p("package $package");
        next unless $package;

        require $file;

        my $actions_from_file = $package->actions();
        for my $act (keys %$actions_from_file) {
            my $key = _get_action_key($act);
            die "Built-in action $act redefined in file $file. Stopped" if $SPECIAL_ACTION{$key};
            $ACTIONS{$key} = ref $actions_from_file->{$act} eq 'HASH' ? $actions_from_file->{$act} : {todo => $actions_from_file->{$act}};
            $ACTIONS{$key}->{source_file} = $file;
        }
        # создаем в загруженном пакете функции perform_action и execute
        eval "*${package}::perform_action = \\&perform_action";
        die $@ if $@;
        eval "*${package}::execute = \\&execute";
        die $@ if $@;
    }

    return '';
}

sub add_metrics {
    my ($action, $duration) = @_;
    if ($ACTIONS{$action} && $ACTIONS{$action}->{no_metrics}) {
        return;
    }
    $metrics->add($action, $duration, {
        recursion_level => $recursion_level,
        step => $step,
    });
}

sub _get_action_key
{
    my ($act) = @_;
    $act =~ tr/_/-/;
    return $act;
}

sub usage
{
    system("podselect -section NAME -section DESCRIPTION $0 | pod2text-utf8 >&2");
    exit(0);
}


sub what_sub_is_this {
    my $coderef = shift;
    return svref_2object( $coderef )->GV->NAME;
}

sub error
{
    my ($message) = @_;

    if ($DIE_ON_ERRORS) {
        die $message;
    } else {
        warn $message;
        push @accumulated_errors, $message;
    }
}
