#!/usr/bin/python
# -*- coding: utf-8 -*-
# vim: set expandtab ts=4 sw=4:
# $Id$

import collections
import errno
import jinja2
import os
import re
import stat
import sys

def render(src_file, dest_file, vars):
    tmpl_obj = jinja2.Template(open(src_file, 'r').read().decode('utf-8'))
    try:
        dest_dir = os.path.dirname(dest_file)
        os.makedirs(dest_dir)
    except OSError as e:
        if e.errno == errno.EEXIST:
            pass
        else:
            raise
    with open(dest_file, 'w') as f:
        f.write(tmpl_obj.render(vars).encode('utf-8'))
    if os.stat(src_file).st_mode & stat.S_IXUSR:
        os.chmod(dest_file, 0755)

def run():
    common_vars = {
        'instance': 'ppc-cache',
    }
    shard_vars = collections.OrderedDict()
    base_port = 6410
    for shard in range(1, 4):
        shard_vars[shard] = common_vars.copy()
        shard_vars[shard].update({
            'shard': shard,
            'port': base_port + shard,
        })
        shard_vars[shard]['opt_dir'] = '/opt/redis/{instance}-{port}'.format(**shard_vars[shard])
    dirs_to_create = [
        'etc/redis',
        'var/log/redis'
    ] + [x['opt_dir'].strip('/').format(**x) for x in shard_vars.values()]
    for d in dirs_to_create:
        try:
            os.makedirs(d)
        except OSError as e:
            if e.errno == errno.EEXIST:
                pass
            else:
                raise
    template_to = {
        'run': 'etc/sv/redis-{instance}-{shard}/run',
        'redis.conf': 'etc/redis/{instance}-{shard}.conf',
        'redis_monrun.conf': 'etc/monrun/conf.d/redis-{instance}-{shard}.conf',
        'redis_graphite.conf': 'etc/yandex-du-redis-graphite/conf.d/redis-{instance}-{port}.conf',
        'redis_cluster_monrun.conf': 'etc/monrun/conf.d/redis-ppc-cache-cluster.conf',
        'postinst': 'actions/postinst',
        'prerm': 'actions/prerm',
    }
    global_vars = common_vars.copy()
    global_vars.update({'shard_vars': shard_vars})
    for tmpl in template_to:
        tmpl_file = os.path.join('tmpl', tmpl)
        if re.search(r'\{shard\}|\{port\}', template_to[tmpl]):
            for shard in shard_vars:
                dest_file = template_to[tmpl].format(**shard_vars[shard])
                render(tmpl_file, dest_file, shard_vars[shard])
        else:
            dest_file = template_to[tmpl].format(**common_vars)
            render(tmpl_file, dest_file, global_vars)

if __name__ == '__main__':
    run()

