# -*- coding: utf-8 -*-

import json
import yaml
import sys
import time
import argparse
import httplib
import os
import re
import zlib
import requests
import fcntl
import smtplib
import socket
import datetime
import errno
from email.mime.text import MIMEText
from kazoo.client import KazooClient, KazooState

class RestarterZookeeperLockNotAcquired(Exception):
    def __init__(self, zk_node_path):
        self.zk_node_path = zk_node_path

class RestarterState:
    # здесь были старые хосты, убрал чтобы не смущали
    # новые не добавил, потому что пакет больше не используется
    zk_servers = []

    def __init__(self, zk_node_path):
        self.zk_node_path = zk_node_path
        self.zk = KazooClient(','.join(self.zk_servers))
        self.zk_lock = self.zk.Lock(zk_node_path, "lock")

    def load(self):
        self.zk.start()
        self.zk.ensure_path(self.zk_node_path)
        if not self.zk_lock.acquire(blocking=False):
            raise RestarterZookeeperLockNotAcquired(self.zk_node_path)
        state_json, stat = self.zk.get(self.zk_node_path)
        if state_json:
            state_struct = json.loads(state_json)
        else:
            state_struct = {}
        self.zk_node_version = stat.version
        self.last_hash_sum = int(state_struct.get('hash', None))
        self.last_release_tag = state_struct.get('lastReleaseTag', None)
        self.last_restart = state_struct.get('lastRerun', {})
        self.last_running_state = state_struct.get('lastRunningState', {})

    def save(self):
        state_json = json.dumps({
                'hash': self.last_hash_sum,
                'lastReleaseTag': self.last_release_tag,
                'lastRerun': self.last_restart,
                'lastRunningState': self.last_running_state
            }, sort_keys=True, indent=4, separators=(',', ': '))
        self.zk.set(self.zk_node_path, state_json, self.zk_node_version)

    def is_connected_to_zk(self):
        return self.zk.state == KazooState.CONNECTED

def log(msg):
    global LOG_FILE
    LOG_FILE.write(('%s: %s\n' % (time.asctime(),  msg)).encode('utf-8'))

def email_log(msg):
    global EMAIL_LOG
    EMAIL_LOG += ('%s: %s\n' % (time.asctime(),  msg)).encode('utf-8')

def make_request(method, domain, path):
    conn = httplib.HTTPConnection(domain, 80, timeout=30)
    conn.request(method, path)
    return conn.getresponse()

#######################################################################################################################
# json для каждого запуска имеет следующую структуру
# {
#  ...
#  pack: {
#         ...
#         projects: [
#           ...
#           launchSuites: [
#               {
#                   suite: {
#                       name: имя тестового класса,
#                   ...
#                   }
#                   ...
#                   startTime: timestamp
#                   launchStatus: статус теста
#               },
#               {
#               }
#               ...
# Данная функция пробегает по всем объектам projects и вложенным launchSuites
# так как упавшие тесты в релизе перезапускаются, то в разных запусках могут быть одни и те же классы, поэтому
# данная функция сохраняет информацию только о самом последнем запуске каждого тестового класса
def process_launch_by_time(packs, pack_name, launch):
    for project in launch['pack']['projects']:
        for suite in project['launchSuites']:
            suite_name = suite['suite']['name']
            if suite_name in packs[pack_name]:
                suite_info = packs[pack_name][suite_name]
                if suite['startTime'] > suite_info['start_time']:
                    suite_info['start_time'] = suite['startTime']
                    suite_info['status'] = suite['launchStatus']
            else:
                packs[pack_name][suite_name] = {'start_time': suite['startTime'], 'status': suite['launchStatus']}

# Функция пробегает по словарю и для кажого тестового пака подсчитывает количество различных статусов
def build_by_time_report(packs):
    report = {}
    for pack_name in packs:
        report[pack_name] = {'FAILED': 0, 'RUNNING': 0, 'REVOKED': 0, 'RUNNABLE': 0, 'FINISHED': 0}
        for suite_name in packs[pack_name]:
            report[pack_name][packs[pack_name][suite_name]['status']] += 1
    return report

def prepare_for_sending(report):
    timestamp = int(time.time())
    row_report = []
    for pack_name in report:
        for result in report[pack_name]:
            row_report.append((re.sub('\W+', '_', pack_name), result, report[pack_name][result], timestamp))
    return row_report

def get_by_time_report(response):
    return prepare_for_sending(build_by_time_report(get_last_packs_state(response)))


def get_last_packs_state(response):
    packs = {}
    if len(response['launches']) == 0:
        return []
    for launch in response['launches']:
        pack_name = launch['pack']['name']
        if pack_name not in packs:
            packs[pack_name] = {}
        if pack_name in packs:
            process_launch_by_time(packs, pack_name, launch)
    return packs

#######################################################################################################################
# Строит отчет по результатам первого пуска, рассчитывает сколько длилось прохождение каждого пака, и сколько тестов

def for_every_pack_was_recived_result(packs):
    for i in packs:
        if not 'execution_result' in packs[i]: return False
    return True

def build_report(packs):
    report = []
    timestamp = int(time.time())
    for x in packs:
        for status in packs[x]['execution_result']:
            report.append((x, status, packs[x]['execution_result'][status], timestamp))
        report.append((x, 'duration',  packs[x]['report_duration'], timestamp))
    return report

def get_first_launch_report(response):
    packs = {}

    for l in response['launches']:
        pack_name = l['pack']['name']
        if pack_name not in packs:
            packs[pack_name] = {'start_time' : sys.maxint}

        if (pack_name in packs and l['launchStatus'] == 'REPORT_READY'
            and l['createdTime'] < packs[pack_name]['start_time']):
            packs[pack_name]['start_time'] = l['createdTime']
            packs[pack_name]['execution_result'] = {'passed_first': l['passedSuites'], 'failed_first': l['failedSuites']}
            td = l['reportStopTime'] - l['startTime']
            packs[pack_name]['report_duration'] = str(td)
    if not for_every_pack_was_recived_result(packs) or len(packs.keys()) == 0 : return []

    return build_report(packs)

#######################################################################################################################
def restart_launch(id):
    return make_request("GET", "aqua.yandex-team.ru", "/aqua-api/services/launch/%s/restart?failed-only=true" % id)

def set_tag(id, tag):
    return make_request("PUT", "aqua.yandex-team.ru", "/aqua-api/services/launch/addtag/%s/%s" % (id, tag) )

def kill_launch(id):
    return make_request("DELETE", "aqua.yandex-team.ru", "/aqua-api/services/launch/%s" % id)

# Проверяет что в current_result и last_result падения одинаковы
def is_same_fail(current_result, last_result):
    same_failes = 0
    for x in current_result:
        if x in last_result:
            same_failes += 1
    return same_failes == len(last_result)

# добавляет в отчет информацию о запуске
def process_launch(packs, pack_name, launch):
    if int(launch['startTime']) > packs[pack_name]['startTime']:
        packs[pack_name]['startTime'] = int(launch['startTime'])
        packs[pack_name]['launchId'] = launch['id']
        packs[pack_name]['status'] = launch['launchStatus']
        for project in launch['pack']['projects']:
            for suite in project['launchSuites']:
                if suite['launchStatus'] in ('FAILED', 'REVOKED'):
                    packs[pack_name]['suites'].append(suite['suite']['name'])

# Строит отчет с информацией о состоянии тестов в текущий момент
# при построении отчета использует информацию из последнего запуска для каждого пака
def get_failed_suites_report(response, pack_names):
    if len(pack_names) != 0:
        packs = dict((x, {'startTime': 0, 'suites': []}) for x in pack_names)
    else:
        packs = {}
    if len(response['launches']) == 0:
        return []
    for launch in response['launches']:
        pack_name = launch['pack']['name']
        if len(pack_names) == 0 and pack_name not in packs:
            packs[pack_name] = {'startTime': 0, 'suites': []}
        if pack_name in packs:
            process_launch(packs, pack_name, launch)
    return packs

#Считает количество тестов, которые еще не завершились
def count_runs(response):
    runs = 0
    for launch in response['launches']:
        runs += int(launch['runningSuites'])
    return runs

def restart_tests(response, packs, last_restart, tag):
    total_running = count_runs(response)    #количество бегущих тестов
    log(u'Всего активных тестов %d' % total_running)
    report = get_failed_suites_report(response, []) #строим отчет по упавшим тестам
    for pack in report:
        failed_suites = report[pack]['suites']
        launch_id = report[pack]['launchId']
        has_failed_suites = len(failed_suites) != 0
        too_many_restarts = pack in last_restart and last_restart[pack]['restartTime'] >= MAX_RESTART_TIMES
        has_stable_fails = pack in last_restart and is_same_fail(failed_suites, last_restart[pack]['suites'])
        if pack in packs and report[pack]['status'] == 'REPORT_READY' and has_failed_suites and (pack not in last_restart or (not too_many_restarts and not has_stable_fails)):
            if len(failed_suites) + total_running < MAX_RUNNING_TESTS:
                log(u'Перезапускаем пак id=%s ' % launch_id)
                restart_response = restart_launch(launch_id)
                if restart_response.status == httplib.OK:
                    if pack not in last_restart:
                        last_restart[pack] = {'restartTime': 0}
                    last_restart[pack]['restartTime'] += 1
                    new_launch_id = json.load(restart_response)['id']
                    set_tag(new_launch_id, tag)
                    log(u'Пак %s, id=%s успешно перезапущен, id перезапуска %s' % (pack, launch_id, new_launch_id))
                    last_restart[pack]['suites'] = failed_suites #запоминаем упавшие в этот раз тесты, понадобится при следующем запуске
                    last_restart[pack]['logFlag'] = False #Флаг нужен для того, что бы в лог вывести сообщание об окончании запусков только один раз
                    total_running += len(failed_suites)
                else:
                    log(u'Ошибка при перезапуске пака %s, id=%s, ответ сервера: %s' % (pack, launch_id, restart_response.read()))
        elif report[pack]['status'] == 'REPORT_READY' and pack in last_restart and not last_restart[pack]['logFlag']:
            last_restart[pack]['logFlag'] = True
            if not has_failed_suites:
                reason = u'успешно прошел'
            elif too_many_restarts:
                reason = u'был перезапущен максимальное число раз'
            else:
                reason = u'имеет устойчивые падения тестов'
            last_restart[pack]['reasonForStoppingRestarting'] = reason
            msg = u'Пак %s, id=%s, %s, больше не перезапускаем' % (pack, launch_id, reason)
            log(msg)
            email_log(msg)
    return last_restart

def check_release_stat(profile, tag):
    if profile == 'all':
        profiles = PROFILES.keys()
    else:
        profiles = profile.split(',')
    response = json.load(get_release_launches(tag))
    packs = {}
    for launch in response['launches']:
        pack_name = launch['pack']['name']
        if  get_profile_by_packname(pack_name, profiles) is None: continue
        if pack_name not in packs:
            packs[pack_name] = {}
        if pack_name in packs:
            process_launch_by_time(packs, pack_name, launch)
    profile_stat_dict = dict((key, {'profile_name': key, 'total': 0}) for key in profiles)
    for pack_name in packs:
        profile_result = profile_stat_dict[get_profile_by_packname(pack_name, profiles)]
        profile_result['total'] += len(packs[pack_name])
        for case_name in packs[pack_name]:
            case = packs[pack_name][case_name]
            profile_result[case['status']] =  profile_result.get(case['status'], 0) + 1

    result = []
    for profile_name in profile_stat_dict:
        profile_result = profile_stat_dict[profile_name]
        profile_result['is_passed'] = profile_result.get('FINISHED', -1) == profile_result['total']
        result.append(profile_result)

    print json.dumps(result)





def get_profile_by_packname(pack_name, profiles):
    for profile in profiles:
        try:
            if pack_name in PROFILES[profile]:
                return profile
        except KeyError as e:
            print(u'Указан несуществующий профайл %s, используйте опцию --show-profiles для ознакомления со списком профайлов' % profile)
            exit(1)
    return None

#######################################################################################################################
PROFILES = {
        'api_java': [
            u'Autotests Direct.API5 AudienceTargets',
            u'Autotests Direct.API5 RetargetingLists',

            u'Autotests Direct.API5 Clients',
            u'Autotests Direct.API5 AgencyClients',

            u'Autotests Direct.API5. KeywordsResearch',

            u'Autotests Direct.API5. Bids',
            u'Autotests Direct.API5. Sitelinks',
            u'Autotests Direct.API5. VCards',
            u'Autotests Direct.API5. AdExtensions',
            ],
        'api': [
            u'Autotests Direct.API5. Ads',
            u'Autotests Direct API Mediaplan (Media)',
            u'Autotests Direct API Mediaplan (Teamleader)',
            u'Autotests Direct. API Images',
            u'Autotests Direct. API Retargeting',
            u'Autotests Direct.API Finance',
            u'Autotests Direct.API. Banners',
            u'Autotests Direct.API. Campaigns',
            u'Autotests Direct.API. Clients',
            u'Autotests Direct.API. Sandbox',
            u'Autotests Direct.API. Statistics',
            u'Autotests Direct.API5. AdGroups',
            u'Autotests Direct.API5. AdImages',
            u'Autotests Direct.API5. Ads',
            u'Autotests Direct.API5. BidModifiers',
            u'Autotests Direct.API5. Campaigns',
            u'Autotests Direct.API5. Changes',
            u'Autotests Direct.API5. Dictionaries',
            u'Autotests Direct.API5. Keywords',
            u'Autotests Direct.API5 Reports',
            u'Autotests Direct.API5 Reports: back-to-back tests',
            ],
        'cmd': [
            u'Direct Backend',
            ],
        'transport': [
            u'Autotests Direct B2B BS Transport',
            u'Autotests Direct BS Transport Full Export',
            u'Autotests Direct B2B Transport Moderation',
            u'Autotests Direct BS Transport',
            u'Autotests Direct Moderation Transport',
            u'directmoderation: Интеграция с Директом',
            ],
        'intapi': [
            u'Autotests Direct INTAPI'
            ]
        }

# Достает отчет из аквы по тегу
def get_release_launches(release_tag):
    return make_request("GET", "aqua.yandex-team.ru", "/aqua-api/services/launch/tag/%s" % release_tag)

def ping_host():
    if STAGE == '':
        return httplib.OK
    else:
        return requests.get(STAGE, verify=False).status_code

# делает парсер для разбора аргументов командной строки
def create_parser():
    parser = argparse.ArgumentParser()
    parser.add_argument('-p', '--packs', default="", help='список паков для которых нужна статистика, пример "Autotests Direct.API5. Ads, Autotests Direct.API Finance"')
    parser.add_argument('-t', '--tag', help='релизный тег', required=True)
    parser.add_argument('-r', '--restart', help='при наличии флага скрипт перезапускает паки', action='store_true')
    parser.add_argument('-s', '--stat', help='при наличии флага скрипт отсылает статистику в графит', action='store_true')
    parser.add_argument('-n', '--namespace', default="",  help='префикс добавляемый ко всем именам файлов, производимых скриптом')
    parser.add_argument('-gp', '--graphite-prefix', default="", help='префикс для отсылки данных в графит')
    parser.add_argument('-ld', '--log-dir', default="", help='директория для логов')
    parser.add_argument('-szkn', '--state-zk-parent-node', help='zookeeper-нода, в дочерних нодах которой хранится состояние')
    parser.add_argument('-st', '--stage', default="", help='адрес тестируемой среды, если 500-тит не перезапускаем тесты')
    parser.add_argument('-mt', '--max-tests', default=700, type=int, help='максимальное количество тестов, идущих в момент времени')
    parser.add_argument('-mp', '--max-pack-restart', default=5, type=int,  help='максимальное количество попыток перезапуска одного пака')
    parser.add_argument('-ap', '--all-packs', action='store_true', help='для всех паков в релизе, при указании параметра нельзя использовать параметр --packs')
    parser.add_argument('-rap', '--restart-after-pack', default="", help='начинать перезапуск только после запуска указанного пака')
    parser.add_argument('-e', '--email', default="", help='email для отправки отчета о перезапуске')
    parser.add_argument('-cs', '--current-state', action='store_true', help='показать текущую статистику')
    parser.add_argument('-pr', '--profile', default="all", help='профайл по которому необходимо отразить статистику, по дефолту статитска по всем пускам')
    parser.add_argument('-j', '--json', action='store_true', help = 'выводить текущую статистику в json') #DIRECT-83546
    parser.add_argument('-sp', '--show-profiles', action='store_true', help='вывести список доступных профайлов')
    parser.add_argument('-mrt', '--max-running-timeout', default=0, type=int, help='устанавливает таймаут в минутах после которого пак будет ревоукнут и перезапущен в случае зависания')
    parser.add_argument('--opt-out', default='', help='opt-out список паков, которые нельзя убивать как зависшие')
    parser.add_argument('--only-check-if-stopped', action='store_true', help='проверить, будут ли ещё перезапускаться тесты или нет по этому тэгу. Код выхода 0, если перезапусков больше не будет, не 0, если ещё будут или по этому тегу нет запусков или состояния')
    return parser

# Отсылает отчет демону который отсылает все это в графит
def send_report(row_report):
    sock = socket.socket()
    sock.connect((HOST, PORT))
    for row in row_report:
        sock.send(row)
    sock.close()

# Графит ждет от нас данные в формате ТЕКСТОВАЯ_МЕТКА ЗНАЧЕНИЕ ТАЙМСТАМП
# эта функция форматирует параметры в соответствии с требуем форматом
def to_graphite_row(key1, key2, value, timestamp):
    return ('%s.%s.%s %s %s\n' % (GRAPHITE_PREFIX, re.sub('\W+', '_', key1), key2, value, timestamp)).lower()

# рассчитывает crc32 на основании отчета по времени
def count_hash(report):
    report_slice = [x[:3] for x in report]
    report_slice.sort(key=lambda x: (x[0], x[1]))
    hashsum = 0
    for i in report_slice:
        hashsum = zlib.adler32(str(i), hashsum)
    return hashsum

# определяет изменилось ли состояние всех тестов или нет
def is_new_state(hashsum, last_hash_sum):
    if last_hash_sum is None or hashsum != last_hash_sum:
        return True
    else:
        return False

def get_params():
    args = [arg.decode('utf-8') for arg in sys.argv[1:]]
    try:
        conf_index = args.index('--conf')
        with open(args[conf_index + 1], 'r') as param_file:
            content = yaml.load(param_file)
        args.pop(conf_index)
        args.pop(conf_index)
    except IOError:
        log(u'Невозможно открыть файл %s' % args[conf_index + 1])
        exit(1)
    except Exception:
        content = {}

    params = []
    for k, v in content.items():
        if isinstance(v, bool):
            if v:
                params.append('--' + k)
        else:
            if k == 'packs' or k == 'opt-out':
                v = ','.join(v)
            params.append('--' + k)
            if isinstance(v, int):
                v = str(v)
            params.append(v)
#размещаем список из командой строки после списка из файла,
#в таком случае, параметры из командной строки у нас в приоритете,
#параметры флаги при этом комбинируются
    params.extend(args)
    return create_parser().parse_args(params)

def get_file_name(params):
    file_name = ''
    if params.namespace != '':
        file_name = params.namespace + '_'
    file_name += os.path.splitext(os.path.basename(__file__))[0]
    if params.stat:
        file_name += '_stat'
    else:
        file_name += '_restart'
    return file_name

def get_all_release_packs(response):
    packs = set()
    for launch in response['launches']:
        pack_name = launch['pack']['name']
        if pack_name not in packs:
            packs.add(pack_name)
    return [x for x in packs]


def kill_all_dead_packs(response, last_running_state, packs, timeout, opt_out):
    launch_states = {}
    for launch in response['launches']:
        if launch['launchStatus'] not in ('REPORT_READY', 'REPORT_REQUESTED', 'REPORT_FAILED') and launch['pack']['name'] not in opt_out:
            launch_states[launch['id']] = {
                    'hash': count_launch_state_hash(launch['pack']),
                    'time': time.time(),
                    }
   # Удаляем уже прошедшие паки
    for launch_id in last_running_state:
        if launch_id not in launch_states:
            last_running_state[launch_id] = None

    for launch_id in launch_states:
        last_state = last_running_state.pop(launch_id, None)
        current_state = launch_states[launch_id]
        if last_state:
            if last_state['hash'] != current_state['hash']:
                continue
            diff = current_state['time'] - float(last_state['time'])
            if diff > timeout * 60:
                log(u'запуск с id: %s не изменял свое состояние более %f секунд, убиваем его' % (launch_id, diff))
                response = kill_launch(launch_id)
                if response.status != httplib.OK:
                    log(u"Аква ответила кодом %s на попытку остановить пак id %s" % (response.status, launch_id))
            current_state['time'] = last_state['time']

    return launch_states

def count_launch_state_hash(launch):
    suites = []
    for projects in launch['projects']:
        for launchSuite in projects['launchSuites']:
            suites.append(launchSuite['suite']['name'] + '_' + launchSuite['launchStatus'])
    suites.sort()
    hashsum = 0
    for s in suites:
        hashsum = zlib.adler32(s, hashsum)
    return hashsum



if __name__ == "__main__":
    params = get_params()
    FILE_NAME = get_file_name(params)

# хост и порт демона, который умеет отсылать данные в графит, установлен на всех ppcdev-ах
    HOST = 'localhost'
    PORT = 42000
    MAX_RUNNING_TESTS = params.max_tests
    MAX_RESTART_TIMES = params.max_pack_restart
    STAGE = params.stage
    FROM_ADDR = 'release-restart@' + socket.gethostname() + '.yandex.ru'
    LOG_FILE = sys.stdout
    EMAIL_LOG = ''

    if params.log_dir != '':
        if not os.path.exists(params.log_dir):
            os.makedirs(params.log_dir)
        LOG_FILE =  open(os.path.join(params.log_dir, '%s_%s.log' % (FILE_NAME, time.strftime('%Y%m%d'))), 'a')

    if params.current_state:
        check_release_stat(params.profile, params.tag)
        exit()

    if params.show_profiles:
        for profile in PROFILES:
            print('%s:' % profile.encode('utf-8'))
            for pack in PROFILES[profile]:
                print('\t%s' % pack.encode('utf-8'))
        exit()

    if not (params.stat or params.restart or params.only_check_if_stopped):
        log(u'Необходимо указать как минимум один из флагов --stat/--restart/--only-check-if-stopped')
        exit(1)
    if params.restart and params.packs == '' and not params.all_packs:
        log(u'При указании режима --restart необходимо явно указать список паков для перезапуска с помощью параметра --packs или --all_packs')
        exit(1)
    if params.all_packs and params.packs != '':
        log(u'Одновременное использование параметров --packs --all-packs недопустимо')
        exit(1)

    # для --stat состояние как будто не нужно, но отдельное может создаваться (см. get_file_name), нужно разобраться
    if params.state_zk_parent_node:
        state_zk_node = params.state_zk_parent_node + '/' + FILE_NAME
    else:
        state_zk_node = '/direct/release-stat-restart/' + FILE_NAME
    state = RestarterState(state_zk_node)
# префикс ипользуются при отсылке в графит
    if params.stat and params.graphite_prefix == '':
        log(u'При запуске с флагом --stat необходимо указать префикс для отправки данных в графит')
        exit(1)
    GRAPHITE_PREFIX = params.graphite_prefix

# Устанавливаем блокировку, лишь один экземпляр скрипта с данным режимом в заданном namespace может быть запущен в один момент
    lock_file = open('/var/lock/%s.lock' % FILE_NAME, 'w')
    try:
        fcntl.flock(lock_file, fcntl.LOCK_EX | fcntl.LOCK_NB)
    except IOError:
        log(u"Can't lock file %s, stop" % lock_file)
        exit(1)

    if params.packs == '':
        packs = []
    else:
        packs = params.packs.split(',')

    if params.opt_out == '':
        opt_out = []
    else:
        opt_out = params.opt_out.split(',')

    last_hash_sum = None
    last_release_tag = None
    last_restart = {}
    last_running_state = {}

    try:
        state.load()
    except RestarterZookeeperLockNotAcquired as e:
        log(u"Can't get Zookeeper lock, node '{}', stop".format(e.zk_node_path))
        exit(1)
    last_hash_sum = state.last_hash_sum
    last_release_tag = state.last_release_tag
    last_restart = state.last_restart
    last_running_state = state.last_running_state

    response = json.load(get_release_launches(params.tag))

    if params.only_check_if_stopped and (not response['launches'] or last_release_tag != params.tag):
        exit(111)

    # Костыль: если тесты запустили слишком поздно, то с момента запуска тестов до момента запуска этого скрипта может поменяться дата и за переданный тег ничего не найдётся.
    # Поэтому дополнительно ищем запуски за "вчера"
    m = re.match(r'direct-release-([0-9]{4})-([0-9]{2})-([0-9]{2})', params.tag)
    yesterday_tag = None
    if m:
        year, month, day = map(int, m.groups())
        yesterday = datetime.date(year, month, day) - datetime.timedelta(1)
        yesterday_tag = yesterday.strftime('direct-release-%Y-%m-%d')

    if len(response['launches']) == 0: #Если сегодня нет релиза
        if yesterday_tag:   # вдруг был вчера, а мы пропустили
            response = json.load(get_release_launches(yesterday_tag))
            if len(response['launches']) == 0:
                log(u'Релизов по тегам %s, %s не найдено, выходим' % (params.tag, yesterday_tag))
                exit(0)
            last_release_tag = yesterday_tag
        elif last_release_tag is None:  #и релиза не было на нашей памяти
            log(u'Релиза по тегу %s не найдено, выходим' % params.tag)
            exit(0)
        else: #был релиз когда то в прошлом
            response = json.load(get_release_launches(last_release_tag)) #грузим его
    else:
        if last_release_tag != params.tag:
            last_hash_sum = None
            last_restart = {}
            last_running_state = {}
        last_release_tag = params.tag

    if params.only_check_if_stopped:
        state_report = build_by_time_report(get_last_packs_state(response))
        packs_in_restarter = [p for p in last_restart if p in packs]
        other_packs = [p for p in state_report if p in packs and p not in packs_in_restarter]
        packs_in_restarter_in_progress = [p for p in packs_in_restarter if 'reasonForStoppingRestarting' not in last_restart[p]]
        # могут упасть и попасть в перезапускалку потом
        other_packs_in_progress = [p for p in other_packs if state_report[p]['RUNNING'] + state_report[p]['RUNNABLE'] > 0]
        # уже упали, но рестартилка ещё не дошла
        recently_failed_packs = [p for p in other_packs if state_report[p]['FAILED'] + state_report[p]['REVOKED'] > 0]
        exit_code = min(len(packs_in_restarter_in_progress) + len(other_packs_in_progress) + len(recently_failed_packs), 99)
        exit(exit_code)

    log(u'Обрабатываем релиз с тегом: %s' % last_release_tag)

    all_packs = get_all_release_packs(response)
    if params.all_packs:
        packs = all_packs

    time_report = get_by_time_report(response) #строим для него отчет во времени
    hs = count_hash(time_report)
    if is_new_state(hs, last_hash_sum): #если что то поменялось, то есть релиз не закончен и тесты еще бегают
        if params.stat: send_report([to_graphite_row(*i) for i in time_report]) #шлем в графит отчет
        last_hash_sum = hs

    report = get_first_launch_report(response) #строим отчет по первому пуску
    if len(report) != 0 and params.stat: #если построился (Для этого надо, что бы все отслеживаемые паки, запущенные в первый раз уже прошли полностью)
        send_report([to_graphite_row(*i) for i in report]) #то шлем отчет в графит
    last_pack = params.restart_after_pack

    if float(params.max_running_timeout) > 0:
        if not state.is_connected_to_zk():
            log(u'Соединение с Zookeeper потеряно, завершаемся')
            exit(1)
        last_running_state = kill_all_dead_packs(response, last_running_state, packs, float(params.max_running_timeout), opt_out)

    if params.restart and (last_pack == "" or last_pack in all_packs):
        if not state.is_connected_to_zk():
            log(u'Соединение с Zookeeper потеряно, завершаемся')
            exit(1)
        ts_response_code = ping_host()
        if ts_response_code == httplib.OK:
            last_restart = restart_tests(response, packs, last_restart, last_release_tag)
        else:
            log(u'ТС отвечает кодом %d, не перезапускаем тесты' % ts_response_code)

    state.last_hash_sum = last_hash_sum
    state.last_release_tag = last_release_tag
    state.last_restart = last_restart
    state.last_running_state = last_running_state
    state.save()
# отправляем отчет по почте
    if params.email != '' and EMAIL_LOG != '':
        server = smtplib.SMTP('localhost')
        msg = MIMEText(EMAIL_LOG, 'plain', 'utf-8')
        server.sendmail(FROM_ADDR, params.email, msg.as_string())
        server.quit()
