package Release::Check;

# $Id$

=head1 NAME
    
    Release::Check -- проверить консистенстность релиза

=head1 DESCRIPTION


=cut

use strict;
use warnings;


use base qw/Exporter/;
our @EXPORT = qw/
    release_errors
    release_errors_count
    release_errors_to_text
/;


use Getopt::Long;
use XML::LibXML;

use ProjectSpecific qw/svn_url/;

use Release::Issue;

use utf8;

sub release_errors
{
    my %O = @_;

    my ($rc, $st);
    if ($O{rc}) {
        $rc = Release::Issue::issue_info(key => $O{rc}, eval_hotfixes => 1);
        die "$O{rc} isn't a release" unless $rc->{type} eq 'release';
    }
    if ($O{st}) {
        $st = Release::Issue::issue_info(key => $O{st}, eval_hotfixes => 1);
        die "$O{st} isn't a release" unless $st->{type} eq 'release';
    }

    my $rl = svn_url('releases');
    my @release_bases = sort { $a <=> $b} 
        map { m!^release-(\d+)/?$! ? $1 : die "incorrect release branch name: $_" } 
        split "\n", `svn ls $rl`; 

    my $errors;

    $errors->{candidate} = tracker_cmp_svn($rc, "candidate", \@release_bases ) if $rc;
    $errors->{stable} = tracker_cmp_svn($st, "stable", [ grep { !$rc || $_ < $rc->{base_rev} } @release_bases] ) if $st;

    # TODO
    #push @errors, check_package_existance($rc, "candidate") if $rc;
    #push @errors, check_package_existance($st, "stable") if $st;

    # TODO
    #push @errors, check_deployed_version($st) if $st;

    $errors->{continuity} = check_continuity($rc, $st) if $rc && $st;
    $errors->{'hotfixes missed in candidate'} = check_hotfixes($rc, $st) if $rc && $st;

    return $errors;
}

sub release_errors_count
{
    my ($errors) = @_;

    return scalar map { @{$_||[]} } values %$errors;
}


sub release_errors_to_text
{
    my ($errors) = @_;
    
    my @text;

    for my $type (sort keys %$errors){
        next unless @{$errors->{$type}};
        push @text, "## $type";
        push @text, map { "  * $_\n" } @{$errors->{$type}};
    }

    return join "\n", @text;
}


# сравнивает "непрерывность последовательности" стабильного релиза и кандидата
sub check_continuity
{
    my ($rc, $st) = @_;

    my @errors;

    #Stable-base >= rc-base --- error
    if ( $rc->{base_rev} <= $st->{base_rev} ){
        push @errors, "base revision of candidate ($rc->{base_rev}) <= base revision of stable $st->{base_rev}";
    }

    return \@errors;
}


# Все хотфиксы из стабильного должны быть и в кандидате тоже
sub check_hotfixes 
{
    my ($rc, $st) = @_;

    my @missed_hotfixes;

    for my $h ( keys %{$st->{hotfixes}} ){
        #$_ <= rc-base || $_ in rc-hotfixes --- ok, else errror
        unless ( $h <= $rc->{base_rev} || $rc->{hotfixes}->{$h} ){
            push @missed_hotfixes, $h;
        }
    }

    return  @missed_hotfixes ? [ join(",", sort {$a <=> $b} @missed_hotfixes) ] : [];
}

# Сравнивает состояние тикета и svn для релиза
sub tracker_cmp_svn
{
    my ($r, $name, $branches) = @_;

    my @errors;
    
    # релизные бранчи, более новые, чем проверяемый релиз
    my @upper_branches = grep { $r->{svn_type} eq 'trunk' ? $_ >= $r->{base_rev} : $_ > $r->{base_rev} } @$branches;
    # более новые бранчи -- подозрительно 
    if ( @upper_branches ){
        push @errors, "Tracker: $r->{version}, svn: 1.$_.nnnnn-1 (releases/release-$_)" for @upper_branches;
    }

    # нет бранча, соответствующего версии -- ошибка
    if ($r->{svn_type} eq 'branch' && ! grep { $_ == $r->{base_rev} } @$branches ){
        push @errors, "Tracker: $r->{version}, svn: no releases/release-$r->{base_rev}";
    }

    # в бранче есть коммиты более новые, чем версия в Джире -- ошибка
    if ( $r->{svn_type} eq 'branch' && grep { $_ == $r->{base_rev} } @$branches ){
        my $xml = `svn info --xml $r->{svn_path}`;
        my $doc = XML::LibXML->new()->parse_string( $xml )->documentElement();
        my $head_rev = ( $doc->findnodes('/info/entry/commit') )[0]->attributes()->getNamedItem('revision')->textContent();
        if ( $head_rev != $r->{last_rev} ) {
            push @errors, "Tracker: $r->{version}, svn: 1.$r->{base_rev}.$head_rev-1";
        }
    }

    return \@errors;
}

1;
