package Release::Issue;

# $Id$

=head1 NAME
    
    Release::Issue -- работа с релизными тикетами

=head1 DESCRIPTION

    Работа с трекером:
      * создание/обновление Директ-релиз-тикета в трекере в соответствии с обновлением версии пакетов
      * созданеие ревью-тикета, комментарии о ревью

=head1 INTERNALS

    обработку pod см. в ./protected/nginx/nginx_update_geo_conf.pl

    Статусы релиз-тикета: 
    New               -- 10009
    Testing           -- 10002
    "RM  Acceptance " -- 10040
    Need Acceptance   -- 10034
    Ready to deploy   -- 10035
    Closed            -- 6

    Действия: 
    Ignore Testing -- 171
    Accept -- 61


=cut



use List::Util qw/max/;
use List::MoreUtils qw/before/;
use JSON;

use Startrek::Client::Easy;
use MigratorB::Parse;

use ProjectSpecific qw/version2rev svn_url parse_release_summary make_release_summary change_release_summary/;

use Data::Dumper;
use YAML;
use Getopt::Long;
use Text::Diff;
use Text::Diff::Parser;

use strict;
use feature qw/state/;

use utf8;
use open ':locale';

#................................................

my %ST_NAME = (
    10009 => "New",
    10002 => "Testing",
    10040 => "RM Acceptance",
    10034 => "Need Acceptance",
    10035 => "Ready to deploy",
        6 => "Closed",
);

my $DEFAULT_RELEASE_NAME = 'небольшие изменения';


# TODO можно не таскать $tracker через параметры, а создавать по необходимости
sub get_tracker_object {
    state $tracker;
    $tracker //= Startrek::Client::Easy->new(startrek => $ProjectSpecific::STARTREK_INSTANCE);
    return $tracker;
}

=head2 tracker_release

=cut

sub tracker_release
{
    my %OPT = @_;
    my $res = {};

    my $tracker = get_tracker_object();

    if ( $OPT{inspect_issue} ){
       
        inspect_issue($tracker, %OPT);

    } elsif( 1 ) {
        $OPT{name} ||= $DEFAULT_RELEASE_NAME;

        my $release = $OPT{release_issue_key} eq 'create' ? {} : $tracker->get( key => $OPT{release_issue_key} );
        my $to_do_struct = what_to_do_in_tracker($tracker, $release, %OPT);

        if( $OPT{print_only} ){
            print_updates($release, $to_do_struct);
            return;
        }

        my $release_queue = ProjectSpecific::get_project_data('default_tracker_queue');
        my $key;
        $key = $release->{key} if $OPT{release_issue_key} ne 'create';
        my @migrations_created;
        for my $type (qw/migrations release/){
            for my $to_do (@{$to_do_struct->{$type} || []}){
                if ( $type eq 'release' && scalar @migrations_created > 0 ){
                    my $migrations_list = join "\n", @migrations_created;
                    $to_do->{description} =~ s/__MIGRATIONS_PLACEHOLDER__/$migrations_list/;
                }
                my $tracker_result = $tracker->do(%$to_do);
                if ( $type eq 'migrations' && $to_do->{create} ) {
                    push @migrations_created, $tracker_result;
                } elsif ( $type eq 'release' && $OPT{release_issue_key} eq 'create' && $tracker_result =~ /^$release_queue-/){
                    die "2 issues in default queue: $key, $tracker_result; stop" if $key;
                    $key = $tracker_result;
                }
            }
        }

        if ($OPT{release_issue_key} eq 'create') {
            print "created: $key\n";
            $res->{created} = $key;
        }

        #print "версия $OPT{version} , тикет $jira->{'.easy_jiraurl'}/browse/$release->{key}\n"; # $OPT{version} уже недоступна
    }

    return $res;
}

sub version_in_tracker
{
    my ($issue_key) = @_;
    my $tracker = get_tracker_object();
    my $release = $tracker->get( key => $issue_key );

    my $summary_details = parse_release_summary($release->{summary});
    my $version = $summary_details->{version};

    return $version;
}

# Печатает номер, имя, статус релиз-тикета
sub inspect_issue
{
    my ($tracker, %O) = @_;

    # no-svnroot ready
    my $deploy_path = $O{use_svnroot} ? "svnroot/deploy" : "../deploy";

    my $release = $tracker->get( key => $O{release_issue_key} );

    print "\nkey:    \t$release->{key}\nsummary:\t$release->{summary}\nstatus:  \t".($ST_NAME{$release->{status}}|| "unknown ($release->{status})")."\n\n";

    my $summary_details = parse_release_summary($release->{summary});
    my ($version, $deploy) = ($summary_details->{version}, $summary_details->{migration_title});

    my $issue_url;
    my $startrek_web_url = ProjectSpecific::get_data(startrek_url => $ProjectSpecific::STARTREK_INSTANCE); 
    $issue_url = "$startrek_web_url/$release->{key}";
    print "версия $version , тикет $issue_url".($deploy ? " . Есть инструкция к выкладке! " : "")."\n";

    #####
    # вытаскиваем список непримененных миграций
    my $desc_parts = description2logs($release->{description});
    my $rev_2 = version2rev($O{version},   name => 'new', rev => 'last');
    (my $rev_0) = $desc_parts->{changelog} =~ m/^.*?r(\d+)/sm or die "can't determine base revision for release";
    my $migration_details = migration_details(start_rev => $rev_0 - 1, end_rev => $rev_2, deploy_path => $deploy_path );
    my $migration_tickets = $migration_details->{migration_tickets_all};
    print "\n";
    for my $m (sort @$migration_tickets){
        ### TODO выводить статус тикета из Трекера
        print "    $m\n";
    }
    #####

    return;
}


=head2 what_to_do_in_tracker

=cut
sub what_to_do_in_tracker
{
    my ($tracker, $old_values, %O) = @_;
    my $to_do_struct = {release => [], migrations => [],};

    # no-svnroot ready
    my $deploy_path = $O{use_svnroot} ? "svnroot/deploy" : "../deploy";

    if ( $O{release_issue_key} eq 'create' ){
        my $rev_2 = version2rev($O{version}, name => 'new', rev => 'last');
        my $rev_1 = version2rev($O{prev_version}, name => 'prev', rev => 'base');

        my $desc_parts = {};
        $desc_parts->{changelog} = log_text(start_rev => $rev_1 + 1, end_rev => $rev_2 );
        # hash_merge
        my $migration_details = migration_details(start_rev => $rev_1, end_rev => $rev_2, deploy_path => $deploy_path );
        # to_do на миграции
        # 1. Создаем
        for my $migr_file ( sort keys %{$migration_details->{migration_files}||{}} ){
            my $migr_params = {
                create => 1, 
                type => 'task',
                description => "%%\n$migration_details->{migration_files}->{$migr_file}\n%%",
                summary => "direct: Миграция $migr_file",
                queue => "DIRECTMIGR",
            };
            push @{$to_do_struct->{migrations}}, $migr_params;
        }
        # 2. Обновляем
        for my $migr_ticket ( sort keys %{$migration_details->{migration_tickets_to_update}||{}} ){
            #print "migr_ticket $migr_ticket\n$migration_details->{migration_tickets_to_update}->{$migr_ticket}\n\n";
            my $migr_params = {
                key => $migr_ticket,
                description => "%%\n$migration_details->{migration_tickets_to_update}->{$migr_ticket}\n%%",
            };
            push @{$to_do_struct->{migrations}}, $migr_params;
        }

        # to_do на создание релиза
        for ( qw/migration_tickets_all migration_warning/ ){
            next unless exists $migration_details->{$_};
            $desc_parts->{$_} = $migration_details->{$_}; 
        }
        $desc_parts->{need_create_migration_tickets} = 1 if keys %{$migration_details->{migration_files}||{}};

        my $description = logs2description(%$desc_parts);

        my $summary = make_release_summary(name => $O{name}, version => $O{version} ); 

        my $release_params = {
            create => 1, 
            assignee => [getpwuid($<)]->[0],
            description => $description,
            summary => $summary,
            type => 'release',
            queue => ProjectSpecific::get_project_data('default_tracker_queue'),
        };
        my $release_component;
        $release_component = eval { ProjectSpecific::get_project_data('startrek_release_component') };
        if ($release_component) {
            $release_params->{components} = $release_component;
        }

        if ($O{testserver}) {
            $release_params->{tags} = ["testserver:$O{testserver}" ];
        }

        push @{$to_do_struct->{release}}, $release_params;
    } else {
        my $release_params = {};
        $release_params->{key} = $old_values->{key};

        my $summary_details = parse_release_summary($old_values->{summary}); 
        my $prev_version = $summary_details->{version};
        $release_params->{summary} = $old_values->{summary};

        # если надо -- переименуем релиз...
        $release_params->{summary} = change_release_summary($release_params->{summary}, name => $O{rename})  if ($O{rename});

        # если сменилась версия -- поменяем заголовок, описание, статусы + прокомментируем
        if ( $O{version} ne $prev_version || $O{force}){
            if ( $O{version} lt $prev_version && ! $O{downgrade} ){
                print "VERY suspicious: new version of packages less than previous one. Use --downgrade if absolutely indispensable.";
                exit(1);
            }

            my $rev_1 = version2rev($prev_version, name => 'prev', rev => 'last');
            my $rev_2 = version2rev($O{version},   name => 'new', rev => 'last');
            $release_params->{summary} = change_release_summary($release_params->{summary}, version => $O{version});

            my $desc_parts = description2logs($old_values->{description});

            # каждый раз полностью пересоставляем migration-сообщение
            # ??? что должно быть, если модифицирован старый-старый migration-файл?
            my $rev_0;
            ($rev_0) = $desc_parts->{changelog} =~ m/^.*?r(\d+)/sm or die "can't determine base revision for release";
            #hash_merge
            my $migration_details = migration_details(start_rev => $rev_0 - 1, end_rev => $rev_2, deploy_path => $deploy_path );
            # to_do на миграции
            # 1. Создаем
            for my $migr_file ( sort keys %{$migration_details->{migration_files}||{}} ){
                my $migr_params = {
                    create => 1, 
                    type => 'task',
                    description => "%%\n$migration_details->{migration_files}->{$migr_file}\n%%",
                    summary => "direct: Миграция $migr_file",
                    queue => "DIRECTMIGR",
                };
                push @{$to_do_struct->{migrations}}, $migr_params;
            }
            # 2. Обновляем
            for my $migr_ticket ( sort keys %{$migration_details->{migration_tickets_to_update}||{}} ){
                #print "migr_ticket $migr_ticket\n$migration_details->{migration_tickets_to_update}->{$migr_ticket}\n\n";
                my $migr_params = {
                    key => $migr_ticket,
                    description => "%%\n$migration_details->{migration_tickets_to_update}->{$migr_ticket}\n%%",
                    comment => 'текст миграции обновлен',
                };
                push @{$to_do_struct->{migrations}}, $migr_params;
            }
            for ( qw/migration_tickets_all migration_warning/ ){
                next unless exists $migration_details->{$_};
                $desc_parts->{$_} = $migration_details->{$_}; 
            }
            $desc_parts->{need_create_migration_tickets} = 1 if keys %{$migration_details->{migration_files}||{}};

            my $migration_svn_changed = qx!svn diff -r$rev_1:$rev_2 $deploy_path!;

            # приписываем куда следует дополнения к changelog -- для нового тикета в description, для тестирующегося -- в комментарий
            my $addition_log_text = log_text(start_rev => $rev_1 + 1, end_rev => $rev_2 );
            if ( $addition_log_text ){
                if ($old_values->{status} eq 'new' || $old_values->{status} eq 'readyForTest') {
                    $desc_parts->{changelog} = $desc_parts->{changelog}."\n".$addition_log_text;
                } else {
                    my $testing = $old_values->{status} eq 'testing';
                    $release_params->{comment} = 
                    ($testing ? "В релиз включен новый код": "Добавлены хотфиксы")
                    . ", версия: $prev_version --> $O{version}"
                    .($testing ? "\n$O{tupdate_message}": "")
                    ."\n$addition_log_text\n"
                    # .($testing && !$O{repeat_tests} ? "Изменения небольшие и локальные. Повторное регрессионное тестирование не требуется." : "");
                    .''; 
                }
            }

            # Собираем полный description
            if (1){
                $release_params->{description} = logs2description(%$desc_parts);
            }

            # если у тестирующегося тикета меняется migration-сообщение -- отдельно отмечаем это в комментарии
            if ($migration_svn_changed && $old_values->{status} ne 'new') { 
                $release_params->{comment} = $release_params->{comment}."\nВНИМАНИЕ!!! Обновлена инструкция по выкладке!"
            }

            # Если релиз уже выложен и закрыт -- переоткрываем, игнорируем тестирование, принимаем
            if( ($O{version} ne $prev_version) && $old_values->{status} eq 'closed') {
                $release_params->{actions} = [ qw/reopen ignore_testing accept/ ]; 
                push @{$release_params->{actions}}, 'accept' unless $O{repeat_tests};
            }
        }

        # Если надо -- добавляем комментарий, указанный в командной строке
        if ( $O{comment} ){
            $release_params->{comment} = join "\n\n", grep {$_} ($release_params->{comment}, $O{comment});
        }

        if ($O{cc}) {
            $release_params->{followers} = [ split /,/, $O{cc} ];
        }

        if ($O{testserver}) {
            my @tags_to_keep = grep { ! /^testserver:/ } @{ $old_values->{tags} || [] };
            $release_params->{tags} = [ @tags_to_keep, "testserver:$O{testserver}" ];
        }

        push @{$to_do_struct->{release}}, $release_params;
    }

    return $to_do_struct;
}


# из changelog и migration собирает description для тикета
# передаем сюда migration_tickets_all (миграции с прошлого раза) и need_create_migration_tickets (флаг "будут созданы новые миграции")
sub logs2description
{
    my %options = @_;

    # составляем описание релиза: берем changelog...
    my $description = qq!====<# <a name="changelog">Изменения</a> #>==\n$options{changelog}!;

    # ...и если есть migration-инструкции -- дописываем их сверху
    my @migration_parts;
    # если были готовые тикеты
    push @migration_parts, join "", map {"$_\n"} @{$options{migration_tickets_all}||[]};
    push @migration_parts, "__MIGRATIONS_PLACEHOLDER__\n" if $options{need_create_migration_tickets};

    my $migration_text = join '', @migration_parts;

    $migration_text =~ s/[\n]{3,}/\n\n/g;
    if ( $migration_text ) {
        $description = qq!\n==<# <span style="color:orange;">Инструкция к выкладке </span> #>==\n{{anchor name='migration'}}\n$migration_text\n----\n$description!;
    }
    return $description;
}


# разбирает description в хеш {changelog => ...,}
sub description2logs
{
    my $description = shift;

    my $h = {};
    (my $migration_text, $h->{changelog}) = $description =~ m!^(?:[\n]*[^\n]*\n+[^\n]*name='migration'[^\n]*\n(.*)----\n)?===[^\n]*name="changelog"[^\n]*==\n(.*?)$!s;

    return $h;
}


# отдает текст svn-лога начиная с ревизии $O{start_rev} и до ревизии $O{end_rev}
sub log_text 
{
    my %O = @_;

    return '' unless $O{start_rev} <= $O{end_rev};
    my @log = qx/make release_log START=$O{start_rev} END=$O{end_rev}/;
    shift @log;
    my $log_text = join "", @log;

    # Для хотфиксов: подчищаем комментарии от ненужного
    $log_text =~ s/^(r\d+\s)[^\n]*\n(RELEASE: )?(Merged|Created)[^\n]*//gsm; 
    # Для релизных бранчей -- убираем маркер "RELEASE"
    $log_text =~ s/^RELEASE: *\n//gsm; 
    # И просто убираем лишние пустые строки
    $log_text=~s/\n{3,}/\n\n/gsm;

    return $log_text;
}

sub migration_log_text
{
    die "use migration_details";
}


# TODO (???) сначала получать списки: отдельно добавленных deploy-файлов, отдельно модифицированнных. 
# Добавленные безусловно вписывать в лог, а модифицированные -- в зависимости от ключей командной строки (чтобы можно было сначала прочитать, что случилось, и была возможность отказаться от этого предупреждения в тикете)

# отдает хеш с данными о миграционных файлах (/deploy), модифицированными с ревизии $O{start_rev} и до $O{end_rev}

# jira-release-v2.pl -j 9276 -ver 1.17445-1 -prev 1.17435-1 
# jira-release-v2.pl -j create -ver 1.17445-1 -prev 1.17443-1 -n 'модифицирование старой deploy-записи'
sub migration_details
{
    my %O = @_;
    die unless $O{deploy_path};

    # filterdiff используем на случай, если в deploy у файлов менялись svn-properties: Text::Diff::Parser не умеет такое разбирать
    my @log = qx!svn diff --depth=files --no-diff-deleted -r $O{start_rev}:$O{end_rev} $O{deploy_path} |filterdiff --clean!;
    my $raw_log_text = join '', @log;
    my $diff_parser = Text::Diff::Parser->new( Diff=>$raw_log_text );

    my $warning = '';
    my %modified;
    my $M = qx!svn diff -r $O{start_rev}:$O{end_rev} --summarize $O{deploy_path} | "grep" '^M' --color=never!;
    if ($M) {
        # модифицирована (не добавлена, а именно модифицирована) migration-запись
        $warning = "ВНИМАНИЕ!!!\nМодифицирована migration-запись, необходима проверка:\n$M\n";
        %modified = map { s!.*\s.*/(\S+)!$1!; $_ => 1} split /\n/, $M;
    }

    my $files = {};
    foreach my $change ( $diff_parser->changes ) {
        my $filename = $change->filename2;
        $filename =~ s!^$O{deploy_path}/!!;
        # Соглашение: файлы *.data не включаем в список
        next if $filename =~ m/\.data$/;
        # бывают файлы *.csv (по смыслу те же *.data), их тоже пропускаем
        next if $filename =~ m/\.csv$/;
        # shell-ных миграций не бывает, но разрешаем скрипты складывать в /deploy как данные (инструкция по запуску будет в отдельном .migr)
        next if $filename =~ m/\.sh$/;
        # модифицированные (не добавленные, а отредактированные) файлы пропускем
        next if $modified{$filename};

        my $size = $change->size;
        my @lines = map { $change->text( $_ ) } 0..($size-1);

        my $text=join "\n", @lines;
        # YAML::Load не работает, если текст не заканчивается переводом строки
        $text .= "\n" unless $text =~ /\n$/;
        $files->{ $filename } = MigratorB::Parse::to_text($filename => $text);
    }

    # миграции, что уже созданы тикетами (захотфиксились, например), записываем в отдельный список
    my $migr_tickets_all = [];
    my $migr_tickets_to_update = {};
    for my $migr_file ( sort keys %$files ){
        my $ticket = find_migration_ticket($migr_file);
        if ( $ticket ){
            my $migr_text = delete $files->{$migr_file};
            push @$migr_tickets_all, $ticket->{key};
            ### если миграция есть, но текст неактуальный -- надо обновлять текст
            if ( $ticket->{description} ne "%%\n$migr_text\n%%" ){
                $migr_tickets_to_update->{$ticket->{key}} = $migr_text; 
            }
        }
    }

    return {
        migration_files => $files,
        migration_tickets_all => $migr_tickets_all,
        migration_tickets_to_update => $migr_tickets_to_update,
        migration_warning => $warning,
    };
}

sub find_migration_ticket
{
    my ($migr_file) = @_;
    # TODO для Модерации -- либо своя очередь, либо DIRECTMOD + компонента
    my $issues = get_tracker_object()->get( query => qq!Queue: DIRECTMIGR Summary: #"direct: Миграция $migr_file"!, array => 1);
    if (scalar @$issues == 0){
        return '';
    } elsif ( scalar @$issues == 1 ) {
        return $issues->[0];
    } elsif (scalar @$issues > 1){
        die "found multiple tickets for the same migration ($migr_file): ".join ", ", @$issues;
    } else {
        die "can't be";
    }
}


# печатает сводку по запланированным изменениям в релиз-тикете
sub print_updates
{
    my ($release, $to_do_struct) = @_;

    for my $type (qw/migrations release/){
        for my $to_do (@{$to_do_struct->{$type} || []}){
            print "\n";
            if( $to_do->{create} ){ 
                my $warning_msg = '';
                $warning_msg = "\nWARNING: DEFAULT RELEASE NAME! Use -n <description> to name release more meaningful\n" if $type eq 'release' && parse_release_summary($to_do->{summary})->{name} eq $DEFAULT_RELEASE_NAME;

                print "===To create===\n$to_do->{summary}\n$warning_msg\n\n";
                print "Description:\n$to_do->{description}\n"
            } else {
                print "===To update: $to_do->{key}===\nSummary: ".(exists $to_do->{summary} ? $to_do->{summary} : "no changes")."\n\n"; 
                if (exists $to_do->{description} && $type eq 'release') {
                    my $old_desc = "$release->{description}\n";
                    my $new_desc = "$to_do->{description}\n";
                    my $desc_diff = diff \$old_desc, \$new_desc;
                    print "Description diff:\n$desc_diff\n";
                } elsif ( exists $to_do->{description} ) {
                    ### TODO дифф хорошо показывается только для релиза, а надо и для миграций тоже
                    print "New description:\n$to_do->{description}\n";
                } else {
                    print "Description: no changes\n";
                }
            }
            print "Comment:\n$to_do->{comment}\n" if $to_do->{comment};
            print "Actions:\n".join(', ', @{$to_do->{actions}})."\n" if exists $to_do->{actions};
            print "Add followers:\n" . join(', ', @{$to_do->{followers}}) . "\n" if exists $to_do->{followers};
        }
    }
}


our @FIELDS_ORDER = (qw/
key
type
summary
priority
status
version
/,
'base_rev',
'last_rev',
'created',
'svn path',
'hotfixes',
);

sub issue_info_to_text
{
    my ($info) = @_;

    my $len = max map {length $_} keys %$info;
    $len ||= 10;

    return join "\n", map { exists $info->{$_} ? ("$_: ".(" " x ($len - length $_)) . $info->{$_} ) : () } @FIELDS_ORDER;
}

sub issue_info
{
    my (%O) = @_;

    die unless $O{key} || $O{last_release};

    my %get_options;
    if ($O{last_release}) {
        %get_options = (query => ProjectSpecific::make_last_releases_query(), limit => 1);
    } else {
        %get_options = (key => $O{key});
    }
    my $issue = get_tracker_object()->get( %get_options );

    my $info = {
        key      => $issue->{key},
        type     => $issue->{type},
        summary  => $issue->{summary},
        priority => $issue->{priority},
        status   => $issue->{status},
    };

    if ($info->{type} eq 'release') {
        my $summary_details = parse_release_summary($issue->{summary});
        my $version = $summary_details->{version};
        my $base_rev = version2rev($version, rev => 'base');
        my $last_rev = version2rev($version, rev => 'last');
        my $svn_path = $base_rev eq $last_rev ? svn_url('trunk') : svn_url('releases')."/release-$base_rev";
        $issue->{createdAt} =~ s/^([\d-]*)T.*$/$1/;

        $info->{version} = $version;
        $info->{base_rev} = $base_rev;
        $info->{last_rev} = $last_rev;
        $info->{created} =  $issue->{createdAt};
        $info->{svn_path} = $svn_path;
        $info->{svn_type} = $base_rev eq $last_rev ? 'trunk' : 'branch';
        if ( $base_rev ne $last_rev) {
            my $merged_revisions = qx!svn propget svn:mergeinfo $svn_path |"grep" '^/trunk' --color=never!;

            $merged_revisions =~ s!^/trunk:!!;
            if ($O{eval_hotfixes}) {
                $merged_revisions =~ s/-/../g;
                $info->{hotfixes} = { map { $_ => 1 } eval($merged_revisions) };
                die if $@;
            } else {
                $info->{hotfixes} = $merged_revisions;
            }
        }
        
        my ($testserver_tag, $another_testserver_tag) = grep {/^testserver:/} @{ $issue->{tags} || [] };
        (my $testserver = $testserver_tag) =~ s/^testserver://;
        if ($another_testserver_tag) {
            die "ERROR: в релизном тикете может быть не больше одного тега с префиксом 'testserver: '";
        }
        $info->{testserver} = $testserver;
    }

    return $info;
}


sub create_review
{
    my %opt = @_;
    my $release_issue_key = $opt{release};
    die "Релизный тикет должен быть уже создан" if $release_issue_key eq 'create';

    my $tracker = get_tracker_object();

    my $url = sprintf("https://direct-dev.yandex-team.ru/metatracker/issuesToReview?release_jira_id=%s", $release_issue_key);
    my $to_review_json = qx!curl -s -k '$url'!; 
    my $to_review = decode_json($to_review_json);
    my $summary = "Ревью непродуктовых коммитов в релизе $release_issue_key"; 
    my $description = join "", map { "st:$_\n" } @$to_review;

    print "$summary\n$description\n";
    my $review_key = $tracker->do(
        create => 1, 
        queue => 'DIRECT',
        summary => $summary, 
        description => $description,
    );
    $tracker->do(
        key => $release_issue_key,
        comment => "создан тикет на ревью: $review_key",
    );

    return { created => $review_key, };
}

# для ревью 
sub post_comment
{
    my %OPT = @_;
    die "no key" unless $OPT{key};
    die "empty comment" unless $OPT{comment};

    my $tracker = get_tracker_object();
    my %to_do = (
        key => $OPT{key},
        comment => $OPT{comment},
    );
    $tracker->do(%to_do);

    return;
}


1;
