#!/usr/bin/perl

=head1 NAME

    cloud-app-build.pl -- собрать образ для релиза облачного приложения

=head1 SYNOPSIS

    cloud-app-build.pl --app direct_rbac --version 1_12345_1

=head1 DESCRIPTION

    Запуск скрипта соберёт образ приложения, сгенерирует changelog, и выведет с нужнымим параметрами команду на создание релизного тикета (cloud-release.pl)

    Параметры:
        
        --app <имя>
            Имя приложения.

        -hf, --hotfix
            Собрать хотфикс к существующему релизу. На процесс сборки никак не влияет, влияет на выводимую в конце команду для работы с трекером.

        --no-build
            Не собирать образ, только сгенерировать changelog и команду для создания тикета.

        --version <версия>
            Версия приложения. По умолчанию берётся из ревизии текущей рабочей копии.

=cut

use strict;
use warnings;

use utf8;

use Getopt::Long;
use File::Slurp;
use File::Temp;

use Startrek::Client::Easy;
use Yandex::Shell;
use Yandex::Svn;

use open ":std" => ":encoding(UTF-8)";

$Yandex::Shell::PRINT_COMMANDS = 1;

run() unless caller();

sub get_options {
    my %O;
    GetOptions(
        "app=s" => \$O{app_name},
        "hf|hotfix" => \$O{hotfix},
        "no-build" => \$O{no_build},
        "version=s" => \$O{version},
    ) || die "can't parse options, stop";
    die "app name missing" unless $O{app_name};
    return \%O;
}

sub get_last_release_issue {
    my ($app_name) = @_;

    my $st = Startrek::Client::Easy->new();
    my $last_release = $st->get(query => qq/Queue: DIRECT Type: Release Components: "App: $app_name" sort by: Key desc/, limit => 1, array => 1)->[0];

    return $last_release;
}

sub get_changelog {
    my ($app_name, $start_rev, $end_rev) = @_;

    my $changelog = yash_qx("./docker_build.pl svnlog $app_name --revision=$start_rev:$end_rev");
    for ($changelog) {
        s/\n{2,}/\n/gs;
        s/^-+$//gms;
        s/^(?:r[0-9]+\s+\|)/\n/gs;
    }

    return $changelog;
}

sub get_version_from_working_copy {
    (my $version = yash_qx('direct-pkg-version')) =~ s/\W/_/gx; # аналогично docker_build.pl

    return $version;
}

sub ver2rev {
    my ($version) = @_;
    $version =~ /^1_(.*)_[0-9]+$/;
    my $rev = $1;
    return $rev;
}

sub run {
    my $opt = get_options();
    my $svn_info = svn_info('.');
    die "Скрипт должен запускаться из поддиректории packages рабочей копии транка" unless $svn_info->{url} eq 'svn+ssh://svn.yandex.ru/direct/trunk/packages';

    if (!$opt->{version}) {
        $opt->{version} = get_version_from_working_copy();
    }

    my ($start_rev, $end_rev);
    my $last_release = get_last_release_issue($opt->{app_name});
    if ($last_release && $last_release->{status} ne 'closed' && !$opt->{hotfix}) {
        die "Уже сщуествует незакрытый релиз $last_release->{key}, если нужно собрать хотфикс, используйте опцию --hotfix";
    }

    if ($last_release) {
        $last_release->{summary} =~ /\b\Q$opt->{app_name}\E=([0-9_]+)/;
        my $prev_version = $1;
        $start_rev = ver2rev($prev_version);
    } else {
        $start_rev = 70000;
    }
    
    $end_rev = ver2rev($opt->{version});

    my $changelog = get_changelog($opt->{app_name}, $start_rev, $end_rev);
    my $changelog_file = "$opt->{app_name}-$opt->{version}.changelog";
    write_file($changelog_file, $changelog);

    yash_system('./docker_build.pl', 'build', $opt->{app_name}, '--version' => $opt->{version}, '--no-image-cache') unless $opt->{no_build};

    print "Команда для создания релизного тикета:\n";
    print "./cloud-release.pl --app=$opt->{app_name} --version=$opt->{version} --changelog-file=$changelog_file" . ($opt->{hotfix} ? " --hotfix" : " --name <имя>") . "\n";

    exit 0;
}
