#!/usr/bin/perl -CA

=head1 NAME

    cloud-release-pl -- создать релизный тикет для облачного приложения или добавить комментарий о хотфиксе к существующему

=head1 SYNOPSIS

    cloud-release.pl --app direct_rbac --version 1_12345_1 -f direct_rbac_1_12345_1.changelog --name 'очередной релиз'          # вывести то, что попадёт в тикет
    cloud-release.pl --app direct_rbac --version 1_12345_1 -f direct_rbac_1_12345_1.changelog --name 'очередной релиз'  --do    # на самом деле создать тикет
    cloud-release.pl --app direct_rbac --version 1_12346_1 -f direct_rbac_1_12346_1.changelog --do      # добавить комментарий об обновлении до версии 1_12346_1

=head1 DESCRIPTION

    В штатном режиме команда для запуска скрипта (с нужными параметрами) и файл с историей изменений генерируются скриптом cloud-app-build.pl

    Параметры:
        --app <имя приложения>
            Имя приложения.

        --version <версия>
            Версия приложения.

        -f, --changelog-file <имя файла>
            Файл с svn log'ом для релиза.

        -n, --name <имя>
            Имя (summary в Стартреке) создаваемого релиза.
            
        --cc <логин1,логин2,...>
            Кого добавить в наблюдатели (список логинов через запятую без пробелов)

        -hf, --hotfix
            Добавить комментарий о хотфиксе в уже существующий релиз (выберет последний незакрытый).

        --do
            Создать тикет/комментарий в тикете. Без этого ключа будет только выведена информация о том, что собираемся сделать в трекере.

=cut

use strict;
use warnings;

use utf8;

use Encode;
use File::Slurp;
use Getopt::Long;

use Startrek::Client::Easy;

use open ":std" => ":encoding(UTF-8)";

my $cmdline = join " ", $0, map { /\s/ ? "'$_'" : $_ } @ARGV;
my $st = Startrek::Client::Easy->new();
print $st->{api_url} . "\n";

my %O;
GetOptions(
    "app=s" => \$O{app},
    "v|version=s" => \$O{version},
    "f|changelog-file=s" => \$O{changelog_file},
    "n|name=s" => \$O{name},
    "cc=s" => \$O{cc},
    "hf|hotfix" => \$O{hotfix},
    "do" => \$O{do},
);
die "placeholder detected instead of actual release name" if $O{name} && $O{name} =~ /</;

my $app_name = $O{app};
$app_name = "direct_$app_name" if $app_name !~ /^direct_/;

my %issue_data;
if ($O{hotfix}) {
    my $last_release = $st->get(query => qq/Queue: DIRECT Type: Release Components: "App: $app_name" sort by: Key desc/, limit => 1);
    $last_release->{summary} =~ /\b\Q$app_name\E=([0-9_]+)/;
    my $prev_version = $1;
    (my $new_summary = $last_release->{summary}) =~ s/$prev_version/$O{version}/g;
    %issue_data = (
        key => $last_release->{key},
        comment => "В релиз включен новый код, версия: $prev_version --> $O{version}\n" . read_file($O{changelog_file}, {binmode => ":encoding(UTF-8)"}),
        summary => $new_summary,
    );
} else {
    $O{name} ||= "очередной релиз";
    %issue_data = (
        queue => "DIRECT",
        type => "release",
        assignee => (getpwuid($<))[0],
        summary => "cocaine/$app_name: $O{name} - выложить $app_name=$O{version}",
        description => "==== Изменения==\n" . read_file($O{changelog_file}, {binmode => ":encoding(UTF-8)"}),
        components => [ "App: $app_name" ],
        $O{cc} ? (followers => [ split /,/, $O{cc} ]) : (),
    );
}

use YAML;
print Dump(\%issue_data);

if ($O{do}) {
    my $new_issue_key = $st->do(create => $O{hotfix} ? 0 : 1, %issue_data);
    if ($new_issue_key) {
        print "Создан новый релизный тикет: " . "https://st.yandex-team.ru/$new_issue_key\n";
    }
} else {
    print "Создать релизный тикет:\n$cmdline --do\n";
}
