#!/usr/bin/perl

=head1 NAME

    get-release-components.pl — задачи из релиза по компонентам

=head1 SYNOPSIS

    get-release-components.pl DIRECT-41631
    get-release-components.pl DIRECT-41631 --count
    get-release-components.pl DIRECT-41631 --count --comment

=head1 DESCRIPTION

    Скрипт для получения распределения тикетов, вошедших в релиз по компонентам. Считаем, что тикет входит в релиз, если он связан с релизным тикетом и находится в той же очереди.
    Некоторые компоненты (например, "Поддержка", "Недельные планы", которые не описывают части системы, затронутые правками по задаче), не учитываются.

    Без указаний опций выводит YAML примерно такого вида:

        ---
        '<нет компоненты>':
          - DIRECT-41637
          - DIRECT-41657
          - DIRECT-41663
        Взаимодействие c Модерацией:
          - DIRECT-41572
        Взаимодействие с БК:
          - DIRECT-41566
        Движок:
          - DIRECT-41194
        Клиентсайд:
          - DIRECT-37521

    Параметры командной строки:

        -h, --help
            вывести справку

        --count
            Добавлять к имени компоненты количество тикетов с этой компонентой (например, "Движок (1)"; добавляется в ключи хэша, сериализуемого в YAML)

        --comment
            Добавить отформатированный комментарий в релизный тикет (если не указана опция --dry-run, см. ниже).

        -n, --dry-run
            Используется вместе с опцией --comment: не писать комментарий в Стартрек, а вывести его в STDOUT (как есть, с инструкциями языка разметки)

=head1 TODO

    Получать статистику не только для релизного тикета, но и из отрезка svn log'а (будущий релиз).

=cut

use strict;
use warnings;

use utf8;

use open qw/:std :encoding(UTF-8)/;

use Getopt::Long;
use YAML;

use ProjectSpecific;
use Startrek::Client::Easy;

my $st = Startrek::Client::Easy->new();
run() unless caller();

sub run {
    my $opt = get_options();

    my $release_issue_key = $ARGV[0] || die "no release key given";
    $release_issue_key =~ /^([A-Z]+)-[0-9]+$/ || die "invalid release issue key";
    my $queue = $1;
    my $links = $st->request(GET => "/issues/$release_issue_key/links");

    my $issues_by_components = group_issues_by_components([ grep { /^$queue-/ } map { $_->{object}->{key} } @$links ]);

    if ($opt->{count}) {
        for my $component (keys %$issues_by_components) {
            my $new_key = $component . ' (' . scalar @{ $issues_by_components->{$component} } . ')';
            $issues_by_components->{ $new_key } = delete $issues_by_components->{ $component };
        }
    }
    if ($opt->{comment}) {
        my $comment = "Задачи по компонентам:\n";
        for my $component (sort keys %$issues_by_components) {
            # убираем список тикетов под кат
            $comment .= '<{';
            # экранируем название компоненты
            $comment .= qq/""$component""/ . ":\n";

            $comment .= "  - $_\n" for @{ $issues_by_components->{ $component } };
            $comment .= "}>\n";
        }

        if ($opt->{dry_run}) {
            print "Добавить комментарий в https://st.yandex-team.ru/$release_issue_key :\n\n" . $comment;
        } else {
            eval { $st->do(key => $release_issue_key, comment => $comment) };
            if ($@) {
                print STDERR "Не удалось добавить комментарий в тикет https://st.yandex-team.ru/$release_issue_key : " . $@;
                print STDERR "Текст комментария для добавления вручную:\n\n" . $comment . "\n";
                exit 1;
            }
        }
    } else {
        print Dump($issues_by_components);
    }
}

sub get_options {
    my %O;
    GetOptions(
        "h|help" => sub { system("podselect -section NAME -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8"); exit 0; },
        "comment" => \$O{comment},
        "count" => \$O{count},
        "n|dry-run" => \$O{dry_run},
    ) || die "can't parse options, stop";
    return \%O;
}

sub group_issues_by_components {
    my ($issue_keys) = @_;
    my $issues_by_components = {};
    for my $issue_key (@$issue_keys) {
        # $st->request вместо $st->get потому get сократит структуры вида { self => ..., id => ..., display => ... } в ответе API до значений id. То есть, список
        # components => [
        #     { self => ..., id => <id1>, display => ... },
        #     { self => ..., id => <id2>, display => ... },
        #       ...
        # ]
        # превратится в
        # components => [ <id1>, <id2>, ... ],
        # а нам нужны человекочитаемые (отображаемые в интерфейсе Стартрека) названия компонент (display).
        my $issue = $st->request(GET => "/issues/$issue_key");

        my @components = @{ $issue->{components} || [] };
        if (!@components) {
            # угловые скобки для выделения и удобства сортировки
            push @{ $issues_by_components->{ '<нет компоненты>' } }, $issue_key;
        }
        for my $component (@components) {
            push @{ $issues_by_components->{ $component->{display} } }, $issue_key;
        }
    }
    return $issues_by_components;
}
